import pool from './database.js';

export class NotificationModel {
    static async getByUserId(userId) {
        const result = await pool.query(
            'SELECT * FROM notifications WHERE user_id = $1 ORDER BY created_at DESC',
            [userId]
        );
        return result.rows;
    }

    static async getUnreadCount(userId) {
        const result = await pool.query(
            'SELECT COUNT(*) FROM notifications WHERE user_id = $1 AND is_read = FALSE',
            [userId]
        );
        return parseInt(result.rows[0].count);
    }

    static async create(notification) {
        const { user_id, type, title, message, related_service_id } = notification;
        const result = await pool.query(
            `INSERT INTO notifications (user_id, type, title, message, related_service_id)
             VALUES ($1, $2, $3, $4, $5)
             RETURNING *`,
            [user_id, type, title, message, related_service_id]
        );
        return result.rows[0];
    }

    static async markAsRead(id) {
        const result = await pool.query(
            'UPDATE notifications SET is_read = TRUE WHERE id = $1 RETURNING *',
            [id]
        );
        return result.rows[0];
    }

    static async markAllAsRead(userId) {
        await pool.query(
            'UPDATE notifications SET is_read = TRUE WHERE user_id = $1',
            [userId]
        );
        return true;
    }

    // Helper to check duplicates (avoid spamming expiring alerts)
    static async exists(userId, type, relatedServiceId) {
        // Check if created in the last 24 hours to avoid daily spam if not read?
        // Or just check if exists at all for this service and type?
        // User wants alert for expiring. We should alerting once per threshold.
        // For now, let's checking if exists Unread or created recently.
        const result = await pool.query(
            `SELECT * FROM notifications 
             WHERE user_id = $1 AND type = $2 AND related_service_id = $3 
             AND (is_read = FALSE OR created_at > NOW() - INTERVAL '1 day')`,
            [userId, type, relatedServiceId]
        );
        return result.rows.length > 0;
    }
}
