import { query } from './database.js';

export const QuestionModel = {
    async getAll(filters = {}) {
        // Primero verificar si los campos first_name, last_name existen
        let checkColumnsSql = `
            SELECT column_name 
            FROM information_schema.columns 
            WHERE table_schema = 'public' 
            AND table_name = 'users' 
            AND column_name IN ('first_name', 'last_name', 'second_last_name', 'name')
        `;

        let columnCheck;
        try {
            columnCheck = await query(checkColumnsSql);
        } catch (error) {
            console.error('Error checking columns:', error);
            columnCheck = { rows: [] };
        }

        const hasNewFields = columnCheck.rows.some(col => ['first_name', 'last_name'].includes(col.column_name));
        const hasOldField = columnCheck.rows.some(col => col.column_name === 'name');

        let professorNameSql;
        if (hasNewFields) {
            // Usar los nuevos campos - construir nombre completo
            professorNameSql = `
                TRIM(CONCAT(
                    COALESCE(u.first_name, ''), 
                    ' ', 
                    COALESCE(u.last_name, ''), 
                    CASE 
                        WHEN u.second_last_name IS NOT NULL AND TRIM(u.second_last_name) != '' 
                        THEN CONCAT(' ', u.second_last_name) 
                        ELSE '' 
                    END
                ))
            `;
        } else if (hasOldField) {
            // Usar el campo antiguo name
            professorNameSql = `COALESCE(TRIM(u.name), 'Sin nombre')`;
        } else {
            // No hay campos de nombre disponibles
            professorNameSql = `'Sin nombre'`;
        }

        let sql = `
            SELECT q.*, 
                   ${professorNameSql} as professor_name 
            FROM question_bank q
            LEFT JOIN users u ON q.professor_id = u.id
        `;
        const params = [];
        let whereClauses = [];
        let idx = 1;

        if (filters.institution_id) {
            whereClauses.push(`q.institution_id = $${idx++}`);
            params.push(filters.institution_id);
        }
        if (filters.professor_id) {
            whereClauses.push(`q.professor_id = $${idx++}`);
            params.push(filters.professor_id);
        }

        if (whereClauses.length > 0) {
            sql += ' WHERE ' + whereClauses.join(' AND ');
        }

        sql += ' ORDER BY q.professor_id, q.category, q.created_at ASC';
        const res = await query(sql, params);
        return res.rows;
    },

    async checkDuplicate(professorId, questionText) {
        const res = await query(
            `SELECT * FROM question_bank 
             WHERE professor_id = $1 AND content->>'question' = $2`,
            [professorId, questionText]
        );
        return res.rows[0];
    },

    async getById(id) {
        const res = await query('SELECT * FROM question_bank WHERE id_question = $1', [id]);
        return res.rows[0];
    },

    async create(question) {
        const { institution_id, professor_id, category, type, content, correct_answer } = question;
        const res = await query(
            `INSERT INTO question_bank (institution_id, professor_id, category, type, content, correct_answer)
             VALUES ($1, $2, $3, $4, $5, $6) RETURNING *`,
            [institution_id, professor_id, category, type, content, correct_answer]
        );
        return res.rows[0];
    },

    async update(id, question) {
        const fields = [];
        const values = [];
        let idx = 1;

        // Campos que NO deben actualizarse (son calculados o no existen en la tabla)
        const excludedFields = ['updated_at', 'created_at', 'id_question', 'professor_name'];

        for (const [key, value] of Object.entries(question)) {
            if (!excludedFields.includes(key)) {
                fields.push(`${key} = $${idx}`);
                values.push(value);
                idx++;
            }
        }
        values.push(id);

        const res = await query(
            `UPDATE question_bank SET ${fields.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE id_question = $${idx} RETURNING *`,
            values
        );
        return res.rows[0];
    },

    async delete(id) {
        // Verificar si la pregunta está siendo usada en algún examen
        const usageCheck = await query(
            'SELECT COUNT(*) as count FROM exam_questions WHERE question_id = $1',
            [id]
        );

        const isUsed = parseInt(usageCheck.rows[0].count) > 0;

        if (isUsed) {
            // Si está usada, solo desactivarla
            await query('UPDATE question_bank SET is_active = false WHERE id_question = $1', [id]);
            return { status: 'deactivated' };
        } else {
            // Si no está usada, eliminarla físicamente
            await query('DELETE FROM question_bank WHERE id_question = $1', [id]);
            return { status: 'deleted' };
        }
    },

    async deleteBulk(ids) {
        let deleted = 0;
        let deactivated = 0;
        let errors = 0;

        for (const id of ids) {
            try {
                // Verificar uso
                const usageCheck = await query(
                    'SELECT COUNT(*) as count FROM exam_questions WHERE question_id = $1',
                    [id]
                );
                const isUsed = parseInt(usageCheck.rows[0].count) > 0;

                if (isUsed) {
                    await query('UPDATE question_bank SET is_active = false WHERE id_question = $1', [id]);
                    deactivated++;
                } else {
                    await query('DELETE FROM question_bank WHERE id_question = $1', [id]);
                    deleted++;
                }
            } catch (error) {
                console.error(`Error processing bulk delete for id ${id}:`, error);
                errors++;
            }
        }
        return { deleted, deactivated, errors };
    }
};

