import { query } from './database.js';

export const SpecialtyModel = {
    async getAll(institutionId = null) {
        let sql = `
            SELECT s.*, s.id_specialty as id
            FROM specialties s
        `;
        const params = [];
        
        if (institutionId) {
            sql += ' WHERE s.institution_id = $1';
            params.push(institutionId);
        }
        
        sql += ' ORDER BY s.name ASC';
        
        const res = await query(sql, params);
        return res.rows;
    },

    async getById(id) {
        const res = await query('SELECT * FROM specialties WHERE id_specialty = $1', [id]);
        return res.rows[0];
    },

    async create(specialty) {
        const { institution_id, name, description, is_active } = specialty;
        const res = await query(
            `INSERT INTO specialties (institution_id, name, description, is_active)
             VALUES ($1, $2, $3, $4) RETURNING *`,
            [institution_id, name, description, is_active !== undefined ? is_active : true]
        );
        return res.rows[0];
    },

    async update(id, specialty) {
        const fields = [];
        const values = [];
        let idx = 1;

        const excludedFields = ['id_specialty', 'created_at', 'updated_at'];

        for (const [key, value] of Object.entries(specialty)) {
            if (!excludedFields.includes(key)) {
                fields.push(`${key} = $${idx}`);
                values.push(value);
                idx++;
            }
        }
        values.push(id);

        const res = await query(
            `UPDATE specialties SET ${fields.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE id_specialty = $${idx} RETURNING *`,
            values
        );
        return res.rows[0];
    },

    async delete(id) {
        await query('DELETE FROM specialties WHERE id_specialty = $1', [id]);
        return true;
    },

    async checkDuplicate(institutionId, name, excludeId = null) {
        let sql = `
            SELECT * FROM specialties 
            WHERE institution_id = $1 AND LOWER(TRIM(name)) = LOWER(TRIM($2))
        `;
        const params = [institutionId, name];
        
        if (excludeId) {
            sql += ' AND id_specialty != $3';
            params.push(excludeId);
        }
        
        const res = await query(sql, params);
        return res.rows[0];
    }
};


