import { query } from './database.js';

export const UserModel = {
    async getAll() {
        try {
            // Verificar qué campos existen en la tabla users
            let checkColumnsSql = `
                SELECT column_name 
                FROM information_schema.columns 
                WHERE table_schema = 'public' 
                AND table_name = 'users'
            `;

            let columnCheck;
            let availableColumns = [];
            try {
                columnCheck = await query(checkColumnsSql);
                availableColumns = columnCheck.rows.map(row => row.column_name);
            } catch (error) {
                console.error('Error checking columns:', error);
                // Si falla la verificación, usar consulta simple
                const res = await query(`
                    SELECT id, email, password_hash, role, institution_id, is_active, photo_url,
                           id as user_id, 
                           institution_id as company_id,
                           COALESCE(name, email) as name
                    FROM users
                `);
                return res.rows.map(row => UserModel.processUserRow(row));
            }

            const hasNewFields = availableColumns.includes('first_name') && availableColumns.includes('last_name');
            const hasOldField = availableColumns.includes('name');
            const hasInstitutionId = availableColumns.includes('institution_id');
            const hasCompanyId = availableColumns.includes('company_id');

            // Construir lista de columnas a seleccionar (evitar SELECT *)
            const baseColumns = [
                'id', 'email', 'password_hash', 'role', 'is_active',
                'photo_url', 'phone', 'specialty', 'client_number', 'curp',
                'created_at', 'updated_at'
            ].filter(col => availableColumns.includes(col));

            // Agregar institution_id o company_id según lo que exista
            if (hasInstitutionId) {
                baseColumns.push('institution_id');
            } else if (hasCompanyId) {
                baseColumns.push('company_id');
            }

            // Agregar campos de nombre según lo que exista
            if (hasNewFields) {
                if (availableColumns.includes('first_name')) baseColumns.push('first_name');
                if (availableColumns.includes('last_name')) baseColumns.push('last_name');
                if (availableColumns.includes('second_last_name')) baseColumns.push('second_last_name');
            } else if (hasOldField) {
                baseColumns.push('name');
            }

            let nameSql;
            if (hasNewFields) {
                nameSql = `TRIM(CONCAT(
                    COALESCE(first_name, ''), 
                    ' ', 
                    COALESCE(last_name, ''), 
                    CASE 
                        WHEN second_last_name IS NOT NULL AND TRIM(second_last_name) != '' 
                        THEN CONCAT(' ', second_last_name) 
                        ELSE '' 
                    END
                ))`;
            } else if (hasOldField) {
                nameSql = `COALESCE(TRIM(name), 'Sin nombre')`;
            } else {
                nameSql = `COALESCE(email, 'Sin nombre')`;
            }

            // Construir mapeo de institution_id/company_id
            let institutionMapping = '';
            if (hasInstitutionId) {
                institutionMapping = 'institution_id, institution_id as company_id';
            } else if (hasCompanyId) {
                institutionMapping = 'company_id, company_id as institution_id';
            } else {
                institutionMapping = 'NULL as institution_id, NULL as company_id';
            }

            const res = await query(`
                SELECT ${baseColumns.join(', ')}, 
                       id as user_id, 
                       ${institutionMapping},
                       ${nameSql} as name
                FROM users
            `);
            console.log('UserModel.getAll: Found', res.rows.length, 'users');
            return res.rows.map(row => UserModel.processUserRow(row));
        } catch (error) {
            console.error('UserModel.getAll Error:', error.message);
            throw error;
        }
    },

    async getById(id) {
        // Verificar qué campos existen
        let checkColumnsSql = `
            SELECT column_name 
            FROM information_schema.columns 
            WHERE table_schema = 'public' 
            AND table_name = 'users'
        `;

        let columnCheck;
        let availableColumns = [];
        try {
            columnCheck = await query(checkColumnsSql);
            availableColumns = columnCheck.rows.map(row => row.column_name);
        } catch (error) {
            // Si falla la verificación, intentar con ambos nombres de columna
            try {
                const res = await query(`
                    SELECT id, email, password_hash, role, institution_id, is_active, photo_url,
                           id as user_id, 
                           institution_id as company_id,
                           COALESCE(name, email) as name
                    FROM users 
                    WHERE id = $1
                `, [id]);
                return res.rows[0] ? UserModel.processUserRow(res.rows[0]) : undefined;
            } catch (err) {
                // Si falla con institution_id, intentar con company_id
                const res = await query(`
                    SELECT id, email, password_hash, role, company_id, is_active, photo_url,
                           id as user_id, 
                           company_id as institution_id,
                           company_id as company_id,
                           COALESCE(name, email) as name
                    FROM users 
                    WHERE id = $1
                `, [id]);
                return res.rows[0] ? UserModel.processUserRow(res.rows[0]) : undefined;
            }
        }

        const hasNewFields = availableColumns.includes('first_name') && availableColumns.includes('last_name');
        const hasOldField = availableColumns.includes('name');
        const hasInstitutionId = availableColumns.includes('institution_id');
        const hasCompanyId = availableColumns.includes('company_id');

        // Construir lista de columnas a seleccionar (evitar SELECT *)
        const baseColumns = [
            'id', 'email', 'password_hash', 'role', 'is_active',
            'photo_url', 'phone', 'specialty', 'client_number', 'curp',
            'created_at', 'updated_at'
        ].filter(col => availableColumns.includes(col));

        // Agregar institution_id o company_id según lo que exista
        if (hasInstitutionId) {
            baseColumns.push('institution_id');
        } else if (hasCompanyId) {
            baseColumns.push('company_id');
        }

        // Agregar campos de nombre según lo que exista
        if (hasNewFields) {
            if (availableColumns.includes('first_name')) baseColumns.push('first_name');
            if (availableColumns.includes('last_name')) baseColumns.push('last_name');
            if (availableColumns.includes('second_last_name')) baseColumns.push('second_last_name');
        } else if (hasOldField) {
            baseColumns.push('name');
        }

        let nameSql;
        if (hasNewFields) {
            nameSql = `TRIM(CONCAT(
                COALESCE(first_name, ''), 
                ' ', 
                COALESCE(last_name, ''), 
                CASE 
                    WHEN second_last_name IS NOT NULL AND TRIM(second_last_name) != '' 
                    THEN CONCAT(' ', second_last_name) 
                    ELSE '' 
                END
            ))`;
        } else if (hasOldField) {
            nameSql = `COALESCE(TRIM(name), 'Sin nombre')`;
        } else {
            nameSql = `COALESCE(email, 'Sin nombre')`;
        }

        // Construir mapeo de institution_id/company_id
        let institutionMapping = '';
        if (hasInstitutionId) {
            institutionMapping = 'institution_id, institution_id as company_id';
        } else if (hasCompanyId) {
            institutionMapping = 'company_id, company_id as institution_id';
        } else {
            institutionMapping = 'NULL as institution_id, NULL as company_id';
        }

        const res = await query(`
            SELECT ${baseColumns.join(', ')}, 
                   id as user_id, 
                   ${institutionMapping},
                   ${nameSql} as name
            FROM users 
            WHERE id = $1
        `, [id]);
        return res.rows[0] ? UserModel.processUserRow(res.rows[0]) : undefined;
    },

    async getByEmail(email) {
        try {
            // Normalizar email para búsqueda case-insensitive
            const normalizedEmail = (email || '').trim().toLowerCase();

            // Verificar qué campos existen en la tabla
            let checkColumnsSql = `
                SELECT column_name 
                FROM information_schema.columns 
                WHERE table_schema = 'public' 
                AND table_name = 'users'
            `;

            let columnCheck;
            let availableColumns = [];
            try {
                columnCheck = await query(checkColumnsSql);
                availableColumns = columnCheck.rows.map(row => row.column_name);
            } catch (error) {
                console.error('Error checking columns:', error);
                // Si falla la verificación, intentar con ambos nombres de columna
                try {
                    const res = await query(
                        `SELECT id, email, password_hash, role, institution_id, is_active, photo_url,
                                id as user_id, 
                                institution_id as company_id,
                                COALESCE(name, email) as name
                         FROM users 
                         WHERE LOWER(TRIM(email)) = $1`,
                        [normalizedEmail]
                    );
                    return res.rows[0] ? UserModel.processUserRow(res.rows[0]) : undefined;
                } catch (err) {
                    // Si falla con institution_id, intentar con company_id
                    const res = await query(
                        `SELECT id, email, password_hash, role, company_id, is_active, photo_url,
                                id as user_id, 
                                company_id as institution_id,
                                company_id as company_id,
                                COALESCE(name, email) as name
                         FROM users 
                         WHERE LOWER(TRIM(email)) = $1`,
                        [normalizedEmail]
                    );
                    return res.rows[0] ? UserModel.processUserRow(res.rows[0]) : undefined;
                }
            }

            const hasNewFields = availableColumns.includes('first_name') && availableColumns.includes('last_name');
            const hasOldField = availableColumns.includes('name');
            const hasInstitutionId = availableColumns.includes('institution_id');
            const hasCompanyId = availableColumns.includes('company_id');

            // Construir lista de columnas a seleccionar (evitar SELECT *)
            const baseColumns = [
                'id', 'email', 'password_hash', 'role', 'is_active',
                'photo_url', 'phone', 'specialty', 'client_number', 'curp',
                'created_at', 'updated_at'
            ].filter(col => availableColumns.includes(col));

            // Agregar institution_id o company_id según lo que exista
            if (hasInstitutionId) {
                baseColumns.push('institution_id');
            } else if (hasCompanyId) {
                baseColumns.push('company_id');
            }

            // Agregar campos de nombre según lo que exista
            if (hasNewFields) {
                if (availableColumns.includes('first_name')) baseColumns.push('first_name');
                if (availableColumns.includes('last_name')) baseColumns.push('last_name');
                if (availableColumns.includes('second_last_name')) baseColumns.push('second_last_name');
            } else if (hasOldField) {
                baseColumns.push('name');
            }

            let nameSql;
            if (hasNewFields) {
                nameSql = `TRIM(CONCAT(
                    COALESCE(first_name, ''), 
                    ' ', 
                    COALESCE(last_name, ''), 
                    CASE 
                        WHEN second_last_name IS NOT NULL AND TRIM(second_last_name) != '' 
                        THEN CONCAT(' ', second_last_name) 
                        ELSE '' 
                    END
                ))`;
            } else if (hasOldField) {
                nameSql = `COALESCE(TRIM(name), 'Sin nombre')`;
            } else {
                nameSql = `COALESCE(email, 'Sin nombre')`;
            }

            // Construir mapeo de institution_id/company_id
            let institutionMapping = '';
            if (hasInstitutionId) {
                institutionMapping = 'institution_id, institution_id as company_id';
            } else if (hasCompanyId) {
                institutionMapping = 'company_id, company_id as institution_id';
            } else {
                institutionMapping = 'NULL as institution_id, NULL as company_id';
            }

            const res = await query(
                `SELECT ${baseColumns.join(', ')}, 
                        id as user_id, 
                        ${institutionMapping},
                        ${nameSql} as name
                 FROM users 
                 WHERE LOWER(TRIM(email)) = $1`,
                [normalizedEmail]
            );
            return res.rows[0] ? UserModel.processUserRow(res.rows[0]) : undefined;
        } catch (error) {
            console.error('Error fetching user by email:', error);
            console.error('Error details:', {
                message: error.message,
                code: error.code,
                detail: error.detail,
                hint: error.hint
            });
            throw new Error('Database error');
        }
    },

    async getAdminsByInstitutionId(institutionId) {
        // Verificar qué campos existen
        let checkColumnsSql = `
            SELECT column_name 
            FROM information_schema.columns 
            WHERE table_schema = 'public' 
            AND table_name = 'users'
        `;

        let columnCheck;
        let availableColumns = [];
        try {
            columnCheck = await query(checkColumnsSql);
            availableColumns = columnCheck.rows.map(row => row.column_name);
        } catch (error) {
            // Si falla, intentar con ambos nombres de columna
            try {
                const res = await query(`
                    SELECT *, COALESCE(name, email) as name
                    FROM users 
                    WHERE institution_id = $1 AND role = 'ADMIN'
                `, [institutionId]);
                return res.rows.map(row => UserModel.processUserRow(row));
            } catch (err) {
                const res = await query(`
                    SELECT *, COALESCE(name, email) as name
                    FROM users 
                    WHERE company_id = $1 AND role = 'ADMIN'
                `, [institutionId]);
                return res.rows.map(row => UserModel.processUserRow(row));
            }
        }

        const hasNewFields = availableColumns.includes('first_name') && availableColumns.includes('last_name');
        const hasOldField = availableColumns.includes('name');
        const hasInstitutionId = availableColumns.includes('institution_id');
        const hasCompanyId = availableColumns.includes('company_id');

        // Determinar qué columna usar para el WHERE
        let whereColumn = '';
        if (hasInstitutionId) {
            whereColumn = 'institution_id';
        } else if (hasCompanyId) {
            whereColumn = 'company_id';
        } else {
            throw new Error('No se encontró columna institution_id ni company_id');
        }

        let nameSql;
        if (hasNewFields) {
            nameSql = `TRIM(CONCAT(
                COALESCE(first_name, ''), 
                ' ', 
                COALESCE(last_name, ''), 
                CASE 
                    WHEN second_last_name IS NOT NULL AND TRIM(second_last_name) != '' 
                    THEN CONCAT(' ', second_last_name) 
                    ELSE '' 
                END
            ))`;
        } else if (hasOldField) {
            nameSql = `COALESCE(TRIM(name), 'Sin nombre')`;
        } else {
            nameSql = `COALESCE(email, 'Sin nombre')`;
        }

        const res = await query(`
            SELECT *, 
                   ${nameSql} as name
            FROM users 
            WHERE ${whereColumn} = $1 AND role = 'ADMIN'
        `, [institutionId]);
        return res.rows.map(row => UserModel.processUserRow(row));
    },

    async create(user) {
        let { institution_id, email, password_hash, role, is_active, first_name, last_name, second_last_name, name, phone, specialty, client_number, photo_url, curp } = user;

        // Si viene como company_id (legacy), mapear a institution_id
        const finalInstitutionId = institution_id || user.company_id;

        // Si viene el campo legacy 'name', intentar dividirlo en los nuevos campos
        let finalFirstName = first_name;
        let finalLastName = last_name;
        let finalSecondLastName = second_last_name;

        if (name && !first_name && !last_name) {
            // Intentar dividir el nombre completo en partes
            const nameParts = name.trim().split(/\s+/);
            if (nameParts.length >= 2) {
                finalFirstName = nameParts[0];
                finalLastName = nameParts[1];
                finalSecondLastName = nameParts.length > 2 ? nameParts.slice(2).join(' ') : null;
            } else if (nameParts.length === 1) {
                finalFirstName = nameParts[0];
                finalLastName = '';
            }
        }

        // Validar que first_name y last_name no estén vacíos
        if (!finalFirstName || !finalLastName) {
            throw new Error('El nombre y primer apellido son obligatorios');
        }

        // Validar unicidad de email, phone y curp antes de insertar
        if (email) {
            const existingEmail = await query(
                'SELECT id FROM users WHERE LOWER(TRIM(email)) = $1',
                [email.toLowerCase().trim()]
            );
            if (existingEmail.rows.length > 0) {
                throw new Error('El correo es utilizado por otro usuario, favor de ingresar otro correo electrónico');
            }
        }

        if (phone && phone.trim()) {
            const existingPhone = await query(
                'SELECT id FROM users WHERE phone = $1 AND phone IS NOT NULL',
                [phone.replace(/\s/g, '')]
            );
            if (existingPhone.rows.length > 0) {
                throw new Error('El teléfono es utilizado por otro usuario, favor de ingresar otro teléfono');
            }
        }

        if (curp && curp.trim()) {
            const existingCURP = await query(
                'SELECT id FROM users WHERE UPPER(TRIM(curp)) = $1 AND curp IS NOT NULL',
                [curp.toUpperCase().trim()]
            );
            if (existingCURP.rows.length > 0) {
                throw new Error('El CURP es utilizado por otro usuario, favor de ingresar otro CURP');
            }
        }

        try {
            // Verificar columnas disponibles
            let checkColumnsSql = `
                SELECT column_name 
                FROM information_schema.columns 
                WHERE table_schema = 'public' 
                AND table_name = 'users'
            `;

            let columnCheck;
            let availableColumns = [];
            try {
                columnCheck = await query(checkColumnsSql);
                availableColumns = columnCheck.rows.map(row => row.column_name);
            } catch (error) {
                // Fallback a columnas legacy si falla
                availableColumns = ['name', 'email', 'password_hash', 'role', 'is_active', 'institution_id', 'phone', 'specialty', 'client_number', 'photo_url', 'curp', 'company_id'];
            }

            const hasNewFields = availableColumns.includes('first_name') && availableColumns.includes('last_name');
            const hasOldField = availableColumns.includes('name');
            const hasInstitutionId = availableColumns.includes('institution_id');
            const hasCompanyId = availableColumns.includes('company_id');

            // Construir campos a insertar
            const insertFields = ['email', 'password_hash', 'role', 'is_active', 'phone', 'specialty', 'client_number', 'photo_url', 'curp'];
            const insertValues = [email, password_hash, role, is_active !== undefined ? is_active : true, phone, specialty, client_number, photo_url, curp];
            const insertPlaceholders = ['$1', '$2', '$3', '$4', '$5', '$6', '$7', '$8', '$9'];
            let paramIdx = 10;

            if (hasInstitutionId) {
                insertFields.push('institution_id');
                insertValues.push(finalInstitutionId);
                insertPlaceholders.push(`$${paramIdx}`);
                paramIdx++;
            } else if (hasCompanyId) {
                insertFields.push('company_id');
                insertValues.push(finalInstitutionId);
                insertPlaceholders.push(`$${paramIdx}`);
                paramIdx++;
            }

            if (hasNewFields) {
                insertFields.push('first_name', 'last_name');
                insertValues.push(finalFirstName, finalLastName);
                insertPlaceholders.push(`$${paramIdx}`, `$${paramIdx + 1}`);
                paramIdx += 2;

                if (finalSecondLastName && availableColumns.includes('second_last_name')) {
                    insertFields.push('second_last_name');
                    insertValues.push(finalSecondLastName);
                    insertPlaceholders.push(`$${paramIdx}`);
                    paramIdx++;
                }
            } else if (hasOldField) {
                const fullName = [finalFirstName, finalLastName, finalSecondLastName].filter(Boolean).join(' ');
                insertFields.push('name');
                insertValues.push(fullName);
                insertPlaceholders.push(`$${paramIdx}`);
                paramIdx++;
            }

            // Construir SQL de retorno
            let nameSql;
            if (hasNewFields) {
                nameSql = `TRIM(CONCAT(
                    COALESCE(first_name, ''), 
                    ' ', 
                    COALESCE(last_name, ''), 
                    CASE 
                        WHEN second_last_name IS NOT NULL AND TRIM(second_last_name) != '' 
                        THEN CONCAT(' ', second_last_name) 
                        ELSE '' 
                    END
                ))`;
            } else if (hasOldField) {
                nameSql = `COALESCE(TRIM(name), 'Sin nombre')`;
            } else {
                nameSql = `'Sin nombre'`;
            }

            let institutionMapping = '';
            if (hasInstitutionId) {
                institutionMapping = 'institution_id as company_id';
            } else if (hasCompanyId) {
                institutionMapping = 'company_id as institution_id';
            }

            const res = await query(
                `INSERT INTO users (${insertFields.join(', ')})
                 VALUES (${insertPlaceholders.join(', ')})
                 RETURNING *, 
                          id as user_id, 
                          ${institutionMapping},
                          ${nameSql} as name`,
                insertValues
            );
            return res.rows[0];
        } catch (error) {
            console.error('UserModel.create - DB Error:', error.message);
            const errorMsg = error.message || '';

            // Detectar error de email duplicado
            if (errorMsg.includes('users_email_key') || (errorMsg.includes('duplicate key') && errorMsg.includes('email'))) {
                throw new Error('El correo es utilizado por otro usuario, favor de ingresar otro correo electrónico');
            }

            // Detectar error de teléfono duplicado
            if (errorMsg.includes('users_phone_key') || (errorMsg.includes('duplicate key') && errorMsg.includes('phone'))) {
                throw new Error('El teléfono es utilizado por otro usuario, favor de ingresar otro teléfono');
            }

            // Detectar error de CURP duplicado
            if (errorMsg.includes('users_curp_key') || (errorMsg.includes('duplicate key') && errorMsg.includes('curp'))) {
                throw new Error('El CURP es utilizado por otro usuario, favor de ingresar otro CURP');
            }

            // Detectar otros errores de constraint único genérico
            if (errorMsg.includes('duplicate key') || errorMsg.includes('unique constraint')) {
                // Intentar determinar qué campo es
                if (errorMsg.includes('email')) {
                    throw new Error('El correo es utilizado por otro usuario, favor de ingresar otro correo electrónico');
                } else if (errorMsg.includes('phone')) {
                    throw new Error('El teléfono es utilizado por otro usuario, favor de ingresar otro teléfono');
                } else if (errorMsg.includes('curp')) {
                    throw new Error('El CURP es utilizado por otro usuario, favor de ingresar otro CURP');
                }
            }

            throw error;
        }
    },


    async update(id, user) {
        let { first_name, last_name, second_last_name, name } = user;

        // Clonar y limpiar datos
        const userData = { ...user };
        delete userData.user_id;
        delete userData.id;
        delete userData.updated_at;
        delete userData.created_at;
        delete userData.password; // Manejado por separado si existe
        delete userData._pendingFile;
        delete userData.name; // Campo virtual o legacy

        // Mapear company_id a institution_id si viene (legacy support)
        if (userData.company_id !== undefined && userData.institution_id === undefined) {
            // Solo si institution_id existe en DB se usará luego, pero preparamos el mapeo
            userData.institution_id = userData.company_id;
        }
        delete userData.company_id;

        // Validar unicidad de email, phone y curp antes de actualizar (excluyendo el usuario actual)
        if (userData.email) {
            const existingEmail = await query(
                'SELECT id FROM users WHERE LOWER(TRIM(email)) = $1 AND id != $2',
                [userData.email.toLowerCase().trim(), id]
            );
            if (existingEmail.rows.length > 0) {
                throw new Error('El correo es utilizado por otro usuario, favor de ingresar otro correo electrónico');
            }
        }

        if (userData.phone && userData.phone.trim()) {
            const cleanPhone = userData.phone.replace(/\s/g, '');
            const existingPhone = await query(
                'SELECT id FROM users WHERE phone = $1 AND phone IS NOT NULL AND id != $2',
                [cleanPhone, id]
            );
            if (existingPhone.rows.length > 0) throw new Error('El teléfono es utilizado por otro usuario');
        }

        if (userData.curp && userData.curp.trim()) {
            const cleanCURP = userData.curp.toUpperCase().trim();
            const existingCURP = await query(
                'SELECT id FROM users WHERE UPPER(TRIM(curp)) = $1 AND curp IS NOT NULL AND id != $2',
                [cleanCURP, id]
            );
            if (existingCURP.rows.length > 0) throw new Error('El CURP es utilizado por otro usuario');
        }

        // Verificar columnas
        let checkColumnsSql = `
            SELECT column_name 
            FROM information_schema.columns 
            WHERE table_schema = 'public' 
            AND table_name = 'users'
        `;

        let columnCheck;
        let availableColumns = [];
        try {
            columnCheck = await query(checkColumnsSql);
            availableColumns = columnCheck.rows.map(row => row.column_name);
        } catch (error) {
            availableColumns = ['name', 'email', 'password_hash', 'role', 'is_active', 'institution_id', 'phone', 'specialty', 'client_number', 'photo_url', 'curp', 'company_id'];
        }

        const hasNewFields = availableColumns.includes('first_name') && availableColumns.includes('last_name');
        const hasOldField = availableColumns.includes('name');

        // Manejo de nombres
        let finalFirstName = first_name;
        let finalLastName = last_name;
        let finalSecondLastName = second_last_name;

        // Si no vienen first_name/last_name pero viene name (o si necesitamos actualizar name legacy)
        if (user.name && (!first_name || !last_name)) {
            const nameParts = user.name.trim().split(/\s+/);
            if (nameParts.length >= 2) {
                finalFirstName = nameParts[0];
                finalLastName = nameParts[1];
                finalSecondLastName = nameParts.length > 2 ? nameParts.slice(2).join(' ') : null;
            } else {
                finalFirstName = nameParts[0];
                finalLastName = '';
            }
        }

        // Validar que first_name y last_name no estén vacíos si se están actualizando
        if (hasNewFields) {
            if ((user.first_name === '' || user.first_name === null) || (user.last_name === '' || user.last_name === null)) {
                // Solo validar si explícitamente se están limpiando, o si es un update parcial que debería tenerlos
                // En un update parcial, si no vienen, no se tocan. Si vienen vacíos, es error.
                if (user.hasOwnProperty('first_name') && !user.first_name) throw new Error('El nombre es obligatorio');
                if (user.hasOwnProperty('last_name') && !user.last_name) throw new Error('El primer apellido es obligatorio');
            }
        }

        // Construir campos a actualizar
        const fields = [];
        const values = [];
        let idx = 1;

        // Agregar campos estándar si existen en userData y en AvailableColumns
        for (const [key, value] of Object.entries(userData)) {
            // Saltamos first_name, last_name, second_last_name, institution_id para manejo especial
            if (['first_name', 'last_name', 'second_last_name', 'institution_id'].includes(key)) continue;

            if (availableColumns.includes(key)) {
                fields.push(`${key} = $${idx}`);
                values.push(value);
                idx++;
            }
        }

        // Manejo institution_id vs company_id
        if (userData.institution_id !== undefined) {
            if (availableColumns.includes('institution_id')) {
                fields.push(`institution_id = $${idx}`);
                values.push(userData.institution_id);
                idx++;
            } else if (availableColumns.includes('company_id')) {
                fields.push(`company_id = $${idx}`);
                values.push(userData.institution_id);
                idx++;
            }
        }

        // Manejo de campos de nombre
        if (hasNewFields) {
            // Actualizar first_name/last_name si se proporcionaron (directamente o vía name split)
            if (finalFirstName !== undefined) {
                fields.push(`first_name = $${idx}`);
                values.push(finalFirstName);
                idx++;
            }
            if (finalLastName !== undefined) {
                fields.push(`last_name = $${idx}`);
                values.push(finalLastName);
                idx++;
            }
            if (finalSecondLastName !== undefined && availableColumns.includes('second_last_name')) {
                fields.push(`second_last_name = $${idx}`);
                values.push(finalSecondLastName);
                idx++;
            }
        } else if (hasOldField) {
            // Actualizar campo name legacy
            // Si tenemos las partes nuevas, reconstruimos name. Si no, usamos user.name si existe
            let newFullName = user.name;
            if ((finalFirstName || finalLastName) && !newFullName) {
                newFullName = [finalFirstName, finalLastName, finalSecondLastName].filter(Boolean).join(' ');
            }

            if (newFullName !== undefined) {
                fields.push(`name = $${idx}`);
                values.push(newFullName);
                idx++;
            }
        }

        if (fields.length === 0) {
            // Nada que actualizar, devolver el usuario actual
            return UserModel.getById(id);
        }

        values.push(id); // ID para el WHERE

        try {
            // Construir SQL de retorno (nombre)
            let nameSql;
            if (hasNewFields) {
                nameSql = `TRIM(CONCAT(COALESCE(first_name, ''), ' ', COALESCE(last_name, ''), CASE WHEN second_last_name IS NOT NULL THEN CONCAT(' ', second_last_name) ELSE '' END))`;
            } else if (hasOldField) {
                nameSql = `COALESCE(TRIM(name), 'Sin nombre')`;
            } else {
                nameSql = `'Sin nombre'`;
            }

            let institutionMapping = '';
            if (availableColumns.includes('institution_id')) {
                institutionMapping = 'institution_id, institution_id as company_id';
            } else if (availableColumns.includes('company_id')) {
                institutionMapping = 'company_id, company_id as institution_id';
            } else {
                institutionMapping = 'NULL as institution_id, NULL as company_id';
            }

            const res = await query(
                `UPDATE users SET ${fields.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE id = $${idx} 
                 RETURNING *, 
                          id as user_id, 
                          ${institutionMapping},
                          ${nameSql} as name`,
                values
            );
            return res.rows[0];

        } catch (error) {
            console.error('UserModel.update - DB Error:', error.message);
            const errorMsg = error.message || '';
            // Propagar errores de duplicados igual que en create
            if (errorMsg.includes('duplicate key') || errorMsg.includes('unique constraint')) {
                if (errorMsg.includes('email')) throw new Error('El correo es utilizado por otro usuario');
                if (errorMsg.includes('phone')) throw new Error('El teléfono es utilizado por otro usuario');
                if (errorMsg.includes('curp')) throw new Error('El CURP es utilizado por otro usuario');
            }
            throw error;
        }
    },

    async delete(id) {
        await query('DELETE FROM users WHERE id = $1', [id]);
        return true;
    },

    processUserRow(row) {
        if (!row) return row;

        // Si no existen first_name/last_name pero existe name, intentar parsear
        if ((!row.first_name || !row.last_name) && row.name && row.name !== 'Sin nombre') {
            const nameParts = row.name.trim().split(/\s+/);

            if (!row.first_name) {
                row.first_name = nameParts[0] || '';
            }

            if (!row.last_name) {
                if (nameParts.length > 2) {
                    row.last_name = nameParts[1];
                    row.second_last_name = nameParts.slice(2).join(' ');
                } else if (nameParts.length === 2) {
                    row.last_name = nameParts[1];
                    row.second_last_name = '';
                } else {
                    row.last_name = '';
                    row.second_last_name = '';
                }
            }
        }

        return row;
    }
};
