import { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import {
  Box,
  Flex,
  Heading,
  FormControl,
  FormLabel,
  Input,
  Button,
  VStack,
  Text,
  Image,
  useToast,
  InputGroup,
  InputLeftElement,
  Icon,
  Link,
  HStack,
  InputRightElement
} from '@chakra-ui/react';
import { FiUser, FiLock, FiArrowRight, FiEye, FiEyeOff } from 'react-icons/fi';
import { login } from '../services/api';

const LoginPage = () => {
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [showPassword, setShowPassword] = useState(false);
  const [isLoading, setIsLoading] = useState(false);

  const navigate = useNavigate();
  const toast = useToast();

  // Cargar credenciales guardadas
  useEffect(() => {
    setEmail(localStorage.getItem('remember_email') || '');
  }, []);

  const handleLogin = async (e) => {
    e.preventDefault();

    if (!email || !password) {
      toast({
        title: 'Error',
        description: 'Por favor ingresa tu correo y contraseña',
        status: 'error',
        duration: 3000,
        isClosable: true,
      });
      return;
    }

    setIsLoading(true);

    try {
      const { token, user } = await login(email, password);

      // Login exitoso
      toast({
        title: 'Bienvenido',
        description: `Acceso concedido como ${user.role}`,
        status: 'success',
        duration: 2000,
        isClosable: true,
      });

      // Guardar sesión
      sessionStorage.setItem('token', token);
      sessionStorage.setItem('user', JSON.stringify(user));

      // Guardar correo para la próxima vez
      localStorage.setItem('remember_email', email);

      // Redirección centralizada por rol
      const role = user.role.toUpperCase();
      if (role === 'ROOT') {
        navigate('/root');
      } else if (role === 'ADMIN') {
        navigate('/admin');
      } else if (role === 'PROFESOR' || role === 'EMPLOYEE') {
        navigate('/professor');
      } else if (role === 'ESTUDIANTE' || role === 'CLIENT') {
        navigate('/student');
      } else {
        navigate('/');
      }

    } catch (error) {
      console.error(error);
      toast({
        title: 'Acceso denegado',
        description: error.message || 'Credenciales incorrectas',
        status: 'error',
        duration: 3000,
        isClosable: true,
      });
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <Flex minH="100vh" overflow="hidden">
      {/* Left Side - Branding */}
      <Box
        w={{ base: '0%', lg: '50%' }}
        bg="gray.900"
        position="relative"
        display={{ base: 'none', lg: 'block' }}
      >
        <Image
          src="/construction_maintenance_bg_1765306718080.png"
          alt="Evaluaciones KeQuiz"
          w="100%"
          h="100%"
          objectFit="cover"
          opacity={0.7}
        />
        <Box
          position="absolute"
          top="0"
          left="0"
          w="100%"
          h="100%"
          bgGradient="linear(to-b, rgba(0,0,0,0.4), rgba(0,0,0,0.8))"
          display="flex"
          flexDirection="column"
          justifyContent="flex-end"
          p={16}
        >
          <VStack align="flex-start" spacing={6} maxW="lg">
            <Box
              bg="white"
              p={2}
              borderRadius="xl"
              boxShadow="lg"
              w="80px"
              h="80px"
              display="flex"
              alignItems="center"
              justifyContent="center"
              overflow="hidden"
            >
              <Image src="/logo.jpeg" alt="Logo" objectFit="cover" w="100%" h="100%" />
            </Box>
            <Heading color="white" fontSize="5xl" fontWeight="bold" lineHeight="1.1">
              Evaluaciones para <br /> Personal Especializado
            </Heading>
            <Text color="gray.300" fontSize="xl" maxW="md">
              Portal integral para la aplicación, gestión y monitoreo en tiempo real de evaluaciones académicas y profesionales.
            </Text>
            <HStack spacing={4} pt={4}>
              <Text color="whiteAlpha.800" fontSize="sm" fontWeight="600">
                © 2026 KeSoftware
              </Text>
              <Box w="1px" h="16px" bg="whiteAlpha.400" />
              <Link color="whiteAlpha.800" fontSize="sm" _hover={{ color: 'white' }}>
                Privacidad
              </Link>
              <Link color="whiteAlpha.800" fontSize="sm" _hover={{ color: 'white' }}>
                Términos
              </Link>
            </HStack>
          </VStack>
        </Box>
      </Box>

      {/* Right Side - Login Form */}
      <Flex
        w={{ base: '100%', lg: '50%' }}
        bg="white"
        align="center"
        justify="center"
        p={8}
      >
        <Box w="100%" maxW="md" mt={{ base: 0, md: -16 }}>
          <VStack spacing={8} align="stretch">
            <Box textAlign="center" mb={8}>
              <Image
                src="/logop.jpg"
                alt="Logo"
                mx="auto"
                mb={8}
                h="160px"
                objectFit="contain"
              />
              <Heading fontSize="3xl" fontWeight="bold" color="gray.900">
                Bienvenido a KeQuiz
              </Heading>
              <Text color="gray.500" mt={2}>
                Ingresa tus credenciales para acceder a tu panel
              </Text>
            </Box>

            <Box as="form" onSubmit={handleLogin}>
              <VStack spacing={5}>
                <FormControl>
                  <FormLabel fontWeight="600" color="gray.700">Correo Electrónico</FormLabel>
                  <InputGroup>
                    <InputLeftElement pointerEvents="none">
                      <Icon as={FiUser} color="gray.400" />
                    </InputLeftElement>
                    <Input
                      type="email"
                      value={email}
                      onChange={(e) => setEmail(e.target.value)}
                      placeholder="tu@email.com"
                      size="lg"
                      borderRadius="lg"
                      bg="gray.50"
                      border="1px solid"
                      borderColor="gray.200"
                      _focus={{ borderColor: 'brand.500', bg: 'white', boxShadow: 'none' }}
                    />
                  </InputGroup>
                </FormControl>

                <FormControl>
                  <FormLabel fontWeight="600" color="gray.700">Contraseña</FormLabel>
                  <InputGroup>
                    <InputLeftElement pointerEvents="none">
                      <Icon as={FiLock} color="gray.400" />
                    </InputLeftElement>
                    <Input
                      type={showPassword ? "text" : "password"}
                      value={password}
                      onChange={(e) => setPassword(e.target.value)}
                      placeholder="••••••••"
                      size="lg"
                      borderRadius="lg"
                      bg="gray.50"
                      border="1px solid"
                      borderColor="gray.200"
                      _focus={{ borderColor: 'brand.500', bg: 'white', boxShadow: 'none' }}
                    />
                    <InputRightElement h="full" pr={1}>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => setShowPassword(!showPassword)}
                      >
                        {showPassword ? <FiEyeOff /> : <FiEye />}
                      </Button>
                    </InputRightElement>
                  </InputGroup>
                </FormControl>

                <Button
                  type="submit"
                  colorScheme="brand"
                  size="lg"
                  w="100%"
                  h="56px"
                  fontSize="md"
                  borderRadius="lg"
                  rightIcon={<FiArrowRight />}
                  _hover={{ transform: 'translateY(-1px)', boxShadow: 'lg' }}
                  transition="all 0.2s"
                  isLoading={isLoading}
                  loadingText="Verificando..."
                >
                  Entrar al Sistema
                </Button>
              </VStack>
            </Box>

            <Box p={5} bg="indigo.50" borderRadius="xl" border="1px dashed" borderColor="indigo.200">
              <Text fontSize="sm" color="indigo.800" fontWeight="600" mb={2}>
                ¿Olvidaste tu contraseña?
              </Text>
              <Text fontSize="xs" color="indigo.600">
                Contacta al administrador de tu institución para restablecer tus credenciales de acceso.
              </Text>
            </Box>
          </VStack>
        </Box>
      </Flex>

      {/* Footer */}
      <Box
        position="absolute"
        bottom="0"
        left="0"
        right="0"
        py={4}
        bg="rgba(255, 255, 255, 0.8)"
        backdropFilter="blur(10px)"
        borderTop="1px solid"
        borderColor="gray.200"
      >
        <Text textAlign="center" fontSize="xs" color="gray.600">
          © 2026 Desarrollado por{' '}
          <Text as="span" fontWeight="700" color="brand.600">
            KeSoftware
          </Text>
          {' '}• Todos los derechos reservados.
        </Text>
      </Box>
    </Flex>
  );
};

export default LoginPage;
