import { useState, useEffect } from 'react';
import {
  Box,
  SimpleGrid,
  Stat,
  StatLabel,
  StatNumber,
  StatHelpText,
  StatArrow,
  Card,
  CardHeader,
  CardBody,
  Heading,
  Text,
  Icon,
  Flex,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Button,
  useDisclosure,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  FormControl,
  FormLabel,
  Select,
  Input,
  useToast,
  Badge
} from '@chakra-ui/react';
import { FiUsers, FiClipboard, FiCheckCircle, FiClock, FiAlertCircle, FiEdit2 } from 'react-icons/fi';
import { getOrders, getClients, getEmployees, updateOrder, getServicesByCompany } from '../../services/api';

// Función para traducir estados a leyendas amigables
const getStatusLabel = (status) => {
  const statusMap = {
    'PENDING': 'Pendiente',
    'IN_PROGRESS': 'En Progreso',
    'FINISHED': 'Finalizado',
    'CANCELLED': 'Cancelado',
    'CERRADO_ADMIN': 'Cerrado',
    'IN_REVIEW': 'Reapertura'
  };
  return statusMap[status] || status;
};

const AdminDashboard = () => {
  // ... existing component code ...

  const [orders, setOrders] = useState([]);
  const [clients, setClients] = useState([]);
  const [employees, setEmployees] = useState([]);
  const [servicesCatalog, setServicesCatalog] = useState([]); // Added services state
  const [currentOrder, setCurrentOrder] = useState(null);
  const { isOpen, onOpen, onClose } = useDisclosure();
  const toast = useToast();

  const fetchDashboardData = async () => {
    try {
      const user = JSON.parse(sessionStorage.getItem('user'));
      const institutionId = user ? (user.institution_id || user.company_id) : null;

      if (institutionId) {
        const [allOrders, allClients, allEmployees, companyServices] = await Promise.all([
          getOrders(),
          getClients(),
          getEmployees(),
          getServicesByCompany(institutionId)
        ]);

        // Filter by company/institution
        setOrders(allOrders.filter(o => o.institution_id === institutionId || o.company_id === institutionId));
        setClients(allClients.filter(c => c.institution_id === institutionId || c.company_id === institutionId));
        setEmployees(allEmployees.filter(e => e.institution_id === institutionId || e.company_id === institutionId));
        setServicesCatalog(companyServices);
      }
    } catch (error) {
      console.error("Error fetching dashboard data:", error);
    }
  };

  useEffect(() => {
    fetchDashboardData();
  }, []);

  const handleEdit = (order) => {
    setCurrentOrder({ ...order }); // Clone order
    onOpen();
  };

  const handleSave = async () => {
    if (!currentOrder) return;

    // Validar conflicto de agenda si hay profesor y fechas asignadas
    if (currentOrder.employee_id && currentOrder.request_date && currentOrder.deadline_date) {
      // Obtener todas las órdenes para verificar conflictos
      try {
        const allOrders = await getOrders();
        const conflict = allOrders.find(o =>
          o.id !== currentOrder.id && // No validarse a sí misma
          o.status !== 'CANCELLED' &&
          o.status !== 'FINISHED' &&
          parseInt(o.employee_id) === parseInt(currentOrder.employee_id) &&
          (
            (new Date(currentOrder.request_date) <= new Date(o.deadline_date)) &&
            (new Date(currentOrder.deadline_date) >= new Date(o.request_date))
          )
        );

        if (conflict) {
          const confirmAssignment = window.confirm(
            `⚠️ CONFLICTO DETECTADO\n\nEl profesor seleccionado ya tiene un examen asignado en este horario (Examen: ${conflict.service_number}).\n\n¿Desea asignar de todas formas?`
          );
          if (!confirmAssignment) {
            return; // Cancelar guardado
          }
        }
      } catch (error) {
        console.error("Error validando conflictos:", error);
        // Decidir si bloquear o permitir. Permitimos con log.
      }
    }

    try {
      // Clean object before sending to avoid SQL errors with joined fields
      const { employee_name, employee_photo_url, ...cleanOrder } = currentOrder;
      await updateOrder(currentOrder.id, cleanOrder);

      toast({
        title: 'Examen actualizado',
        status: 'success',
        duration: 2000,
      });
      fetchDashboardData();
      onClose();
    } catch (error) {
      toast({
        title: 'Error al actualizar',
        description: error.message,
        status: 'error',
        duration: 3000,
      });
    }
  };

  // Estadísticas reales
  const currentYear = new Date().getFullYear();
  const yearlyOrders = orders.filter(o => new Date(o.request_date).getFullYear() === currentYear);

  const activeServices = yearlyOrders.filter(s => s.status === 'IN_PROGRESS' || s.status === 'IN_REVIEW').length;
  const pendingServices = yearlyOrders.filter(s => s.status === 'PENDING').length;
  const finishedServices = yearlyOrders.filter(s => s.status === 'FINISHED' || s.status === 'CERRADO_ADMIN').length;
  const totalClients = clients.length;

  const statusOrder = { 'IN_REVIEW': 1, 'PENDING': 2, 'IN_PROGRESS': 3, 'FINISHED': 4, 'CERRADO_ADMIN': 5, 'CANCELLED': 6 };

  const sortedOrders = [...yearlyOrders].sort((a, b) => {
    const orderA = statusOrder[a.status] || 99;
    const orderB = statusOrder[b.status] || 99;
    if (orderA !== orderB) {
      return orderA - orderB;
    }
    return new Date(b.request_date) - new Date(a.request_date);
  });

  return (
    <Box>
      <Heading mb={6}>Panel Operativo</Heading>

      {/* Stats Grid */}
      <SimpleGrid columns={{ base: 1, md: 2, lg: 4 }} spacing={6} mb={8}>
        <Card>
          <CardBody>
            <Stat>
              <Flex alignItems="center" mb={2}>
                <Box p={2} bg="blue.100" borderRadius="md" mr={3}>
                  <Icon as={FiClipboard} color="blue.500" boxSize={6} />
                </Box>
                <StatLabel fontSize="lg">Exámenes Activos</StatLabel>
              </Flex>
              <StatNumber>{activeServices}</StatNumber>
              <StatHelpText>
                En progreso actualmente
              </StatHelpText>
            </Stat>
          </CardBody>
        </Card>

        <Card>
          <CardBody>
            <Stat>
              <Flex alignItems="center" mb={2}>
                <Box p={2} bg="orange.100" borderRadius="md" mr={3}>
                  <Icon as={FiClock} color="orange.500" boxSize={6} />
                </Box>
                <StatLabel fontSize="lg">Pendientes</StatLabel>
              </Flex>
              <StatNumber>{pendingServices}</StatNumber>
              <StatHelpText>
                Por asignar o iniciar
              </StatHelpText>
            </Stat>
          </CardBody>
        </Card>

        <Card>
          <CardBody>
            <Stat>
              <Flex alignItems="center" mb={2}>
                <Box p={2} bg="green.100" borderRadius="md" mr={3}>
                  <Icon as={FiCheckCircle} color="green.500" boxSize={6} />
                </Box>
                <StatLabel fontSize="lg">Finalizados</StatLabel>
              </Flex>
              <StatNumber>{finishedServices}</StatNumber>
              <StatHelpText>
                Este mes
              </StatHelpText>
            </Stat>
          </CardBody>
        </Card>

        <Card>
          <CardBody>
            <Stat>
              <Flex alignItems="center" mb={2}>
                <Box p={2} bg="purple.100" borderRadius="md" mr={3}>
                  <Icon as={FiUsers} color="purple.500" boxSize={6} />
                </Box>
                <StatLabel fontSize="lg">Estudiantes</StatLabel>
              </Flex>
              <StatNumber>{totalClients}</StatNumber>
              <StatHelpText>
                <StatArrow type="increase" />
                Totales en la institución
              </StatHelpText>
            </Stat>
          </CardBody>
        </Card>
      </SimpleGrid>

      {/* Solicitudes por Confirmar */}
      {orders.filter(o => o.status === 'PENDING' && o.service_number?.startsWith('REQ-') && !o.employee_id).length > 0 && (
        <Card mb={8} borderLeft="4px solid" borderColor="orange.400" bg="orange.50">
          <CardHeader pb={2}>
            <Heading size="md" color="orange.800">
              Solicitudes por Confirmar ({orders.filter(o => o.status === 'PENDING' && o.service_number?.startsWith('REQ-') && !o.employee_id).length})
            </Heading>
          </CardHeader>
          <CardBody>
            <Text mb={4} fontSize="sm" color="orange.900">
              Hay {orders.filter(o => o.status === 'PENDING' && o.service_number?.startsWith('REQ-') && !o.employee_id).length} solicitude(s) nueva(s) esperando asignación.
            </Text>
            <Table variant="simple" size="sm" bg="white" borderRadius="md">
              <Thead>
                <Tr>
                  <Th>Folio</Th>
                  <Th>Estudiante</Th>
                  <Th>Examen</Th>
                  <Th>Fecha</Th>
                  <Th>Acción</Th>
                </Tr>
              </Thead>
              <Tbody>
                {orders.filter(o => o.status === 'PENDING' && o.service_number?.startsWith('REQ-') && !o.employee_id).slice(0, 5).map(order => {
                  const client = clients.find(c => (c.id === order.client_id || c.user_id === order.client_id));
                  return (
                    <Tr key={order.id || order.id_exam}>
                      <Td fontWeight="bold">{order.service_number}</Td>
                      <Td>{client?.name || 'Desconocido'}</Td>
                      <Td>{order.service_catalogo_id}</Td>
                      <Td>{new Date(order.request_date).toLocaleDateString()}</Td>
                      <Td>
                        <Button size="xs" colorScheme="blue" leftIcon={<FiEdit2 />} onClick={() => handleEdit(order)}>
                          Confirmar
                        </Button>
                      </Td>
                    </Tr>
                  );
                })}
              </Tbody>
            </Table>
          </CardBody>
        </Card>
      )}

      {/* Recent Orders */}
      <Card mb={8}>
        <CardHeader>
          <Heading size="md">Actividad Reciente</Heading>
        </CardHeader>
        <CardBody>
          {sortedOrders.length === 0 ? (
            <Flex direction="column" align="center" justify="center" p={8}>
              <Icon as={FiClipboard} boxSize={10} color="gray.300" mb={3} />
              <Text color="gray.500">No hay actividad reciente</Text>
            </Flex>
          ) : (
            <Table variant="simple" size="sm">
              <Thead>
                <Tr>
                  <Th>Folio</Th>
                  <Th>Estudiante</Th>
                  <Th>Estado</Th>
                  <Th>Fecha Solicitud</Th>
                </Tr>
              </Thead>
              <Tbody>
                {sortedOrders.slice(0, 7).map((service) => {
                  const client = clients.find(c => (c.id === service.client_id || c.user_id === service.client_id));
                  return (
                    <Tr key={service.id || service.id_exam} bg={service.status === 'IN_REVIEW' ? 'purple.50' : 'transparent'}>
                      <Td fontWeight="bold">{service.service_number}</Td>
                      <Td>{client?.name || 'Desconocido'}</Td>
                      <Td>
                        <Badge
                          colorScheme={
                            service.status === 'FINISHED' ? 'green' :
                              service.status === 'IN_PROGRESS' ? 'blue' :
                                service.status === 'PENDING' ? 'orange' :
                                  service.status === 'IN_REVIEW' ? 'purple' : 'gray'
                          }
                        >
                          {getStatusLabel(service.status)}
                        </Badge>
                      </Td>
                      <Td>{new Date(service.request_date).toLocaleDateString()}</Td>
                    </Tr>
                  );
                })}
              </Tbody>
            </Table>
          )}
        </CardBody>
      </Card>






      {/* Edit Modal */}
      <Modal isOpen={isOpen} onClose={onClose} size="lg">
        <ModalOverlay />
        <ModalContent>
          <ModalHeader>Editar Solicitud {currentOrder?.service_number}</ModalHeader>
          <ModalCloseButton />
          <ModalBody>
            <Box display="flex" flexDirection="column" gap={4}>
              <FormControl>
                <FormLabel>Estudiante</FormLabel>
                <Select
                  value={currentOrder?.client_id || ''}
                  onChange={(e) => setCurrentOrder({ ...currentOrder, client_id: e.target.value })}
                >
                  <option value="">Seleccionar...</option>
                  {clients.map(c => (
                    <option key={c.id || c.user_id} value={c.id || c.user_id}>{c.name}</option>
                  ))}
                </Select>
              </FormControl>

              <FormControl>
                <FormLabel>Examen</FormLabel>
                <Select
                  value={currentOrder?.service_catalogo_id || ''}
                  onChange={(e) => setCurrentOrder({ ...currentOrder, service_catalogo_id: parseInt(e.target.value) })}
                >
                  <option value="">Seleccionar...</option>
                  {servicesCatalog.map(s => (
                    <option key={s.id || s.id_exam} value={s.id || s.id_exam}>{s.name}</option>
                  ))}
                </Select>
              </FormControl>

              <FormControl>
                <FormLabel>Asignar Profesor</FormLabel>
                <Select
                  placeholder="Seleccionar profesor"
                  value={currentOrder?.employee_id || ''}
                  onChange={(e) => setCurrentOrder({ ...currentOrder, employee_id: e.target.value ? parseInt(e.target.value) : null })}
                >
                  {employees.map(e => (
                    <option key={e.id || e.user_id} value={e.id || e.user_id}>{e.name} ({e.specialty})</option>
                  ))}
                </Select>
              </FormControl>

              <Flex gap={4}>
                <FormControl>
                  <FormLabel>Fecha Solicitud</FormLabel>
                  <Input
                    type="date"
                    value={currentOrder?.request_date?.split('T')[0] || ''}
                    onChange={(e) => setCurrentOrder({ ...currentOrder, request_date: e.target.value })}
                  />
                </FormControl>

                <FormControl>
                  <FormLabel>Fecha Límite</FormLabel>
                  <Input
                    type="date"
                    value={currentOrder?.deadline_date?.split('T')[0] || ''}
                    onChange={(e) => setCurrentOrder({ ...currentOrder, deadline_date: e.target.value })}
                  />
                </FormControl>
              </Flex>

              <FormControl>
                <FormLabel>Estado</FormLabel>
                <Select
                  value={currentOrder?.status || ''}
                  onChange={(e) => setCurrentOrder({ ...currentOrder, status: e.target.value })}
                >
                  <option value="PENDING">Pendiente</option>
                  <option value="IN_PROGRESS">En Progreso</option>
                  <option value="FINISHED">Finalizado</option>
                  <option value="CANCELLED">Cancelado</option>
                  <option value="CERRADO_ADMIN">Cerrado</option>
                  <option value="IN_REVIEW">Reapertura</option>
                </Select>
              </FormControl>

              {/* Show address/notes if present */}
              {currentOrder?.notes && (
                <Box bg="gray.50" p={3} borderRadius="md" fontSize="sm">
                  <Text fontWeight="bold">Notas del Estudiante:</Text>
                  <Text whiteSpace="pre-wrap">{currentOrder.notes}</Text>
                </Box>
              )}
            </Box>
          </ModalBody>
          <ModalFooter>
            <Button variant="ghost" mr={3} onClick={onClose}>Cancelar</Button>
            <Button colorScheme="brand" onClick={handleSave}>Guardar Cambios</Button>
          </ModalFooter>
        </ModalContent>
      </Modal>
    </Box>
  );
};

export default AdminDashboard;
