import { useState, useMemo } from 'react';
import {
  Box,
  Heading,
  FormControl,
  FormLabel,
  Select,
  Input,
  Button,
  VStack,
  useToast,
  HStack,
  Text,
  Flex,
  Grid,
  GridItem,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalFooter,
  ModalBody,
  ModalCloseButton,
  useDisclosure,
  Badge,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  TableContainer,
  IconButton,
  Icon,
  ButtonGroup,
  Checkbox,
  CheckboxGroup,
  Stack,
  Divider,
  Avatar
} from '@chakra-ui/react';
import { BiTrash, BiPencil, BiCalendar, BiListUl, BiDownload, BiPlus } from 'react-icons/bi';
import Layout from '../../components/Layout';
import { adminLinks } from './adminLinks';
import { mockProfesores, mockAlumnos, mockMaterias, mockClases } from '../../data/mockData';
import WeeklyGrid from '../../components/Scheduler/WeeklyGrid';

const DAYS_OF_WEEK = ['Lunes', 'Martes', 'Miércoles', 'Jueves', 'Viernes', 'Sábado', 'Domingo'];

const AsignacionClases = () => {
  const [clases, setClases] = useState(mockClases);
  const [selectedProfesor, setSelectedProfesor] = useState('');
  const [selectedAlumno, setSelectedAlumno] = useState('');
  const [selectedMateria, setSelectedMateria] = useState('');
  const [viewMode, setViewMode] = useState('calendar'); // 'calendar' | 'list'

  // Estado para la selección de días y horarios (Batch Creation)
  const [selectedDays, setSelectedDays] = useState([]);
  const [daySchedules, setDaySchedules] = useState({});

  // Estado para la asignación actual (modal - edición)
  const [pendingAssignment, setPendingAssignment] = useState(null);
  const [isEditing, setIsEditing] = useState(false);
  const { isOpen, onOpen, onClose } = useDisclosure();
  const toast = useToast();

  // Filtrar clases para mostrar en el grid
  const displayedClasses = useMemo(() => {
    if (!selectedProfesor) return clases; // Mostrar todas si no hay filtro
    return clases.filter(c => c.id_profesor === parseInt(selectedProfesor));
  }, [clases, selectedProfesor]);

  const handleDayChange = (days) => {
    setSelectedDays(days);
    // Inicializar horarios para días nuevos
    const newSchedules = { ...daySchedules };
    days.forEach(day => {
      if (!newSchedules[day]) {
        newSchedules[day] = { start: '', end: '' };
      }
    });
    setDaySchedules(newSchedules);
  };

  const handleScheduleChange = (day, field, value) => {
    setDaySchedules(prev => ({
      ...prev,
      [day]: {
        ...prev[day],
        [field]: value
      }
    }));
  };

  const handleBatchAssign = () => {
    if (!selectedProfesor || !selectedAlumno || !selectedMateria) {
      toast({
        title: 'Faltan datos',
        description: 'Selecciona Profesor, Alumno y Materia.',
        status: 'warning',
        duration: 3000,
      });
      return;
    }

    if (selectedDays.length === 0) {
      toast({
        title: 'Selecciona días',
        description: 'Debes seleccionar al menos un día de clase.',
        status: 'warning',
        duration: 3000,
      });
      return;
    }

    // Validar horarios completos
    const incomplete = selectedDays.some(day => !daySchedules[day].start || !daySchedules[day].end);
    if (incomplete) {
      toast({
        title: 'Horarios incompletos',
        description: 'Define hora de inicio y fin para todos los días seleccionados.',
        status: 'warning',
        duration: 3000,
      });
      return;
    }

    // Crear clases
    const newClasses = selectedDays.map((day, index) => ({
      id: Date.now() + index, // ID temporal único
      id_profesor: parseInt(selectedProfesor),
      id_alumno: parseInt(selectedAlumno),
      id_materia: parseInt(selectedMateria),
      dia_semana: day,
      hora_inicio: daySchedules[day].start,
      hora_fin: daySchedules[day].end,
      fecha_pago: null // Eliminado
    }));

    setClases([...clases, ...newClasses]);
    toast({
      title: 'Clases asignadas exitosamente',
      description: `Se han creado ${newClasses.length} clases.`,
      status: 'success',
      duration: 3000,
    });

    // Limpiar selección
    setSelectedDays([]);
    setDaySchedules({});
  };

  const handleSlotClick = (day, hour, existingClass = null) => {
    if (existingClass) {
      // Modo Edición
      setPendingAssignment({
        id: existingClass.id,
        dia: existingClass.dia_semana,
        horaInicio: existingClass.hora_inicio,
        horaFin: existingClass.hora_fin,
        fechaPago: existingClass.fecha_pago,
        id_profesor: existingClass.id_profesor,
        id_alumno: existingClass.id_alumno,
        id_materia: existingClass.id_materia
      });
      setIsEditing(true);
      onOpen();
    } else {
      // Modo Creación (Click directo en grid)
      if (!selectedProfesor || !selectedAlumno || !selectedMateria) {
        toast({
          title: 'Faltan datos',
          description: 'Selecciona Profesor, Alumno y Materia antes de elegir un horario.',
          status: 'warning',
          duration: 3000,
        });
        return;
      }

      setPendingAssignment({
        dia: day,
        horaInicio: `${hour}:00`,
        horaFin: `${hour + 1}:00`,
        fechaPago: null
      });
      setIsEditing(false);
      onOpen();
    }
  };

  const handleSaveAssignment = () => {
    if (isEditing) {
      // Actualizar
      const clasesActualizadas = clases.map(c =>
        c.id === pendingAssignment.id ? { ...c, ...pendingAssignment } : c
      );
      setClases(clasesActualizadas);
      toast({
        title: 'Clase actualizada',
        status: 'success',
        duration: 2000,
      });
    } else {
      // Crear (desde modal)
      const nuevaClase = {
        id: Date.now(),
        id_profesor: parseInt(selectedProfesor),
        id_alumno: parseInt(selectedAlumno),
        id_materia: parseInt(selectedMateria),
        dia_semana: pendingAssignment.dia,
        hora_inicio: pendingAssignment.horaInicio,
        hora_fin: pendingAssignment.horaFin,
        fecha_pago: null
      };
      setClases([...clases, nuevaClase]);
      toast({
        title: 'Clase asignada',
        status: 'success',
        duration: 2000,
      });
    }
    onClose();
    setPendingAssignment(null);
  };

  const handleDeleteClass = () => {
    if (!pendingAssignment?.id) return;
    setClases(clases.filter(c => c.id !== pendingAssignment.id));
    onClose();
    setPendingAssignment(null);
    toast({
      title: 'Clase eliminada',
      status: 'success',
      duration: 2000,
    });
  };

  const handleExportCSV = () => {
    const headers = ['ID', 'Profesor', 'Alumno', 'Materia', 'Día', 'Hora Inicio', 'Hora Fin'];

    const rows = clases.map(clase => {
      const prof = mockProfesores.find(p => p.id === clase.id_profesor)?.nombre || 'N/A';
      const alum = mockAlumnos.find(a => a.id === clase.id_alumno)?.nombre || 'N/A';
      const mat = mockMaterias.find(m => m.id === clase.id_materia)?.nombre || 'N/A';

      return [
        clase.id,
        `"${prof}"`,
        `"${alum}"`,
        `"${mat}"`,
        clase.dia_semana,
        clase.hora_inicio,
        clase.hora_fin
      ].join(',');
    });

    const csvContent = [headers.join(','), ...rows].join('\n');
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    if (link.download !== undefined) {
      const url = URL.createObjectURL(blob);
      link.setAttribute('href', url);
      link.setAttribute('download', 'clases_asignadas.csv');
      link.style.visibility = 'hidden';
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
    }
  };

  const getProfesorInfo = () => mockProfesores.find(p => p.id === parseInt(selectedProfesor));

  return (
    <Layout links={adminLinks} userType="Administrador" userName="Admin User">
      <Box h={{ base: 'auto', lg: 'calc(100vh - 100px)' }}>
        <Flex gap={6} h="100%" direction={{ base: 'column', lg: 'row' }}>

          {/* Panel de Control (Sidebar) */}
          <Box
            w={{ base: '100%', lg: '400px' }}
            bg="white"
            p={6}
            borderRadius="xl"
            boxShadow="sm"
            h="fit-content"
            maxH="100%"
            overflowY="auto"
            display={viewMode === 'list' ? { base: 'none', xl: 'block' } : 'block'}
          >
            <VStack spacing={6} align="stretch">
              <Box>
                <Heading size="md" mb={2}>Asignar Clase</Heading>
                <Text fontSize="sm" color="gray.500">
                  Configura los detalles y selecciona los días.
                </Text>
              </Box>

              <FormControl isRequired>
                <FormLabel>Profesor</FormLabel>
                <Select
                  placeholder="Selecciona profesor"
                  value={selectedProfesor}
                  onChange={(e) => setSelectedProfesor(e.target.value)}
                >
                  {mockProfesores.map(p => (
                    <option key={p.id} value={p.id}>{p.nombre}</option>
                  ))}
                </Select>
              </FormControl>

              {selectedProfesor && (
                <Box p={3} bg="blue.50" borderRadius="md">
                  <Text fontSize="xs" fontWeight="bold" color="blue.700">ESPECIALIDAD</Text>
                  <Text fontSize="sm" color="blue.900">{getProfesorInfo()?.especialidad}</Text>
                </Box>
              )}

              <FormControl isRequired>
                <FormLabel>Alumno</FormLabel>
                <Select
                  placeholder="Selecciona alumno"
                  value={selectedAlumno}
                  onChange={(e) => setSelectedAlumno(e.target.value)}
                >
                  {mockAlumnos.map(a => (
                    <option key={a.id} value={a.id}>{a.nombre}</option>
                  ))}
                </Select>
              </FormControl>

              <FormControl isRequired>
                <FormLabel>Materia</FormLabel>
                <Select
                  placeholder="Selecciona materia"
                  value={selectedMateria}
                  onChange={(e) => setSelectedMateria(e.target.value)}
                >
                  {mockMaterias.map(m => (
                    <option key={m.id} value={m.id}>{m.nombre}</option>
                  ))}
                </Select>
              </FormControl>

              <Divider />

              <Box>
                <FormLabel mb={3}>Días de Clase</FormLabel>
                <CheckboxGroup colorScheme="blue" value={selectedDays} onChange={handleDayChange}>
                  <Stack direction="row" flexWrap="wrap" spacing={3}>
                    {DAYS_OF_WEEK.map(day => (
                      <Checkbox key={day} value={day}>{day.substring(0, 3)}</Checkbox>
                    ))}
                  </Stack>
                </CheckboxGroup>
              </Box>

              {selectedDays.length > 0 && (
                <VStack spacing={3} align="stretch" bg="gray.50" p={3} borderRadius="md">
                  <Text fontSize="sm" fontWeight="bold" color="gray.600">Horarios por Día</Text>
                  {selectedDays.map(day => (
                    <Grid key={day} templateColumns="80px 1fr 1fr" gap={2} alignItems="center">
                      <Text fontSize="sm" fontWeight="medium">{day}</Text>
                      <Input
                        type="time"
                        size="sm"
                        value={daySchedules[day]?.start || ''}
                        onChange={(e) => handleScheduleChange(day, 'start', e.target.value)}
                      />
                      <Input
                        type="time"
                        size="sm"
                        value={daySchedules[day]?.end || ''}
                        onChange={(e) => handleScheduleChange(day, 'end', e.target.value)}
                      />
                    </Grid>
                  ))}
                </VStack>
              )}

              <Button
                colorScheme="blue"
                leftIcon={<Icon as={BiPlus} />}
                onClick={handleBatchAssign}
                isDisabled={!selectedProfesor || !selectedAlumno || !selectedMateria || selectedDays.length === 0}
              >
                Asignar Clases
              </Button>

              {viewMode === 'calendar' && (
                <Box pt={4} borderTop="1px solid" borderColor="gray.100">
                  <Text fontSize="xs" fontWeight="bold" color="gray.500" mb={2}>LEYENDA</Text>
                  <HStack spacing={4}>
                    <HStack>
                      <Box w="12px" h="12px" bg="blue.100" borderRadius="sm" border="1px solid" borderColor="blue.200" />
                      <Text fontSize="xs">Clase Asignada</Text>
                    </HStack>
                    <HStack>
                      <Box w="12px" h="12px" bg="gray.50" borderRadius="sm" border="1px solid" borderColor="gray.200" />
                      <Text fontSize="xs">Libre</Text>
                    </HStack>
                  </HStack>
                </Box>
              )}
            </VStack>
          </Box>

          {/* Área Principal */}
          <Box flex={1} overflow="hidden" display="flex" flexDirection="column">
            <Box mb={4} display="flex" justifyContent="space-between" alignItems="center">
              <HStack>
                <Heading size="md">
                  {viewMode === 'calendar' ? 'Disponibilidad Semanal' : 'Listado de Clases'}
                </Heading>
                {viewMode === 'calendar' && selectedProfesor && (
                  <Badge colorScheme="green" px={3} py={1} borderRadius="full">
                    {getProfesorInfo()?.nombre}
                  </Badge>
                )}
                {viewMode === 'calendar' && !selectedProfesor && (
                  <Badge colorScheme="gray" px={3} py={1} borderRadius="full">
                    Vista General
                  </Badge>
                )}
              </HStack>

              <HStack spacing={2}>
                {viewMode === 'list' && (
                  <Button
                    leftIcon={<Icon as={BiDownload} />}
                    size="sm"
                    colorScheme="green"
                    variant="solid"
                    onClick={handleExportCSV}
                  >
                    Exportar CSV
                  </Button>
                )}
                <ButtonGroup size="sm" isAttached variant="outline">
                  <Button
                    leftIcon={<Icon as={BiCalendar} />}
                    colorScheme={viewMode === 'calendar' ? 'blue' : 'gray'}
                    variant={viewMode === 'calendar' ? 'solid' : 'outline'}
                    onClick={() => setViewMode('calendar')}
                  >
                    Calendario
                  </Button>
                  <Button
                    leftIcon={<Icon as={BiListUl} />}
                    colorScheme={viewMode === 'list' ? 'blue' : 'gray'}
                    variant={viewMode === 'list' ? 'solid' : 'outline'}
                    onClick={() => setViewMode('list')}
                  >
                    Lista
                  </Button>
                </ButtonGroup>
              </HStack>
            </Box>

            <Box flex={1} overflow="auto" borderRadius="xl" border="1px solid" borderColor="gray.200" bg="white">
              {viewMode === 'calendar' ? (
                <WeeklyGrid
                  existingClasses={displayedClasses}
                  onSlotClick={handleSlotClick}
                  selectedDay={pendingAssignment?.dia}
                  selectedHour={pendingAssignment ? parseInt(pendingAssignment.horaInicio) : null}
                />
              ) : (
                <TableContainer>
                  <Table variant="simple">
                    <Thead bg="gray.50">
                      <Tr>
                        <Th>Profesor</Th>
                        <Th>Alumno</Th>
                        <Th>Materia</Th>
                        <Th>Día</Th>
                        <Th>Horario</Th>
                        <Th>Acciones</Th>
                      </Tr>
                    </Thead>
                    <Tbody>
                      {clases.map(clase => {
                        const prof = mockProfesores.find(p => p.id === clase.id_profesor);
                        const alum = mockAlumnos.find(a => a.id === clase.id_alumno);
                        const mat = mockMaterias.find(m => m.id === clase.id_materia);
                        return (
                          <Tr key={clase.id} _hover={{ bg: 'gray.50' }}>
                            <Td fontWeight="medium">{prof?.nombre}</Td>
                            <Td>{alum?.nombre}</Td>
                            <Td>{mat?.nombre}</Td>
                            <Td>
                              <Badge colorScheme="blue" variant="subtle">{clase.dia_semana}</Badge>
                            </Td>
                            <Td>{clase.hora_inicio} - {clase.hora_fin}</Td>
                            <Td>
                              <HStack>
                                <IconButton
                                  icon={<Icon as={BiPencil} />}
                                  size="sm"
                                  variant="ghost"
                                  aria-label="Editar"
                                  onClick={() => handleSlotClick(null, null, clase)}
                                />
                                <IconButton
                                  icon={<Icon as={BiTrash} />}
                                  size="sm"
                                  variant="ghost"
                                  colorScheme="red"
                                  aria-label="Eliminar"
                                  onClick={() => {
                                    setPendingAssignment(clase);
                                    setIsEditing(true);
                                    handleSlotClick(null, null, clase);
                                  }}
                                />
                              </HStack>
                            </Td>
                          </Tr>
                        );
                      })}
                    </Tbody>
                  </Table>
                </TableContainer>
              )}
            </Box>
          </Box>
        </Flex>

        {/* Modal de Confirmación / Edición */}
        <Modal isOpen={isOpen} onClose={onClose} size="md">
          <ModalOverlay backdropFilter="blur(5px)" />
          <ModalContent>
            <ModalHeader>{isEditing ? 'Editar Clase' : 'Confirmar Asignación'}</ModalHeader>
            <ModalCloseButton />
            <ModalBody>
              <VStack spacing={4} align="stretch">
                <Box bg="gray.50" p={3} borderRadius="md">
                  <Grid templateColumns="1fr 1fr" gap={2}>
                    <GridItem>
                      <Text fontSize="xs" color="gray.500">DÍA</Text>
                      <Text fontWeight="bold">{pendingAssignment?.dia}</Text>
                    </GridItem>
                    <GridItem>
                      <Text fontSize="xs" color="gray.500">HORARIO</Text>
                      <Text fontWeight="bold">
                        {pendingAssignment?.horaInicio} - {pendingAssignment?.horaFin}
                      </Text>
                    </GridItem>
                    {isEditing && (
                      <>
                        <GridItem colSpan={2} pt={2} borderTop="1px dashed" borderColor="gray.300">
                          <Text fontSize="xs" color="gray.500">ALUMNO</Text>
                          <Text fontWeight="bold">
                            {mockAlumnos.find(a => a.id === pendingAssignment?.id_alumno)?.nombre}
                          </Text>
                        </GridItem>
                      </>
                    )}
                  </Grid>
                </Box>

                <FormControl>
                  <FormLabel>Duración (Horas)</FormLabel>
                  <Select
                    value={pendingAssignment ? parseInt(pendingAssignment.horaFin) - parseInt(pendingAssignment.horaInicio) : 1}
                    onChange={(e) => {
                      const duration = parseInt(e.target.value);
                      const start = parseInt(pendingAssignment.horaInicio);
                      setPendingAssignment({
                        ...pendingAssignment,
                        horaFin: `${start + duration}:00`
                      });
                    }}
                  >
                    <option value={1}>1 Hora</option>
                    <option value={2}>2 Horas</option>
                    <option value={3}>3 Horas</option>
                  </Select>
                </FormControl>
              </VStack>
            </ModalBody>

            <ModalFooter justifyContent="space-between">
              {isEditing ? (
                <Button colorScheme="red" variant="ghost" onClick={handleDeleteClass}>
                  Eliminar Clase
                </Button>
              ) : (
                <Box /> // Spacer
              )}
              <HStack>
                <Button variant="ghost" onClick={onClose}>Cancelar</Button>
                <Button colorScheme="blue" onClick={handleSaveAssignment}>
                  {isEditing ? 'Guardar Cambios' : 'Confirmar Clase'}
                </Button>
              </HStack>
            </ModalFooter>
          </ModalContent>
        </Modal>
      </Box>
    </Layout>
  );
};

export default AsignacionClases;
