import React, { useState, useMemo } from 'react';
import {
  Box,
  Heading,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  TableContainer,
  Badge,
  Flex,
  Text,
  SimpleGrid,
  Stat,
  StatLabel,
  StatNumber,
  StatHelpText,
  Select,
  Input,
  Card,
  CardBody,
  Stack,
  Icon,
  useColorModeValue,
  Button,
  HStack
} from '@chakra-ui/react';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip as RechartsTooltip,
  Legend,
  ResponsiveContainer,
  PieChart,
  Pie,
  Cell
} from 'recharts';
import { FiDollarSign, FiUsers, FiFilter } from 'react-icons/fi';
import { BiDownload } from 'react-icons/bi';
import Layout from '../../components/Layout';
import { adminLinks } from './adminLinks';
import {
  mockPagos,
  mockAlumnos,
  mockPadres,
  mockClases,
  mockProfesores,
  mockMaterias
} from '../../data/mockData';

const CalendarioPagos = () => {
  // --- State for Filters ---
  const [filterAlumno, setFilterAlumno] = useState('');
  const [filterProfesor, setFilterProfesor] = useState('');
  const [filterMateria, setFilterMateria] = useState('');
  const [filterEstatus, setFilterEstatus] = useState('');
  const [filterPeriodo, setFilterPeriodo] = useState(''); // Format: YYYY-MM or empty

  // --- Data Enrichment ---
  const enrichedPagos = useMemo(() => {
    return mockPagos.map(pago => {
      const alumno = mockAlumnos.find(a => a.id === pago.id_alumno);
      const padre = mockPadres.find(p => p.id === pago.id_padre);

      // Find all classes for this student to associate professors and subjects
      const clasesAlumno = mockClases.filter(c => c.id_alumno === pago.id_alumno);

      const profesores = clasesAlumno.map(c => {
        return mockProfesores.find(p => p.id === c.id_profesor);
      }).filter(Boolean);

      const materias = clasesAlumno.map(c => {
        return mockMaterias.find(m => m.id === c.id_materia);
      }).filter(Boolean);

      // Unique IDs for filtering
      const profesorIds = [...new Set(clasesAlumno.map(c => c.id_profesor))];
      const materiaIds = [...new Set(clasesAlumno.map(c => c.id_materia))];

      return {
        ...pago,
        alumno,
        padre,
        profesores,
        materias,
        profesorIds,
        materiaIds,
        fechaObj: new Date(pago.fecha_pago)
      };
    });
  }, []);

  // --- Filtering Logic ---
  const filteredPagos = useMemo(() => {
    return enrichedPagos.filter(pago => {
      // Filter by Alumno Name
      if (filterAlumno && !pago.alumno?.nombre.toLowerCase().includes(filterAlumno.toLowerCase())) {
        return false;
      }
      // Filter by Profesor
      if (filterProfesor && !pago.profesorIds.includes(parseInt(filterProfesor))) {
        return false;
      }
      // Filter by Materia
      if (filterMateria && !pago.materiaIds.includes(parseInt(filterMateria))) {
        return false;
      }
      // Filter by Estatus
      if (filterEstatus && pago.estatus !== filterEstatus) {
        return false;
      }
      // Filter by Period (Month)
      if (filterPeriodo) {
        // Assuming format YYYY-MM-DD, we take first 7 chars
        const pagoDate = pago.fecha_pago.substring(0, 7);
        if (pagoDate !== filterPeriodo) {
          return false;
        }
      }
      return true;
    });
  }, [enrichedPagos, filterAlumno, filterProfesor, filterMateria, filterEstatus, filterPeriodo]);

  // --- Statistics Calculation ---
  const stats = useMemo(() => {
    const totalIngresos = filteredPagos.reduce((acc, curr) => acc + curr.monto, 0);
    const pagosValidados = filteredPagos.filter(p => p.estatus === 'Validado').length;
    const pagosPendientes = filteredPagos.filter(p => p.estatus === 'Pendiente').length;
    const uniqueAlumnos = new Set(filteredPagos.map(p => p.id_alumno)).size;

    return {
      totalIngresos,
      pagosValidados,
      pagosPendientes,
      uniqueAlumnos
    };
  }, [filteredPagos]);

  // --- Chart Data Preparation ---
  const chartDataByProfesor = useMemo(() => {
    const data = {};
    filteredPagos.forEach(pago => {
      const uniqueProfs = new Set(pago.profesores.map(p => p.nombre));
      uniqueProfs.forEach(profName => {
        if (!data[profName]) data[profName] = { amount: 0, students: new Set() };
        data[profName].amount += pago.monto;
        data[profName].students.add(pago.id_alumno);
      });
    });
    return Object.keys(data).map(key => ({
      name: key,
      value: data[key].amount,
      students: data[key].students.size
    }));
  }, [filteredPagos]);

  const chartDataByMateria = useMemo(() => {
    const data = {};
    filteredPagos.forEach(pago => {
      const uniqueMaterias = new Set(pago.materias.map(m => m.nombre));
      uniqueMaterias.forEach(materiaName => {
        if (!data[materiaName]) data[materiaName] = { amount: 0, students: new Set() };
        data[materiaName].amount += pago.monto;
        data[materiaName].students.add(pago.id_alumno);
      });
    });
    return Object.keys(data).map(key => ({
      name: key,
      value: data[key].amount,
      students: data[key].students.size
    }));
  }, [filteredPagos]);

  const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042', '#8884d8', '#82ca9d'];

  const getStatusColor = (estatus) => {
    switch (estatus) {
      case 'Validado': return 'green';
      case 'Pendiente': return 'orange';
      case 'Rechazado': return 'red';
      default: return 'gray';
    }
  };

  // --- Export Function ---
  const handleExport = () => {
    const headers = ['Fecha', 'Alumno', 'Materias', 'Profesores', 'Monto', 'Estatus'];
    const csvContent = [
      headers.join(','),
      ...filteredPagos.map(pago => {
        const materiasStr = pago.materias.map(m => m.nombre).join('; ');
        const profesoresStr = pago.profesores.map(p => p.nombre).join('; ');
        return [
          pago.fecha_pago,
          `"${pago.alumno?.nombre || ''}"`,
          `"${materiasStr}"`,
          `"${profesoresStr}"`,
          pago.monto,
          pago.estatus
        ].join(',');
      })
    ].join('\n');

    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    if (link.download !== undefined) {
      const url = URL.createObjectURL(blob);
      link.setAttribute('href', url);
      link.setAttribute('download', 'reporte_pagos.csv');
      link.style.visibility = 'hidden';
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
    }
  };

  // --- UI Components ---
  const cardBg = useColorModeValue('white', 'gray.700');
  const borderColor = useColorModeValue('gray.200', 'gray.600');

  return (
    <Layout links={adminLinks} userType="Administrador" userName="Admin User">
      <Box>
        <Heading mb={6} size="lg">Tablero Financiero y Pagos</Heading>

        {/* Filters Section */}
        <Card mb={6} bg={cardBg} borderRadius="xl" boxShadow="sm" border="1px" borderColor={borderColor}>
          <CardBody>
            <Stack direction={{ base: 'column', md: 'row' }} spacing={4} align="end">
              <Box flex={1}>
                <Text fontSize="sm" fontWeight="bold" mb={1}>Buscar Alumno</Text>
                <Input
                  placeholder="Nombre del alumno..."
                  value={filterAlumno}
                  onChange={(e) => setFilterAlumno(e.target.value)}
                />
              </Box>
              <Box flex={1}>
                <Text fontSize="sm" fontWeight="bold" mb={1}>Profesor</Text>
                <Select placeholder="Todos los profesores" value={filterProfesor} onChange={(e) => setFilterProfesor(e.target.value)}>
                  {mockProfesores.map(p => (
                    <option key={p.id} value={p.id}>{p.nombre}</option>
                  ))}
                </Select>
              </Box>
              <Box flex={1}>
                <Text fontSize="sm" fontWeight="bold" mb={1}>Clase / Materia</Text>
                <Select placeholder="Todas las materias" value={filterMateria} onChange={(e) => setFilterMateria(e.target.value)}>
                  {mockMaterias.map(m => (
                    <option key={m.id} value={m.id}>{m.nombre}</option>
                  ))}
                </Select>
              </Box>
              <Box flex={1}>
                <Text fontSize="sm" fontWeight="bold" mb={1}>Estatus</Text>
                <Select placeholder="Todos" value={filterEstatus} onChange={(e) => setFilterEstatus(e.target.value)}>
                  <option value="Validado">Validado</option>
                  <option value="Pendiente">Pendiente</option>
                  <option value="Rechazado">Rechazado</option>
                </Select>
              </Box>
              <Box flex={1}>
                <Text fontSize="sm" fontWeight="bold" mb={1}>Periodo (Mes)</Text>
                <Input
                  type="month"
                  value={filterPeriodo}
                  onChange={(e) => setFilterPeriodo(e.target.value)}
                />
              </Box>
              <Button
                leftIcon={<FiFilter />}
                colorScheme="blue"
                variant="outline"
                onClick={() => {
                  setFilterAlumno('');
                  setFilterProfesor('');
                  setFilterMateria('');
                  setFilterEstatus('');
                  setFilterPeriodo('');
                }}
              >
                Limpiar
              </Button>
            </Stack>
          </CardBody>
        </Card>

        {/* Stats Cards */}
        <SimpleGrid columns={{ base: 1, md: 2, lg: 4 }} spacing={6} mb={8}>
          <Stat
            px={4} py={3}
            bg="linear-gradient(135deg, #667eea 0%, #764ba2 100%)"
            color="white"
            borderRadius="xl"
            boxShadow="lg"
            position="relative"
          >
            <StatLabel fontWeight="medium">Ingresos Totales</StatLabel>
            <StatNumber fontSize="3xl">${stats.totalIngresos.toLocaleString()}</StatNumber>
            <StatHelpText color="whiteAlpha.800">Filtrado actual</StatHelpText>
            <Icon as={FiDollarSign} position="absolute" top={4} right={4} w={6} h={6} color="whiteAlpha.600" />
          </Stat>

          <Stat
            px={4} py={3}
            bg={cardBg}
            border="1px"
            borderColor={borderColor}
            borderRadius="xl"
            boxShadow="sm"
          >
            <StatLabel color="gray.500">Pagos Validados</StatLabel>
            <StatNumber fontSize="3xl" color="green.500">{stats.pagosValidados}</StatNumber>
            <StatHelpText>Transacciones completadas</StatHelpText>
          </Stat>

          <Stat
            px={4} py={3}
            bg={cardBg}
            border="1px"
            borderColor={borderColor}
            borderRadius="xl"
            boxShadow="sm"
          >
            <StatLabel color="gray.500">Pagos Pendientes</StatLabel>
            <StatNumber fontSize="3xl" color="orange.500">{stats.pagosPendientes}</StatNumber>
            <StatHelpText>Requieren atención</StatHelpText>
          </Stat>

          <Stat
            px={4} py={3}
            bg={cardBg}
            border="1px"
            borderColor={borderColor}
            borderRadius="xl"
            boxShadow="sm"
            position="relative"
          >
            <StatLabel color="gray.500">Alumnos Activos</StatLabel>
            <StatNumber fontSize="3xl" color="blue.500">{stats.uniqueAlumnos}</StatNumber>
            <StatHelpText>En selección actual</StatHelpText>
            <Icon as={FiUsers} position="absolute" top={4} right={4} w={5} h={5} color="gray.300" />
          </Stat>
        </SimpleGrid>

        {/* Charts Section */}
        <SimpleGrid columns={{ base: 1, lg: 2 }} spacing={8} mb={8}>
          <Card borderRadius="xl" boxShadow="md" border="1px" borderColor={borderColor}>
            <CardBody>
              <Heading size="sm" mb={4} color="gray.600">Ingresos por Profesor (Estimado)</Heading>
              <Box h="300px">
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart data={chartDataByProfesor}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" fontSize={12} tick={{ fill: '#718096' }} />
                    <YAxis fontSize={12} tick={{ fill: '#718096' }} />
                    <RechartsTooltip
                      formatter={(value, name, props) => [`$${value.toLocaleString()} (${props.payload.students} alumnos)`, 'Ingresos']}
                      contentStyle={{ borderRadius: '8px', border: 'none', boxShadow: '0 4px 12px rgba(0,0,0,0.1)' }}
                    />
                    <Bar dataKey="value" fill="#667eea" radius={[4, 4, 0, 0]} />
                  </BarChart>
                </ResponsiveContainer>
              </Box>
            </CardBody>
          </Card>

          <Card borderRadius="xl" boxShadow="md" border="1px" borderColor={borderColor}>
            <CardBody>
              <Heading size="sm" mb={4} color="gray.600">Ingresos por Materia</Heading>
              <Box h="300px">
                <ResponsiveContainer width="100%" height="100%">
                  <PieChart>
                    <Pie
                      data={chartDataByMateria}
                      cx="50%"
                      cy="50%"
                      innerRadius={60}
                      outerRadius={100}
                      fill="#8884d8"
                      paddingAngle={5}
                      dataKey="value"
                    >
                      {chartDataByMateria.map((entry, index) => (
                        <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                      ))}
                    </Pie>
                    <RechartsTooltip formatter={(value, name, props) => [`$${value.toLocaleString()} (${props.payload.students} alumnos)`, name]} />
                    <Legend verticalAlign="bottom" height={36} />
                  </PieChart>
                </ResponsiveContainer>
              </Box>
            </CardBody>
          </Card>
        </SimpleGrid>

        {/* Data Table */}
        <Box
          bg="rgba(255, 255, 255, 0.8)"
          backdropFilter="blur(20px)"
          borderRadius="xl"
          boxShadow="lg"
          border="1px solid"
          borderColor="whiteAlpha.300"
          overflow="hidden"
        >
          <Flex justify="space-between" align="center" p={5} borderBottom="1px" borderColor="gray.100">
            <Heading as="h3" size="md">Detalle de Pagos</Heading>
            <HStack>
              <Button
                leftIcon={<Icon as={BiDownload} />}
                size="sm"
                colorScheme="green"
                variant="solid"
                onClick={handleExport}
              >
                Exportar Excel
              </Button>
              <Badge colorScheme="blue" borderRadius="full" px={3}>
                {filteredPagos.length} registros
              </Badge>
            </HStack>
          </Flex>

          <TableContainer>
            <Table variant="simple">
              <Thead bg="gray.50">
                <Tr>
                  <Th>Fecha</Th>
                  <Th>Alumno</Th>
                  <Th>Clases / Materias</Th>
                  <Th>Profesores</Th>
                  <Th isNumeric>Monto</Th>
                  <Th>Estatus</Th>
                </Tr>
              </Thead>
              <Tbody>
                {filteredPagos.length > 0 ? (
                  filteredPagos.map((pago) => (
                    <Tr key={pago.id} _hover={{ bg: "gray.50" }}>
                      <Td fontSize="sm">{pago.fecha_pago}</Td>
                      <Td fontWeight="medium">{pago.alumno?.nombre}</Td>
                      <Td fontSize="sm">
                        <Stack spacing={1}>
                          {pago.materias.map((m, idx) => (
                            <Text key={idx} fontSize="xs" color="gray.600">• {m.nombre}</Text>
                          ))}
                          {pago.materias.length === 0 && <Text fontSize="xs" color="gray.400">Sin asignar</Text>}
                        </Stack>
                      </Td>
                      <Td fontSize="sm">
                        <Stack spacing={1}>
                          {pago.profesores.map((p, idx) => (
                            <Text key={idx} fontSize="xs" color="gray.600">{p.nombre}</Text>
                          ))}
                        </Stack>
                      </Td>
                      <Td isNumeric fontWeight="bold">${pago.monto.toLocaleString()}</Td>
                      <Td>
                        <Badge colorScheme={getStatusColor(pago.estatus)} borderRadius="md" px={2}>
                          {pago.estatus}
                        </Badge>
                      </Td>
                    </Tr>
                  ))
                ) : (
                  <Tr>
                    <Td colSpan={6} textAlign="center" py={10} color="gray.500">
                      No se encontraron pagos con los filtros seleccionados.
                    </Td>
                  </Tr>
                )}
              </Tbody>
            </Table>
          </TableContainer>
        </Box>
      </Box>
    </Layout>
  );
};

export default CalendarioPagos;
