import { useState, useEffect, useRef } from 'react';
import {
    Box,
    Heading,
    FormControl,
    FormLabel,
    Input,
    Button,
    VStack,
    useToast,
    Text,
    Card,
    CardBody,
    Divider,
    Textarea,
    Image,
    HStack,
    SimpleGrid,
    Icon,
    InputGroup,
    InputLeftElement,
    CardHeader,
    Container,
    Badge,
    Flex,
    useColorModeValue
} from '@chakra-ui/react';
import {
    MdBusiness,
    MdOutlineDescription,
    MdOutlineCloudUpload,
    MdSave,
    MdSchool,
    MdFormatQuote
} from 'react-icons/md';
import { getCompanyById, updateCompany, uploadCompanyLogo } from '../../services/api';
import { processImage } from '../../utils/imageUtils';

const ConfiguracionEmpresa = () => {
    const [company, setCompany] = useState(null);
    const [selectedFile, setSelectedFile] = useState(null);
    const [previewUrl, setPreviewUrl] = useState(null);
    const [isHovering, setIsHovering] = useState(false);
    const fileInputRef = useRef(null);
    const toast = useToast();

    // Theme colors
    const cardBg = useColorModeValue('white', 'gray.800');
    const borderColor = useColorModeValue('gray.200', 'gray.700');
    const brandColor = 'brand.500';

    useEffect(() => {
        const fetchCompany = async () => {
            try {
                const user = JSON.parse(sessionStorage.getItem('user'));
                const institutionId = user ? (user.institution_id || user.company_id) : null;

                if (institutionId) {
                    const data = await getCompanyById(institutionId);
                    setCompany(data);
                }
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar configuración',
                    status: 'error',
                    duration: 3000,
                });
            }
        };
        fetchCompany();
    }, []);

    const handleLogoChange = async (e) => {
        const file = e.target.files[0];
        if (file) {
            try {
                // Ajustar el logo automáticamente
                const processedFile = await processImage(file, {
                    minWidth: 100,
                    minHeight: 100,
                    maxWidth: 1200,
                    maxHeight: 1200
                });
                setSelectedFile(processedFile);
                setPreviewUrl(URL.createObjectURL(processedFile));
                toast({
                    title: 'Logo procesado',
                    description: 'Se ha ajustado el tamaño automáticamente.',
                    status: 'info',
                    duration: 2000,
                });
            } catch (err) {
                toast({
                    title: 'Error al procesar logo',
                    description: err.message,
                    status: 'error',
                    duration: 3000,
                });
            }
        }
    };

    const handleSave = async () => {
        try {
            if (company && (company.id || company.id_institution)) {
                const targetId = company.id || company.id_institution;
                await updateCompany(targetId, company);

                if (selectedFile) {
                    const response = await uploadCompanyLogo(targetId, selectedFile);
                    setCompany(prev => ({ ...prev, logo_url: response.logo_url }));
                    setPreviewUrl(null);
                    setSelectedFile(null);
                }

                toast({
                    title: 'Configuración guardada',
                    description: 'Los cambios se han aplicado correctamente.',
                    status: 'success',
                    duration: 3000,
                    isClosable: true,
                });
            }
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    if (!company) return <Text>Cargando configuración...</Text>;

    return (
        <Container maxW="container.xl" py={8}>
            <VStack spacing={8} align="stretch">
                <Box>
                    <Heading size="lg" mb={2} bgGradient="linear(to-r, brand.600, brand.400)" bgClip="text">
                        Configuración de Institución
                    </Heading>
                    <Text color="gray.500">
                        Personaliza la identidad visual y los mensajes de tu plataforma educativa.
                    </Text>
                </Box>

                <SimpleGrid columns={{ base: 1, lg: 3 }} spacing={8}>
                    {/* Columna Izquierda: Identidad Visual */}
                    <Card bg={cardBg} borderColor={borderColor} borderWidth="1px" shadow="sm" h="fit-content">
                        <CardHeader borderBottomWidth="1px" borderColor={borderColor} pb={4}>
                            <HStack>
                                <Icon as={MdOutlineCloudUpload} color="brand.500" boxSize={6} />
                                <Heading size="md">Identidad Visual</Heading>
                            </HStack>
                        </CardHeader>
                        <CardBody>
                            <VStack spacing={6}>
                                <Box
                                    position="relative"
                                    onMouseEnter={() => setIsHovering(true)}
                                    onMouseLeave={() => setIsHovering(false)}
                                    cursor="pointer"
                                    onClick={() => fileInputRef.current?.click()}
                                >
                                    <Image
                                        src={previewUrl || company.logo_url || '/logo.jpeg'}
                                        onError={(e) => { e.target.onerror = null; e.target.src = '/logo.jpeg'; }}
                                        alt="Logo Institucional"
                                        boxSize="200px"
                                        objectFit="contain"
                                        borderRadius="lg"
                                        border="2px dashed"
                                        borderColor={isHovering ? brandColor : "gray.300"}
                                        transition="all 0.2s"
                                        bg="gray.50"
                                        p={2}
                                    />
                                    {isHovering && (
                                        <Flex
                                            position="absolute"
                                            top={0}
                                            left={0}
                                            right={0}
                                            bottom={0}
                                            bg="blackAlpha.600"
                                            borderRadius="lg"
                                            align="center"
                                            justify="center"
                                            color="white"
                                            flexDirection="column"
                                        >
                                            <Icon as={MdOutlineCloudUpload} boxSize={8} mb={2} />
                                            <Text fontSize="sm" fontWeight="bold">Cambiar Logo</Text>
                                        </Flex>
                                    )}
                                </Box>
                                <input
                                    type="file"
                                    ref={fileInputRef}
                                    onChange={handleLogoChange}
                                    accept="image/*"
                                    style={{ display: 'none' }}
                                />
                                <VStack spacing={1}>
                                    <Text fontSize="sm" fontWeight="medium">Logo de la Institución</Text>
                                    <Text fontSize="xs" color="gray.500" textAlign="center">
                                        Formato recomendado: PNG o JPG<br />Dimensiones: 200x200px
                                    </Text>
                                </VStack>
                            </VStack>
                        </CardBody>
                    </Card>

                    {/* Columna Derecha: Información y Configuración */}
                    <Box gridColumn={{ lg: "span 2" }}>
                        <Card bg={cardBg} borderColor={borderColor} borderWidth="1px" shadow="sm">
                            <CardHeader borderBottomWidth="1px" borderColor={borderColor} pb={4}>
                                <HStack justify="space-between">
                                    <HStack>
                                        <Icon as={MdBusiness} color="brand.500" boxSize={6} />
                                        <Heading size="md">Detalles de la Organización</Heading>
                                    </HStack>
                                    <Badge colorScheme="brand" variant="subtle">ACTIVO</Badge>
                                </HStack>
                            </CardHeader>
                            <CardBody>
                                <VStack spacing={6}>
                                    <SimpleGrid columns={{ base: 1, md: 2 }} spacing={6} w="full">
                                        <FormControl>
                                            <FormLabel>Nombre de la Institución</FormLabel>
                                            <InputGroup>
                                                <InputLeftElement pointerEvents='none'>
                                                    <Icon as={MdSchool} color='gray.400' />
                                                </InputLeftElement>
                                                <Input
                                                    value={company.name}
                                                    onChange={(e) => setCompany({ ...company, name: e.target.value })}
                                                    placeholder="Nombre oficial"
                                                />
                                            </InputGroup>
                                        </FormControl>

                                        <FormControl>
                                            <FormLabel>Eslogan / Tagline</FormLabel>
                                            <InputGroup>
                                                <InputLeftElement pointerEvents='none'>
                                                    <Icon as={MdFormatQuote} color='gray.400' />
                                                </InputLeftElement>
                                                <Input
                                                    value={company.tagline || company.slogan || ''}
                                                    onChange={(e) => setCompany({ ...company, tagline: e.target.value })}
                                                    placeholder="Ej. Excelencia educativa"
                                                />
                                            </InputGroup>
                                        </FormControl>
                                    </SimpleGrid>

                                    <Divider />

                                    <Box w="full">
                                        <Heading size="sm" mb={4} display="flex" alignItems="center" gap={2}>
                                            <Icon as={MdOutlineDescription} color="brand.500" />
                                            Configuración de Exámenes
                                        </Heading>
                                        <FormControl>
                                            <FormLabel>Pie de página del Examen (Footer)</FormLabel>
                                            <Textarea
                                                value={company.exam_footer || company.service_completion_message || ''}
                                                onChange={(e) => setCompany({ ...company, exam_footer: e.target.value })}
                                                rows={4}
                                                placeholder="Este texto aparecerá al final de todos los exámenes generados..."
                                                borderColor="gray.300"
                                                _hover={{ borderColor: 'brand.400' }}
                                                _focus={{ borderColor: 'brand.500', boxShadow: 'none' }}
                                            />
                                            <Text fontSize="xs" color="gray.500" mt={2}>
                                                Utiliza este espacio para añadir avisos legales, agradecimientos o información de contacto.
                                            </Text>
                                        </FormControl>
                                    </Box>

                                    <Flex w="full" justify="flex-end" pt={4}>
                                        <Button
                                            leftIcon={<MdSave />}
                                            colorScheme="brand"
                                            size="lg"
                                            onClick={handleSave}
                                            loadingText="Guardando..."
                                        >
                                            Guardar Cambios
                                        </Button>
                                    </Flex>
                                </VStack>
                            </CardBody>
                        </Card>
                    </Box>
                </SimpleGrid>
            </VStack>
        </Container>
    );
};

export default ConfiguracionEmpresa;
