import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import {
  Box,
  Heading,
  Button,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  TableContainer,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  useDisclosure,
  FormControl,
  FormLabel,
  Input,
  Avatar,
  Select,
  useToast,
  Flex,
  HStack,
  IconButton,
  Icon,
  Text,
  Grid,
  GridItem,
  VStack,
  Tooltip,
  AlertDialog,
  AlertDialogBody,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogContent,
  AlertDialogOverlay,
} from '@chakra-ui/react';
import { BiPencil, BiTrash, BiShow, BiPlus, BiUserMinus } from 'react-icons/bi';
import { FilePond, registerPlugin } from 'react-filepond';
import 'filepond/dist/filepond.min.css';
import FilePondPluginImagePreview from 'filepond-plugin-image-preview';
import 'filepond-plugin-image-preview/dist/filepond-plugin-image-preview.css';
import Layout from '../../components/Layout';
import { adminLinks } from './adminLinks';
// import { mockAlumnos, mockPadres } from '../../data/mockData';
import { estadosMexico } from '../../data/states';

registerPlugin(FilePondPluginImagePreview);

import { getClients, createClient, updateClient, deleteUser, uploadEmployeePhoto, BASE_URL } from '../../services/api';

const GestionAlumnos = () => {
  const [alumnos, setAlumnos] = useState([]);
  const [alumnoEditando, setAlumnoEditando] = useState(null);
  const navigate = useNavigate();
  const { isOpen, onOpen, onClose } = useDisclosure();
  const { isOpen: isEditOpen, onOpen: onEditOpen, onClose: onEditClose } = useDisclosure();
  const { isOpen: isBajaOpen, onOpen: onBajaOpen, onClose: onBajaClose } = useDisclosure();
  const [alumnoParaBaja, setAlumnoParaBaja] = useState(null);
  const cancelRef = React.useRef();
  const toast = useToast();
  const [isLoading, setIsLoading] = useState(false);

  // Form States
  const [formData, setFormData] = useState({
    first_name: '',
    last_name: '',
    email: '',
    fecha_nacimiento: '',
    id_padre: '',
    telefono: '',
    curp: '',
    direccion: {
      estado: '',
      municipio: '',
      cp: '',
      calle: '',
      numero: ''
    }
  });
  const [files, setFiles] = useState([]);
  const [municipiosDisponibles, setMunicipiosDisponibles] = useState([]);

  useEffect(() => {
    fetchAlumnos();
  }, []);

  const fetchAlumnos = async () => {
    try {
      setIsLoading(true);
      const data = await getClients(); // Assuming getClients returns ONLY students/clients
      const mapped = data.map(a => ({
        ...a,
        nombre: (a.first_name && a.last_name) ? `${a.first_name} ${a.last_name}` : (a.name || ''),
        fecha_nacimiento: a.birthdate || a.fecha_nacimiento, // Adapt based on backend field
        id_padre: a.parent_id, // Adapt based on backend field
        telefono: a.phone,
        foto_url: a.photo_url ? (a.photo_url.startsWith('http') ? a.photo_url : `${BASE_URL}${a.photo_url}`) : null
      }));
      setAlumnos(mapped);
    } catch (error) {
      console.error("Error fetching students:", error);
      toast({ title: 'Error al cargar alumnos', status: 'error' });
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    if (formData.direccion?.estado) {
      const estadoObj = estadosMexico.find(e => e.nombre === formData.direccion.estado);
      setMunicipiosDisponibles(estadoObj ? estadoObj.municipios : []);
    } else {
      setMunicipiosDisponibles([]);
    }
  }, [formData.direccion?.estado]);

  // Calcular edad basado en fecha de nacimiento
  const calcularEdad = (fechaNacimiento) => {
    if (!fechaNacimiento) return '';
    const hoy = new Date();
    const nacimiento = new Date(fechaNacimiento);
    let edad = hoy.getFullYear() - nacimiento.getFullYear();
    const mes = hoy.getMonth() - nacimiento.getMonth();
    if (mes < 0 || (mes === 0 && hoy.getDate() < nacimiento.getDate())) {
      edad--;
    }
    return edad;
  };

  const handleCrear = () => {
    setAlumnoEditando(null);
    setFormData({
      first_name: '',
      last_name: '',
      email: '',
      fecha_nacimiento: '',
      id_padre: '',
      telefono: '',
      curp: '',
      direccion: {
        estado: '',
        municipio: '',
        cp: '',
        calle: '',
        numero: ''
      }
    });
    setFiles([]);
    onOpen();
  };

  const handleEditar = (alumno) => {
    setAlumnoEditando(alumno);
    setFormData({
      first_name: alumno.first_name || alumno.nombre?.split(' ')[0] || '',
      last_name: alumno.last_name || alumno.nombre?.split(' ').slice(1).join(' ') || '',
      email: alumno.email,
      fecha_nacimiento: alumno.fecha_nacimiento ? new Date(alumno.fecha_nacimiento).toISOString().split('T')[0] : '',
      id_padre: alumno.id_padre || '',
      telefono: alumno.telefono || '',
      curp: alumno.curp || '',
      direccion: alumno.direccion || {
        estado: '',
        municipio: '',
        cp: '',
        calle: '',
        numero: ''
      }
    });
    setFiles(alumno.foto_url ? [{
      source: alumno.foto_url,
      options: {
        type: 'local'
      }
    }] : []);
    onEditOpen();
  };

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    if (name.includes('.')) {
      const [parent, child] = name.split('.');
      setFormData(prev => ({
        ...prev,
        [parent]: {
          ...prev[parent],
          [child]: value
        }
      }));
    } else {
      setFormData(prev => ({
        ...prev,
        [name]: value
      }));
    }
  };

  const handleVerDetalle = (id) => {
    navigate(`/admin/alumnos/${id}`);
  };

  const handleClickBaja = (alumno) => {
    setAlumnoParaBaja(alumno);
    onBajaOpen();
  };

  const handleConfirmarBaja = async () => {
    if (!alumnoParaBaja) return;
    try {
      await updateClient(alumnoParaBaja.id, { is_active: false });
      toast({
        title: 'Alumno dado de baja',
        description: `El alumno ${alumnoParaBaja.nombre} ha sido dado de baja.`,
        status: 'warning',
        duration: 5000,
        isClosable: true,
      });
      fetchAlumnos();
    } catch (error) {
      toast({ title: 'Error al dar de baja', description: error.message, status: 'error' });
    }
    onBajaClose();
    setAlumnoParaBaja(null);
  };

  const handleGuardar = async (e) => {
    e.preventDefault();
    setIsLoading(true);

    try {
      const payload = {
        first_name: formData.first_name,
        last_name: formData.last_name,
        email: formData.email,
        password: 'password123', // Default for students too?
        role: 'ESTUDIANTE', // or CLIENT
        phone: formData.telefono,
        curp: formData.curp,
        birthdate: formData.fecha_nacimiento, // Assuming backend uses birthdate
        institution_id: JSON.parse(sessionStorage.getItem('user'))?.institution_id,
        // parent_id: formData.id_padre // Need to check if backend supports parenting linking at creation
      };

      payload.name = `${formData.first_name} ${formData.last_name}`.trim();

      let savedUser;
      if (alumnoEditando) {
        delete payload.password;
        savedUser = await updateClient(alumnoEditando.id, payload);
      } else {
        savedUser = await createClient(payload);
      }

      if (files.length > 0 && files[0].file) {
        await uploadEmployeePhoto(savedUser.id, files[0].file);
      }

      toast({
        title: 'Alumno guardado',
        description: alumnoEditando ? 'Alumno actualizado' : 'Alumno creado',
        status: 'success',
        duration: 2000,
      });

      onClose();
      onEditClose();
      fetchAlumnos();

    } catch (error) {
      console.error("Error saving student:", error);
      toast({
        title: 'Error al guardar',
        description: error.message,
        status: 'error',
        duration: 4000
      });
    } finally {
      setIsLoading(false);
    }
  };

  const getNombrePadre = (id) => {
    return 'No asignado (TBD)'; // We don't have parents loaded yet, mock was removed
  };

  return (
    <Layout links={adminLinks} userType="Administrador" userName="Admin User" userPhoto="https://i.pravatar.cc/150?img=68">
      <Box>

        {/* Instrucciones minimalistas */}
        <Text fontSize="sm" color="gray.600" mb={6}>
          Administra los alumnos inscritos en la escuela. Puedes ver su información, editarla o agregar nuevos alumnos.
        </Text>

        {/* ¡LA TARJETA! Aquí está la magia */}
        <Box
          bg="rgba(255, 255, 255, 0.6)"
          backdropFilter="blur(20px) saturate(180%)"
          WebkitBackdropFilter="blur(20px) saturate(180%)"
          borderRadius="xl"
          boxShadow="0 8px 32px 0 rgba(0, 0, 0, 0.1)"
          border="1px solid"
          borderColor="rgba(255, 255, 255, 0.3)"
          overflow="hidden"
        >

          {/* Header de la Tarjeta (como en "Facit") */}
          <Flex justify="space-between" align="center" p={5} borderBottom="1px" borderColor="rgba(255, 255, 255, 0.2)">
            <Heading as="h3" size="md">Lista de Alumnos</Heading>
            {/* Botón "Píldora" */}
            <Button
              colorScheme="blue"
              borderRadius="full"
              leftIcon={<Icon as={BiPlus} />}
              onClick={handleCrear}
              isLoading={isLoading}
            >
              Crear Nuevo
            </Button>
          </Flex>

          {/* La tabla va DENTRO del cuerpo de la tarjeta */}
          <TableContainer>
            {/* ¡Quitar 'variant="striped"'! */}
            <Table variant="simple">
              <Thead>
                {/* Estilo de cabecera moderno */}
                <Tr>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Alumno</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Padre/Tutor</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Edad</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Teléfono</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Acciones</Th>
                </Tr>
              </Thead>
              <Tbody>
                {alumnos.map((alumno) => (
                  <Tr key={alumno.id} _hover={{ bg: "gray.50" }}>
                    <Td>
                      <HStack spacing={3}>
                        <Avatar size="sm" name={alumno.nombre} src={alumno.foto_url} />
                        <Box>
                          <Text fontWeight="bold">{alumno.nombre}</Text>
                          <Text fontSize="sm" color="gray.500">{alumno.email}</Text>
                        </Box>
                      </HStack>
                    </Td>
                    <Td>{getNombrePadre(alumno.id_padre)}</Td>
                    <Td>{calcularEdad(alumno.fecha_nacimiento)} años</Td>
                    <Td>{alumno.telefono || 'N/A'}</Td>
                    <Td>
                      <HStack spacing={2}>
                        <Tooltip label="Ver Detalle">
                          <IconButton
                            icon={<Icon as={BiShow} />}
                            variant="ghost"
                            colorScheme="blue"
                            aria-label="Ver Detalle"
                            onClick={() => handleVerDetalle(alumno.id)}
                          />
                        </Tooltip>
                        <Tooltip label="Editar">
                          <IconButton
                            icon={<BiPencil />}
                            aria-label="Editar"
                            size="sm"
                            variant="ghost"
                            colorScheme="blue"
                            onClick={() => handleEditar(alumno)}
                          />
                        </Tooltip>
                        <Tooltip label="Dar de baja">
                          <IconButton
                            icon={<BiUserMinus />}
                            aria-label="Dar de baja"
                            size="sm"
                            variant="ghost"
                            colorScheme="orange"
                            onClick={() => handleClickBaja(alumno)}
                          />
                        </Tooltip>
                        <Tooltip label="Eliminar">
                          <IconButton
                            icon={<BiTrash />}
                            aria-label="Eliminar"
                            size="sm"
                            variant="ghost"
                            colorScheme="red"
                          />
                        </Tooltip>
                      </HStack>
                    </Td>
                  </Tr>
                ))}
                {alumnos.length === 0 && !isLoading && (
                  <Tr>
                    <Td colSpan={5} textAlign="center" py={4}>No hay alumnos registrados.</Td>
                  </Tr>
                )}
              </Tbody>
            </Table>
          </TableContainer>
        </Box>
      </Box>

      <Modal isOpen={isOpen || isEditOpen} onClose={() => { onClose(); onEditClose(); }} isCentered motionPreset="scale" size="6xl">
        <ModalOverlay bg="blackAlpha.600" backdropFilter="blur(4px)" />
        <ModalContent
          borderRadius="16px"
          bg="rgba(255, 255, 255, 0.95)"
          backdropFilter="blur(20px) saturate(180%)"
          boxShadow="0 8px 32px 0 rgba(0, 0, 0, 0.2)"
        >
          <ModalHeader borderBottom="1px solid" borderColor="rgba(255, 255, 255, 0.2)" pb={4}>
            {alumnoEditando ? 'Editar Alumno' : 'Crear Nuevo Alumno'}
          </ModalHeader>
          <ModalCloseButton />
          <form onSubmit={handleGuardar}>
            <ModalBody p={6}>
              <Flex gap={8} flexDirection={{ base: 'column', lg: 'row' }}>
                {/* Columna Izquierda: Foto */}
                <Box w={{ base: '100%', lg: '250px' }} flexShrink={0}>
                  <FormLabel textAlign="center" fontWeight="bold" mb={4} color="gray.700">Fotografía</FormLabel>
                  <Box
                    w="220px"
                    h="220px"
                    mx="auto"
                    borderRadius="full"
                    overflow="hidden"
                    boxShadow="0 4px 12px rgba(0,0,0,0.1)"
                    bg="gray.50"
                    border="2px solid"
                    borderColor="gray.100"
                  >
                    <FilePond
                      files={files}
                      onupdatefiles={setFiles}
                      allowMultiple={false}
                      maxFiles={1}
                      name="photo"
                      labelIdle='<span class="filepond--label-action">Subir Foto</span>'
                      acceptedFileTypes={['image/*']}
                      stylePanelLayout="compact circle"
                      styleLoadIndicatorPosition="center bottom"
                      styleProgressIndicatorPosition="right bottom"
                      styleButtonRemoveItemPosition="left bottom"
                      styleButtonProcessItemPosition="right bottom"
                    />
                  </Box>
                  <Text fontSize="xs" color="gray.500" textAlign="center" mt={3}>
                    Arrastra una imagen o haz clic para seleccionar
                  </Text>
                </Box>

                {/* Columna Derecha: Datos */}
                <Box flex={1}>
                  <VStack spacing={6} align="stretch">

                    {/* Sección Personal */}
                    <Box bg="white" p={4} borderRadius="lg" border="1px solid" borderColor="gray.100" shadow="sm">
                      <Heading size="sm" mb={4} color="blue.600" borderBottom="1px solid" borderColor="gray.100" pb={2} display="flex" alignItems="center">
                        <Icon as={BiPencil} mr={2} /> Información Personal
                      </Heading>
                      <Grid templateColumns={{ base: '1fr', md: 'repeat(2, 1fr)' }} gap={4}>
                        <GridItem>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Nombre(s)</FormLabel>
                            <Input name="first_name" value={formData.first_name} onChange={handleInputChange} placeholder="Ej. Sofía" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Apellidos</FormLabel>
                            <Input name="last_name" value={formData.last_name} onChange={handleInputChange} placeholder="Ej. López" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Email</FormLabel>
                            <Input name="email" type="email" value={formData.email} onChange={handleInputChange} placeholder="correo@ejemplo.com" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Fecha de Nacimiento</FormLabel>
                            <Input name="fecha_nacimiento" type="date" value={formData.fecha_nacimiento} onChange={handleInputChange} borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Edad Calculada</FormLabel>
                            <Input value={formData.fecha_nacimiento ? `${calcularEdad(formData.fecha_nacimiento)} años` : ''} isReadOnly bg="gray.50" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        {/* 
                         TODO: Implement Parent Selection logic. For now removed mockPadres and use placeholder.
                         If parents are users with role 'PARENT', we should fetch them.
                        */}

                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Teléfono</FormLabel>
                            <Input name="telefono" value={formData.telefono} onChange={handleInputChange} placeholder="555-555-5555" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem colSpan={{ base: 1, md: 2 }}>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">CURP</FormLabel>
                            <Input name="curp" value={formData.curp} onChange={handleInputChange} placeholder="ABCD123456..." borderRadius="md" />
                          </FormControl>
                        </GridItem>
                      </Grid>
                    </Box>

                    {/* Sección Dirección */}
                    <Box bg="white" p={4} borderRadius="lg" border="1px solid" borderColor="gray.100" shadow="sm">
                      <Heading size="sm" mb={4} color="blue.600" borderBottom="1px solid" borderColor="gray.100" pb={2} display="flex" alignItems="center">
                        <Icon as={BiPencil} mr={2} /> Dirección
                      </Heading>
                      <Grid templateColumns={{ base: '1fr', md: 'repeat(2, 1fr)' }} gap={4}>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Estado</FormLabel>
                            <Select name="direccion.estado" value={formData.direccion.estado} onChange={handleInputChange} placeholder="Selecciona estado" borderRadius="md">
                              {estadosMexico.map(e => (
                                <option key={e.nombre} value={e.nombre}>{e.nombre}</option>
                              ))}
                            </Select>
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Municipio</FormLabel>
                            <Select name="direccion.municipio" value={formData.direccion.municipio} onChange={handleInputChange} placeholder="Selecciona municipio" borderRadius="md">
                              {municipiosDisponibles.map(m => (
                                <option key={m} value={m}>{m}</option>
                              ))}
                            </Select>
                          </FormControl>
                        </GridItem>
                        <GridItem colSpan={{ base: 1, md: 2 }}>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Calle y Número</FormLabel>
                            <HStack>
                              <Input name="direccion.calle" value={formData.direccion.calle} onChange={handleInputChange} placeholder="Calle" flex={2} borderRadius="md" />
                              <Input name="direccion.numero" value={formData.direccion.numero} onChange={handleInputChange} placeholder="No." flex={1} borderRadius="md" />
                            </HStack>
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Código Postal</FormLabel>
                            <Input name="direccion.cp" value={formData.direccion.cp} onChange={handleInputChange} placeholder="00000" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                      </Grid>
                    </Box>

                  </VStack>
                </Box>
              </Flex>
            </ModalBody>
            <ModalFooter borderTop="1px solid" borderColor="rgba(255, 255, 255, 0.2)" pt={4}>
              <Button variant="ghost" mr={3} onClick={() => { onClose(); onEditClose(); }} borderRadius="8px">Cancelar</Button>
              <Button colorScheme="blue" type="submit" borderRadius="8px" px={6} isLoading={isLoading}>Guardar</Button>
            </ModalFooter>
          </form>
        </ModalContent>
      </Modal>

      {/* Diálogo de Confirmación de Baja */}
      <AlertDialog
        isOpen={isBajaOpen}
        leastDestructiveRef={cancelRef}
        onClose={onBajaClose}
      >
        <AlertDialogOverlay>
          <AlertDialogContent>
            <AlertDialogHeader fontSize="lg" fontWeight="bold">
              Dar de baja alumno
            </AlertDialogHeader>

            <AlertDialogBody>
              ¿Estás seguro que deseas dar de baja a <strong>{alumnoParaBaja?.nombre}</strong>?
              <br /><br />
              <Text color="orange.600" fontWeight="medium">
                Esta acción liberará todas las clases asignadas a este alumno en el calendario.
              </Text>
            </AlertDialogBody>

            <AlertDialogFooter>
              <Button ref={cancelRef} onClick={onBajaClose}>
                Cancelar
              </Button>
              <Button colorScheme="orange" onClick={handleConfirmarBaja} ml={3}>
                Dar de Baja
              </Button>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialogOverlay>
      </AlertDialog>
    </Layout>
  );
};

export default GestionAlumnos;

