import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    useToast,
    HStack,
    Text,
    Avatar,
    Switch,
    Tooltip,
    VStack,
    Icon,
    SimpleGrid
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiEye, FiEyeOff, FiPower, FiUploadCloud } from 'react-icons/fi';
import { InputGroup, InputRightElement } from '@chakra-ui/react';
import { getClients as fetchClients, createClient, updateUser as updateClient, uploadEmployeePhoto } from '../../services/api';
import { processImage } from '../../utils/imageUtils';

const validatePhone = (phone) => /^[0-9]{10}$/.test(phone.replace(/\s/g, ''));
const validateEmail = (email) => /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
const validateCURP = (curp) => {
    if (!curp) return true; // Optional
    return /^[A-Z]{4}[0-9]{6}[HM][A-Z]{5}[A-Z0-9]{2}$/.test(curp);
};
const validateRFC = (rfc) => {
    if (!rfc) return true; // Optional
    const rfc12 = /^[A-Z]{3}[0-9]{6}[A-Z]{1}[A-Z0-9]{2}$/;
    const rfc13 = /^[A-Z]{4}[0-9]{6}[A-Z]{1}[A-Z0-9]{2}$/;
    return rfc12.test(rfc) || rfc13.test(rfc);
};

const formatPhone = (value) => {
    if (!value) return value;
    const phoneNumber = value.replace(/[^\d]/g, '');
    const phoneNumberLength = phoneNumber.length;
    if (phoneNumberLength <= 2) return phoneNumber;
    if (phoneNumberLength <= 6) {
        return `${phoneNumber.slice(0, 2)} ${phoneNumber.slice(2)}`;
    }
    return `${phoneNumber.slice(0, 2)} ${phoneNumber.slice(2, 6)} ${phoneNumber.slice(6, 10)}`;
};

const GestionClientes = () => {
    const [showPassword, setShowPassword] = useState(false);
    const [isDragging, setIsDragging] = useState(false); // State for drag effect

    const [clients, setClients] = useState([]);
    const [currentClient, setCurrentClient] = useState(null);
    const [currentUser, setCurrentUser] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    const loadClients = async () => {
        try {
            const allClients = await fetchClients();
            const user = JSON.parse(sessionStorage.getItem('user'));
            const institutionId = user ? (user.institution_id || user.company_id) : null;

            if (institutionId) {
                setClients(allClients.filter(c => 
                    (c.institution_id === institutionId || c.company_id === institutionId)
                ));
            } else {
                setClients(allClients);
            }
        } catch (error) {
            toast({
                title: 'Error al cargar estudiantes',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    const handleToggleStatus = async (client) => {
        try {
            const newStatus = !client.is_active;
            await updateClient(client.user_id, { ...client, is_active: newStatus });

            // Actualizar el estado local inmediatamente
            setClients(clients.map(c =>
                c.user_id === client.user_id ? { ...c, is_active: newStatus } : c
            ));

            toast({
                title: `Estudiante ${newStatus ? 'activado' : 'desactivado'}`,
                status: 'success',
                duration: 2000,
            });
        } catch (error) {
            toast({
                title: 'Error al cambiar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        const user = JSON.parse(sessionStorage.getItem('user'));
        setCurrentUser(user);
        loadClients();
    }, []);

    const handleEdit = (client) => {
        setCurrentClient(client);
        onOpen();
    };

    const handleCreate = () => {
        const user = JSON.parse(sessionStorage.getItem('user'));
        const institutionId = user ? (user.institution_id || user.company_id) : null;
        
        setCurrentClient({
            first_name: '',
            last_name: '',
            second_last_name: '',
            email: '',
            phone: '',
            client_number: `EST-${new Date().getFullYear()}-${clients.length + 1}`,
            curp: '',
            is_active: true,
            address: '',
            fiscal_address: '',
            rfc: '',
            role: 'ESTUDIANTE',
            institution_id: institutionId,
            photo_url: null,
            _pendingFile: null
        });
        onOpen();
    };

    const handleSave = async () => {
        // Validaciones
        if (!currentClient.first_name || !currentClient.first_name.trim()) {
            toast({ title: 'Nombre(s) requerido', status: 'error', duration: 3000 });
            return;
        }
        if (!currentClient.last_name || !currentClient.last_name.trim()) {
            toast({ title: 'Primer Apellido requerido', status: 'error', duration: 3000 });
            return;
        }
        if (!validateEmail(currentClient.email)) {
            toast({ title: 'Email inválido', status: 'error', duration: 3000 });
            return;
        }
        if (!validatePhone(currentClient.phone)) {
            toast({ title: 'Teléfono inválido (debe tener 10 dígitos)', status: 'error', duration: 3000 });
            return;
        }
        if (currentClient.curp && !validateCURP(currentClient.curp.toUpperCase())) {
            toast({ title: 'CURP inválido', description: 'El formato del CURP es incorrecto.', status: 'error', duration: 3000 });
            return;
        }
        if (currentClient.rfc && !validateRFC(currentClient.rfc.toUpperCase())) {
            toast({ title: 'RFC inválido', description: 'Formato incorrecto para 12 o 13 caracteres.', status: 'error', duration: 3000 });
            return;
        }

        try {
            const finalCURP = currentClient.curp ? currentClient.curp.toUpperCase() : null;
            const finalEmail = currentClient.email ? currentClient.email.toLowerCase().trim() : '';
            const finalPhone = currentClient.phone ? currentClient.phone.replace(/\s/g, '') : '';
            let savedClient;
            const pendingFile = currentClient._pendingFile;

            if (currentClient.user_id) {
                const dataToUpdate = { 
                    ...currentClient, 
                    curp: finalCURP, 
                    email: finalEmail,
                    phone: finalPhone
                };
                delete dataToUpdate._pendingFile; // Limpiar campo temporal
                delete dataToUpdate.user_id; // Ya se pasa como parámetro
                delete dataToUpdate.company_id; // No cambiar ID de institución al editar
                delete dataToUpdate.allow_access_without_service; // Removed field
                
                if (!dataToUpdate.password) {
                    delete dataToUpdate.password;
                }
                savedClient = await updateClient(currentClient.user_id, dataToUpdate);

                // Si hay una foto pendiente en la edición, subirla
                if (pendingFile) {
                    try {
                        await uploadEmployeePhoto(currentClient.user_id, pendingFile);
                    } catch (err) {
                        console.error('Error uploading student photo on update:', err);
                    }
                }

                toast({
                    title: 'Estudiante actualizado',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                const newClientData = {
                    ...currentClient,
                    curp: finalCURP,
                    email: finalEmail,
                    phone: finalPhone,
                    role: 'ESTUDIANTE'
                };
                delete newClientData._pendingFile;
                delete newClientData.allow_access_without_service;
                savedClient = await createClient(newClientData);

                // Si hay una foto pendiente, subirla ahora
                if (pendingFile && savedClient) {
                    try {
                        await uploadEmployeePhoto(savedClient.id || savedClient.user_id, pendingFile);
                    } catch (err) {
                        toast({
                            title: 'Estudiante creado, pero error al subir foto',
                            description: err.message,
                            status: 'warning',
                            duration: 4000,
                        });
                    }
                }

                toast({
                    title: 'Estudiante creado',
                    status: 'success',
                    duration: 2000,
                });
            }

            loadClients();
            onClose();
        } catch (error) {
            // Detectar errores de campos duplicados
            const errorMessage = error.message || error.toString();
            
            let friendlyMessage = errorMessage;
            
            if (errorMessage.includes('correo es utilizado') || errorMessage.includes('users_email_key') || 
                (errorMessage.includes('duplicate key') && errorMessage.includes('email'))) {
                friendlyMessage = 'El correo es utilizado por otro usuario, favor de ingresar otro correo electrónico';
            } else if (errorMessage.includes('teléfono es utilizado') || errorMessage.includes('users_phone_key') || 
                       (errorMessage.includes('duplicate key') && errorMessage.includes('phone'))) {
                friendlyMessage = 'El teléfono es utilizado por otro usuario, favor de ingresar otro teléfono';
            } else if (errorMessage.includes('CURP es utilizado') || errorMessage.includes('users_curp_key') || 
                       (errorMessage.includes('duplicate key') && errorMessage.includes('curp'))) {
                friendlyMessage = 'El CURP es utilizado por otro usuario, favor de ingresar otro CURP';
            }
            
            toast({
                title: 'Error al guardar',
                description: friendlyMessage,
                status: 'error',
                duration: 4000,
            });
        }
    };

    const max_students = currentUser?.max_students || currentUser?.max_clients || 0;
    const isLimitReached = max_students > 0 && clients.length >= max_students;

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <VStack align="start" spacing={1}>
                    <Heading>Gestión de Estudiantes</Heading>
                    {isLimitReached && (
                        <Text color="red.500" fontSize="sm" fontWeight="bold">
                            Límite de estudiantes alcanzado ({max_students}). Por favor, comunícate con tu ejecutivo para ampliar tu plan.
                        </Text>
                    )}
                </VStack>
                <Tooltip
                    label={isLimitReached ? "Límite alcanzado. Contacta a tu ejecutivo." : ""}
                    isDisabled={!isLimitReached}
                >
                    <Button
                        leftIcon={<FiPlus />}
                        colorScheme="brand"
                        onClick={handleCreate}
                        isDisabled={isLimitReached}
                    >
                        Nuevo Estudiante
                    </Button>
                </Tooltip>
            </HStack>


            {clients.length === 0 ? (
                <Box 
                    bg="white" 
                    borderRadius="lg" 
                    boxShadow="sm" 
                    p={12}
                    textAlign="center"
                >
                    <VStack spacing={6}>
                        <Box>
                            <Icon 
                                as={FiPlus} 
                                boxSize={16} 
                                color="gray.300" 
                                mb={4}
                            />
                            <Heading size="lg" color="gray.600" mb={2}>
                                No hay estudiantes registrados
                            </Heading>
                            <Text color="gray.500" fontSize="md" maxW="md" mx="auto">
                                Comienza a gestionar tu institución registrando tus primeros estudiantes. 
                                Podrás asignarles exámenes y dar seguimiento a su progreso académico.
                            </Text>
                        </Box>
                        <Button
                            leftIcon={<FiPlus />}
                            colorScheme="brand"
                            size="lg"
                            onClick={handleCreate}
                            isDisabled={isLimitReached}
                        >
                            Registrar Primer Estudiante
                        </Button>
                        {isLimitReached && (
                            <Text color="red.500" fontSize="sm" fontWeight="bold">
                                Límite de estudiantes alcanzado. Por favor, comunícate con tu ejecutivo para ampliar tu plan.
                            </Text>
                        )}
                    </VStack>
                </Box>
            ) : (
                <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                    <Table variant="simple">
                        <Thead bg="gray.50">
                            <Tr>
                                <Th>Estudiante</Th>
                                <Th>Contacto</Th>
                                <Th>RFC</Th>
                                <Th>Acceso Especial</Th>
                                <Th>Estado</Th>
                                <Th>Acciones</Th>
                            </Tr>
                        </Thead>
                        <Tbody>
                            {clients.map((client) => (
                                <Tr key={client.user_id}>
                                    <Td>
                                        <HStack>
                                            <Avatar 
                                                size="sm" 
                                                name={client.name} 
                                                src={client.photo_url} 
                                                bg="brand.100" 
                                                color="brand.700" 
                                            />
                                            <Box>
                                                <Text fontWeight="bold">{client.name}</Text>
                                                <Text fontSize="xs" color="gray.500">{client.client_number}</Text>
                                            </Box>
                                        </HStack>
                                    </Td>
                                    <Td>
                                        <Text fontSize="sm">{client.email}</Text>
                                        <Text fontSize="xs" color="gray.500">{formatPhone(client.phone)}</Text>
                                    </Td>
                                    <Td>{client.rfc || 'N/A'}</Td>
                                    <Td>
                                        <Badge colorScheme={client.allow_access_without_service ? 'purple' : 'gray'}>
                                            {client.allow_access_without_service ? 'PERMITIDO' : 'NORMAL'}
                                        </Badge>
                                    </Td>
                                    <Td>
                                        <Badge colorScheme={client.is_active ? 'green' : 'red'}>
                                            {client.is_active ? 'ACTIVO' : 'INACTIVO'}
                                        </Badge>
                                    </Td>
                                    <Td>
                                        <HStack spacing={2}>
                                            <Switch
                                                size="sm"
                                                isChecked={client.is_active}
                                                onChange={() => handleToggleStatus(client)}
                                                colorScheme="green"
                                            />
                                            <IconButton
                                                icon={<FiEdit2 />}
                                                size="sm"
                                                aria-label="Editar"
                                                onClick={() => handleEdit(client)}
                                            />
                                        </HStack>
                                    </Td>
                                </Tr>
                            ))}
                        </Tbody>
                    </Table>
                </Box>
            )}

            {/* Modal */}
            <Modal isOpen={isOpen} onClose={onClose} size="xl">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentClient?.user_id ? 'Editar Estudiante' : 'Nuevo Estudiante'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <VStack spacing={4} align="stretch">
                            {/* Primera fila: Nombre(s) y Primer Apellido */}
                            <SimpleGrid columns={2} spacing={4}>
                                <FormControl isRequired>
                                    <FormLabel fontSize="sm">Nombre(s)</FormLabel>
                                    <Input
                                        size="sm"
                                        value={currentClient?.first_name || ''}
                                        onChange={(e) => setCurrentClient({ ...currentClient, first_name: e.target.value })}
                                        placeholder="Ej. Juan Carlos"
                                    />
                                </FormControl>

                                <FormControl isRequired>
                                    <FormLabel fontSize="sm">Primer Apellido</FormLabel>
                                    <Input
                                        size="sm"
                                        value={currentClient?.last_name || ''}
                                        onChange={(e) => setCurrentClient({ ...currentClient, last_name: e.target.value })}
                                        placeholder="Ej. Pérez"
                                    />
                                </FormControl>
                            </SimpleGrid>

                            {/* Segunda fila: Segundo Apellido */}
                            <FormControl>
                                <FormLabel fontSize="sm">Segundo Apellido (Opcional)</FormLabel>
                                <Input
                                    size="sm"
                                    value={currentClient?.second_last_name || ''}
                                    onChange={(e) => setCurrentClient({ ...currentClient, second_last_name: e.target.value })}
                                    placeholder="Ej. González"
                                />
                            </FormControl>

                            {/* Segunda fila: Email y Teléfono */}
                            <SimpleGrid columns={2} spacing={4}>
                                <FormControl isRequired>
                                    <FormLabel fontSize="sm">Email</FormLabel>
                                    <Input
                                        size="sm"
                                        type="email"
                                        value={currentClient?.email || ''}
                                        onChange={(e) => setCurrentClient({ ...currentClient, email: e.target.value })}
                                        onBlur={(e) => {
                                            if (e.target.value && !validateEmail(e.target.value)) {
                                                toast({ title: 'Email inválido', status: 'warning', duration: 2000 });
                                            }
                                        }}
                                    />
                                </FormControl>

                                <FormControl isRequired>
                                    <FormLabel fontSize="sm">Teléfono</FormLabel>
                                    <Input
                                        size="sm"
                                        value={currentClient?.phone || ''}
                                        onChange={(e) => setCurrentClient({ ...currentClient, phone: formatPhone(e.target.value) })}
                                        onBlur={(e) => {
                                            if (e.target.value && !validatePhone(e.target.value)) {
                                                toast({ title: 'Teléfono incompleto', description: 'Debe tener 10 dígitos', status: 'warning', duration: 2000 });
                                            }
                                        }}
                                        placeholder="55 5555 5555"
                                    />
                                </FormControl>
                            </SimpleGrid>

                            {/* Tercera fila: RFC y CURP */}
                            <SimpleGrid columns={2} spacing={4}>
                                <FormControl>
                                    <FormLabel fontSize="sm">RFC (Opcional)</FormLabel>
                                    <Input
                                        size="sm"
                                        value={currentClient?.rfc || ''}
                                        onChange={(e) => setCurrentClient({ ...currentClient, rfc: e.target.value.toUpperCase() })}
                                        onBlur={(e) => {
                                            if (e.target.value && !validateRFC(e.target.value)) {
                                                toast({ title: 'RFC inválido', status: 'warning', duration: 2000 });
                                            }
                                        }}
                                        maxLength={13}
                                        placeholder="ABCD123456EFG"
                                    />
                                    <Text fontSize="xs" color="gray.500" mt={1}>12 o 13 caracteres</Text>
                                </FormControl>

                                <FormControl>
                                    <FormLabel fontSize="sm">CURP (Opcional)</FormLabel>
                                    <Input
                                        size="sm"
                                        value={currentClient?.curp || ''}
                                        onChange={(e) => setCurrentClient({ ...currentClient, curp: e.target.value.toUpperCase() })}
                                        onBlur={(e) => {
                                            if (e.target.value && !validateCURP(e.target.value)) {
                                                toast({ title: 'CURP inválido', status: 'warning', duration: 2000 });
                                            }
                                        }}
                                        placeholder="ABCD123456HXXXXX00"
                                        maxLength={18}
                                    />
                                    <Text fontSize="xs" color="gray.500" mt={1}>18 caracteres</Text>
                                </FormControl>
                            </SimpleGrid>

                            {/* Cuarta fila: Fotografía y Contraseña */}
                            <SimpleGrid columns={2} spacing={4}>
                                <FormControl>
                                    <FormLabel fontSize="sm">Fotografía (Opcional)</FormLabel>
                                    <VStack spacing={2} align="stretch">
                                        <HStack spacing={3}>
                                            <Avatar
                                                size="md"
                                                name={currentClient?.name}
                                                src={currentClient?._pendingFile ? URL.createObjectURL(currentClient._pendingFile) : currentClient?.photo_url}
                                                bg="brand.100"
                                                color="brand.700"
                                                border="2px solid"
                                                borderColor="brand.200"
                                            />
                                            <Box flex="1">
                                                <Box
                                                    position="relative"
                                                    height="80px"
                                                    border="2px dashed"
                                                    borderColor={isDragging ? "brand.500" : "gray.200"}
                                                    borderRadius="md"
                                                    bg={isDragging ? "brand.50" : "gray.50"}
                                                    transition="all 0.2s"
                                                    _hover={{ borderColor: "brand.400", bg: "gray.100" }}
                                                    onDragOver={(e) => { e.preventDefault(); setIsDragging(true); }}
                                                    onDragLeave={() => setIsDragging(false)}
                                                    onDrop={async (e) => {
                                                        e.preventDefault();
                                                        setIsDragging(false);
                                                        const file = e.dataTransfer.files[0];
                                                        if (file && file.type.startsWith('image/')) {
                                                            const processedFile = await processImage(file);
                                                            setCurrentClient({ ...currentClient, _pendingFile: processedFile });
                                                        }
                                                    }}
                                                >
                                                    <VStack
                                                        height="100%"
                                                        justify="center"
                                                        spacing={0}
                                                        cursor="pointer"
                                                        onClick={() => document.getElementById('photo-client-input').click()}
                                                    >
                                                        <Icon as={FiUploadCloud} boxSize={5} color="brand.500" />
                                                        <Text fontSize="xs" color="gray.600">Clic para subir</Text>
                                                    </VStack>
                                                    
                                                    <Input
                                                        id="photo-client-input"
                                                        type="file"
                                                        display="none"
                                                        accept="image/*"
                                                        onChange={async (e) => {
                                                            const file = e.target.files[0];
                                                            if (file) {
                                                                const processedFile = await processImage(file);
                                                                setCurrentClient({ ...currentClient, _pendingFile: processedFile });
                                                            }
                                                        }}
                                                    />
                                                </Box>
                                            </Box>
                                        </HStack>
                                        <Text fontSize="xs" color="gray.500">
                                            JPG, PNG. 400x400px sugerido
                                        </Text>
                                    </VStack>
                                </FormControl>

                                <FormControl>
                                    <FormLabel fontSize="sm">
                                        Contraseña {currentClient?.user_id && '(Opcional)'}
                                    </FormLabel>
                                    <InputGroup size="sm">
                                        <Input
                                            type={showPassword ? "text" : "password"}
                                            placeholder={currentClient?.user_id ? '••••••••' : 'Ingresar contraseña'}
                                            value={currentClient?.password || ''}
                                            onChange={(e) => setCurrentClient({ ...currentClient, password: e.target.value })}
                                        />
                                        <InputRightElement h="full" width="3rem">
                                            <Button
                                                variant="ghost"
                                                size="sm"
                                                h="1.75rem"
                                                onMouseDown={() => setShowPassword(true)}
                                                onMouseUp={() => setShowPassword(false)}
                                                onMouseLeave={() => setShowPassword(false)}
                                                onTouchStart={() => setShowPassword(true)}
                                                onTouchEnd={() => setShowPassword(false)}
                                            >
                                                {showPassword ? <FiEyeOff size={16} /> : <FiEye size={16} />}
                                            </Button>
                                        </InputRightElement>
                                    </InputGroup>
                                    {!currentClient?.user_id && (
                                        <Text fontSize="xs" color="gray.500" mt={1}>
                                            Requerida para nuevos estudiantes
                                        </Text>
                                    )}
                                </FormControl>
                            </SimpleGrid>

                            {/* Quinta fila: Dirección Fiscal */}
                            <FormControl>
                                <FormLabel fontSize="sm">Dirección Fiscal (Opcional)</FormLabel>
                                <Input
                                    size="sm"
                                    value={currentClient?.fiscal_address || ''}
                                    onChange={(e) => setCurrentClient({ ...currentClient, fiscal_address: e.target.value })}
                                />
                            </FormControl>

                            {/* Sexta fila: Dirección y Estado Activo */}
                            <SimpleGrid columns={2} spacing={4}>
                                <FormControl>
                                    <FormLabel fontSize="sm">Dirección (Opcional)</FormLabel>
                                    <Input
                                        size="sm"
                                        value={currentClient?.address || ''}
                                        onChange={(e) => setCurrentClient({ ...currentClient, address: e.target.value })}
                                    />
                                </FormControl>

                                <FormControl display="flex" alignItems="center" justifyContent="flex-end">
                                    <HStack spacing={3}>
                                        <FormLabel mb="0" fontSize="sm">
                                            Estudiante Activo
                                        </FormLabel>
                                        <Switch
                                            size="sm"
                                            isChecked={currentClient?.is_active || false}
                                            onChange={(e) => setCurrentClient({ ...currentClient, is_active: e.target.checked })}
                                        />
                                    </HStack>
                                </FormControl>
                            </SimpleGrid>
                        </VStack>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionClientes;
