import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Select,
    useToast,
    HStack,
    VStack,
    Text,
    Avatar,
    Tooltip,
    Icon,
    SimpleGrid
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiPower, FiEye, FiEyeOff, FiUploadCloud, FiUser } from 'react-icons/fi';
import { InputGroup, InputRightElement } from '@chakra-ui/react';
import { getEmployees, createEmployee, updateEmployee, getSpecialties } from '../../services/api';
import { processImage } from '../../utils/imageUtils';

const validatePhone = (phone) => /^[0-9]{10}$/.test(phone.replace(/\s/g, ''));
const validateEmail = (email) => /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
const validateCURP = (curp) => {
    if (!curp) return true; // Optional
    return /^[A-Z]{4}[0-9]{6}[HM][A-Z]{5}[A-Z0-9]{2}$/.test(curp);
};

const formatPhone = (value) => {
    if (!value) return value;
    const phoneNumber = value.replace(/[^\d]/g, '');
    const phoneNumberLength = phoneNumber.length;
    if (phoneNumberLength <= 2) return phoneNumber;
    if (phoneNumberLength <= 6) {
        return `${phoneNumber.slice(0, 2)} ${phoneNumber.slice(2)}`;
    }
    return `${phoneNumber.slice(0, 2)} ${phoneNumber.slice(2, 6)} ${phoneNumber.slice(6, 10)}`;
};

const GestionEmpleados = () => {
    const [showPassword, setShowPassword] = useState(false);
    const [isDragging, setIsDragging] = useState(false); // State for drag effect

    const [employees, setEmployees] = useState([]);
    const [specialties, setSpecialties] = useState([]);
    const [currentEmployee, setCurrentEmployee] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    const fetchEmployees = async () => {
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            const institutionId = user ? (user.institution_id || user.company_id) : null;

            console.log('GestionProfesores: Current user in session:', user);
            console.log('GestionProfesores: institutionId to filter:', institutionId);

            if (institutionId) {
                const allEmployees = await getEmployees();
                console.log('GestionProfesores: allEmployees from API:', allEmployees);

                // Filtrar por institución y asegurarnos que sean profesores
                const filtered = allEmployees.filter(e => {
                    const matchesInst = Number(e.institution_id) === Number(institutionId) || Number(e.company_id) === Number(institutionId);
                    const isProf = ((e.role || '').toUpperCase() === 'PROFESOR' || (e.role || '').toUpperCase() === 'EMPLOYEE');
                    return matchesInst && isProf;
                });

                console.log('GestionProfesores: filtered professors:', filtered);
                setEmployees(filtered);
            }
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar profesores',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchEmployees();
        fetchSpecialties();
    }, []);

    const fetchSpecialties = async () => {
        try {
            const data = await getSpecialties();
            setSpecialties(data || []);
        } catch (error) {
            console.error("Error fetching specialties:", error);
        }
    };

    const handleEdit = (employee) => {
        console.log('GestionProfesores: Editing professor:', employee);
        setCurrentEmployee({ ...employee });
        onOpen();
    };

    const handleCreate = () => {
        const user = JSON.parse(sessionStorage.getItem('user'));
        setCurrentEmployee({
            first_name: '',
            last_name: '',
            second_last_name: '',
            email: '',
            specialty: '',
            phone: '',
            curp: '',
            is_active: true,
            role: 'PROFESOR',
            institution_id: user ? (user.institution_id || user.company_id) : null,
            photo_url: null,
            _pendingFile: null
        });
        onOpen();
    };

    const handleSave = async () => {
        // Validaciones
        if (!currentEmployee.first_name || !currentEmployee.first_name.trim()) {
            toast({ title: 'Nombre(s) requerido', status: 'error', duration: 3000 });
            return;
        }
        if (!currentEmployee.last_name || !currentEmployee.last_name.trim()) {
            toast({ title: 'Primer Apellido requerido', status: 'error', duration: 3000 });
            return;
        }
        if (!validateEmail(currentEmployee.email)) {
            toast({ title: 'Email inválido', status: 'error', duration: 3000 });
            return;
        }
        if (!validatePhone(currentEmployee.phone)) {
            toast({ title: 'Teléfono inválido (debe tener 10 dígitos)', status: 'error', duration: 3000 });
            return;
        }
        if (currentEmployee.curp && !validateCURP(currentEmployee.curp.toUpperCase())) {
            toast({ title: 'CURP inválido', description: 'El formato del CURP es incorrecto.', status: 'error', duration: 3000 });
            return;
        }

        try {
            const finalCURP = currentEmployee.curp ? currentEmployee.curp.toUpperCase() : null;
            const finalEmail = currentEmployee.email ? currentEmployee.email.toLowerCase().trim() : '';
            const finalPhone = currentEmployee.phone ? currentEmployee.phone.replace(/\s/g, '') : '';
            let savedEmployee;

            if (currentEmployee.user_id) {
                const dataToUpdate = { ...currentEmployee, curp: finalCURP, email: finalEmail, phone: finalPhone };
                const pendingFile = currentEmployee._pendingFile;
                delete dataToUpdate._pendingFile; // Limpiar campo temporal

                if (!dataToUpdate.password) {
                    delete dataToUpdate.password;
                }
                savedEmployee = await updateEmployee(currentEmployee.user_id, dataToUpdate);

                // Si hay una foto pendiente en la edición, subirla
                if (pendingFile) {
                    try {
                        const { uploadEmployeePhoto } = await import('../../services/api');
                        await uploadEmployeePhoto(currentEmployee.user_id, pendingFile);
                    } catch (err) {
                        console.error('Error uploading photo on update:', err);
                    }
                }

                toast({
                    title: 'Profesor actualizado',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                if (!currentEmployee.password) {
                    toast({
                        title: 'Error',
                        description: 'La contraseña es requerida para nuevos profesores',
                        status: 'error',
                        duration: 3000,
                    });
                    return;
                }

                const dataToCreate = {
                    ...currentEmployee,
                    role: 'PROFESOR',
                    curp: finalCURP,
                    email: finalEmail,
                    phone: finalPhone
                };
                const pendingFile = currentEmployee._pendingFile;
                delete dataToCreate._pendingFile; // Limpiar campo temporal

                savedEmployee = await createEmployee(dataToCreate);

                // Si hay una foto pendiente, subirla ahora
                if (pendingFile) {
                    try {
                        const { uploadEmployeePhoto } = await import('../../services/api');
                        await uploadEmployeePhoto(savedEmployee.id || savedEmployee.user_id, pendingFile);
                    } catch (err) {
                        toast({
                            title: 'Profesor creado, pero error al subir foto',
                            description: err.message,
                            status: 'warning',
                            duration: 4000,
                        });
                    }
                }

                toast({
                    title: 'Profesor creado',
                    status: 'success',
                    duration: 2000,
                });
            }
            fetchEmployees();
            onClose();
        } catch (error) {
            // Detectar errores de campos duplicados
            const errorMessage = error.message || error.toString();

            let friendlyMessage = errorMessage;

            if (errorMessage.includes('correo es utilizado') || errorMessage.includes('users_email_key') ||
                (errorMessage.includes('duplicate key') && errorMessage.includes('email'))) {
                friendlyMessage = 'El correo es utilizado por otro usuario, favor de ingresar otro correo electrónico';
            } else if (errorMessage.includes('teléfono es utilizado') || errorMessage.includes('users_phone_key') ||
                (errorMessage.includes('duplicate key') && errorMessage.includes('phone'))) {
                friendlyMessage = 'El teléfono es utilizado por otro usuario, favor de ingresar otro teléfono';
            } else if (errorMessage.includes('CURP es utilizado') || errorMessage.includes('users_curp_key') ||
                (errorMessage.includes('duplicate key') && errorMessage.includes('curp'))) {
                friendlyMessage = 'El CURP es utilizado por otro usuario, favor de ingresar otro CURP';
            }

            toast({
                title: 'Error al guardar',
                description: friendlyMessage,
                status: 'error',
                duration: 4000,
            });
        }
    };

    const toggleStatus = async (id) => {
        const employee = employees.find(e => e.user_id === id);
        if (!employee) return;

        try {
            const newStatus = !employee.is_active;
            await updateEmployee(id, { ...employee, is_active: newStatus });

            toast({
                title: `Profesor ${newStatus ? 'activado' : 'desactivado'}`,
                status: newStatus ? 'success' : 'warning',
                duration: 2000,
            });
            fetchEmployees();
        } catch (error) {
            toast({
                title: 'Error al actualizar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    const user = JSON.parse(sessionStorage.getItem('user'));
    const max_teachers = user?.max_teachers || user?.max_employees || 0;
    const isLimitReached = max_teachers > 0 && employees.length >= max_teachers;

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <VStack align="start" spacing={1}>
                    <Heading>Gestión de Profesores</Heading>
                    {isLimitReached && (
                        <Text color="red.500" fontSize="sm" fontWeight="bold">
                            Límite de profesores alcanzado ({max_teachers}). Por favor, comunícate con tu ejecutivo para ampliar tu plan.
                        </Text>
                    )}
                </VStack>
                <Tooltip
                    label={isLimitReached ? "Límite alcanzado. Contacta a tu ejecutivo." : ""}
                    isDisabled={!isLimitReached}
                >
                    <Button
                        leftIcon={<FiPlus />}
                        colorScheme="brand"
                        onClick={handleCreate}
                        isDisabled={isLimitReached}
                    >
                        Nuevo Profesor
                    </Button>
                </Tooltip>
            </HStack>


            {employees.length === 0 ? (
                <Box
                    bg="white"
                    borderRadius="lg"
                    boxShadow="sm"
                    p={12}
                    textAlign="center"
                >
                    <VStack spacing={6}>
                        <Box>
                            <Icon
                                as={FiUser}
                                boxSize={16}
                                color="gray.300"
                                mb={4}
                            />
                            <Heading size="lg" color="gray.600" mb={2}>
                                No hay profesores registrados
                            </Heading>
                            <VStack spacing={3} align="stretch" maxW="2xl" mx="auto" mt={4}>
                                <Text color="gray.600" fontSize="md" fontWeight="500">
                                    Los profesores son fundamentales para crear y gestionar exámenes en tu institución.
                                </Text>
                                <Box
                                    bg="blue.50"
                                    p={4}
                                    borderRadius="md"
                                    borderLeft="4px solid"
                                    borderColor="blue.500"
                                    textAlign="left"
                                >
                                    <Text color="blue.900" fontSize="sm" fontWeight="600" mb={2}>
                                        ¿Por qué es importante tener profesores registrados?
                                    </Text>
                                    <VStack align="start" spacing={2} fontSize="sm" color="blue.800">
                                        <Text>• Los profesores pueden crear preguntas y exámenes para sus estudiantes</Text>
                                        <Text>• Permiten organizar el contenido académico por especialidad</Text>
                                        <Text>• Facilitan la asignación de exámenes a grupos específicos de estudiantes</Text>
                                        <Text>• Proporcionan seguimiento y calificación de evaluaciones</Text>
                                        <Text>• Mejoran la gestión académica de tu institución</Text>
                                    </VStack>
                                </Box>
                            </VStack>
                        </Box>
                        <Button
                            leftIcon={<FiPlus />}
                            colorScheme="brand"
                            size="lg"
                            onClick={handleCreate}
                            isDisabled={isLimitReached}
                        >
                            Registrar Primer Profesor
                        </Button>
                        {isLimitReached && (
                            <Text color="red.500" fontSize="sm" fontWeight="bold">
                                Límite de profesores alcanzado. Por favor, comunícate con tu ejecutivo para ampliar tu plan.
                            </Text>
                        )}
                    </VStack>
                </Box>
            ) : (
                <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                    <Table variant="simple">
                        <Thead bg="gray.50">
                            <Tr>
                                <Th>Profesor</Th>
                                <Th>Especialidad</Th>
                                <Th>Contacto</Th>
                                <Th>Estado</Th>
                                <Th>Acciones</Th>
                            </Tr>
                        </Thead>
                        <Tbody>
                            {employees.map((employee) => (
                                <Tr key={employee.user_id}>
                                    <Td>
                                        <HStack>
                                            <Avatar
                                                size="sm"
                                                name={employee.name}
                                                src={employee.photo_url}
                                                bg="brand.100"
                                                color="brand.700"
                                            />
                                            <Box>
                                                <Text fontWeight="bold">{employee.name}</Text>
                                                <Text fontSize="xs" color="gray.500">ID: {employee.user_id}</Text>
                                            </Box>
                                        </HStack>
                                    </Td>
                                    <Td>
                                        <Badge colorScheme="purple">{employee.specialty}</Badge>
                                    </Td>
                                    <Td>
                                        <Text fontSize="sm">{employee.email}</Text>
                                        <Text fontSize="xs" color="gray.500">{formatPhone(employee.phone)}</Text>
                                    </Td>
                                    <Td>
                                        <Badge colorScheme={employee.is_active ? 'green' : 'red'}>
                                            {employee.is_active ? 'ACTIVO' : 'INACTIVO'}
                                        </Badge>
                                    </Td>
                                    <Td>
                                        <HStack spacing={2}>
                                            <IconButton
                                                icon={<FiEdit2 />}
                                                size="sm"
                                                aria-label="Editar"
                                                onClick={() => handleEdit(employee)}
                                            />
                                            <IconButton
                                                icon={<FiPower />}
                                                size="sm"
                                                colorScheme={employee.is_active ? 'red' : 'green'}
                                                aria-label="Cambiar estado"
                                                onClick={() => toggleStatus(employee.user_id)}
                                            />
                                        </HStack>
                                    </Td>
                                </Tr>
                            ))}
                        </Tbody>
                    </Table>
                </Box>
            )}

            {/* Modal */}
            <Modal isOpen={isOpen} onClose={onClose} size="xl">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentEmployee?.user_id ? 'Editar Profesor' : 'Nuevo Profesor'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <VStack spacing={4} align="stretch">
                            {/* Primera fila: Nombre(s) y Primer Apellido */}
                            <SimpleGrid columns={2} spacing={4}>
                                <FormControl isRequired>
                                    <FormLabel fontSize="sm">Nombre(s)</FormLabel>
                                    <Input
                                        size="sm"
                                        value={currentEmployee?.first_name || ''}
                                        onChange={(e) => setCurrentEmployee({ ...currentEmployee, first_name: e.target.value })}
                                        placeholder="Ej. María"
                                    />
                                </FormControl>

                                <FormControl isRequired>
                                    <FormLabel fontSize="sm">Primer Apellido</FormLabel>
                                    <Input
                                        size="sm"
                                        value={currentEmployee?.last_name || ''}
                                        onChange={(e) => setCurrentEmployee({ ...currentEmployee, last_name: e.target.value })}
                                        placeholder="Ej. García"
                                    />
                                </FormControl>
                            </SimpleGrid>

                            {/* Segunda fila: Segundo Apellido y Email */}
                            <SimpleGrid columns={2} spacing={4}>
                                <FormControl>
                                    <FormLabel fontSize="sm">Segundo Apellido (Opcional)</FormLabel>
                                    <Input
                                        size="sm"
                                        value={currentEmployee?.second_last_name || ''}
                                        onChange={(e) => setCurrentEmployee({ ...currentEmployee, second_last_name: e.target.value })}
                                        placeholder="Ej. López"
                                    />
                                </FormControl>

                                <FormControl isRequired>
                                    <FormLabel fontSize="sm">Email</FormLabel>
                                    <Input
                                        size="sm"
                                        type="email"
                                        value={currentEmployee?.email || ''}
                                        onChange={(e) => setCurrentEmployee({ ...currentEmployee, email: e.target.value })}
                                        onBlur={(e) => {
                                            if (e.target.value && !validateEmail(e.target.value)) {
                                                toast({ title: 'Email inválido', status: 'warning', duration: 2000 });
                                            }
                                        }}
                                    />
                                </FormControl>
                            </SimpleGrid>

                            {/* Segunda fila: Teléfono y Especialidad */}
                            <SimpleGrid columns={2} spacing={4}>
                                <FormControl isRequired>
                                    <FormLabel fontSize="sm">Teléfono</FormLabel>
                                    <Input
                                        size="sm"
                                        value={currentEmployee?.phone || ''}
                                        onChange={(e) => setCurrentEmployee({ ...currentEmployee, phone: formatPhone(e.target.value) })}
                                        onBlur={(e) => {
                                            if (e.target.value && !validatePhone(e.target.value)) {
                                                toast({ title: 'Teléfono incompleto', description: 'Debe tener 10 dígitos', status: 'warning', duration: 2000 });
                                            }
                                        }}
                                        placeholder="55 5555 5555"
                                    />
                                </FormControl>

                                <FormControl>
                                    <FormLabel fontSize="sm">Especialidad (Opcional)</FormLabel>
                                    {specialties.length > 0 ? (
                                        <Select
                                            size="sm"
                                            value={currentEmployee?.specialty || ''}
                                            onChange={(e) => setCurrentEmployee({ ...currentEmployee, specialty: e.target.value })}
                                        >
                                            <option value="">Seleccionar...</option>
                                            {specialties.map((spec) => (
                                                <option key={spec.id_specialty || spec.id} value={spec.name}>
                                                    {spec.name}
                                                </option>
                                            ))}
                                        </Select>
                                    ) : (
                                        <Box>
                                            <Text fontSize="xs" color="orange.600" fontWeight="medium">
                                                No hay especialidades registradas en el catálogo.
                                            </Text>
                                            <Text fontSize="xs" color="gray.500">
                                                Puede continuar sin asignar especialidad.
                                            </Text>
                                        </Box>
                                    )}
                                </FormControl>
                            </SimpleGrid>

                            {/* Tercera fila: CURP */}
                            <FormControl>
                                <FormLabel fontSize="sm">CURP (Opcional)</FormLabel>
                                <Input
                                    size="sm"
                                    value={currentEmployee?.curp || ''}
                                    onChange={(e) => setCurrentEmployee({ ...currentEmployee, curp: e.target.value.toUpperCase() })}
                                    onBlur={(e) => {
                                        if (e.target.value && !validateCURP(e.target.value)) {
                                            toast({ title: 'CURP inválido', status: 'warning', duration: 2000 });
                                        }
                                    }}
                                    placeholder="ABCD123456HXXXXX00"
                                    maxLength={18}
                                />
                                <Text fontSize="xs" color="gray.500" mt={1}>18 caracteres alfanuméricos</Text>
                            </FormControl>

                            {/* Cuarta fila: Fotografía y Contraseña en grid */}
                            <SimpleGrid columns={2} spacing={4}>
                                <FormControl>
                                    <FormLabel fontSize="sm">Fotografía (Opcional)</FormLabel>
                                    <VStack spacing={2} align="stretch">
                                        <HStack spacing={3}>
                                            <Avatar
                                                size="md"
                                                name={currentEmployee?.name}
                                                src={currentEmployee?._pendingFile ? URL.createObjectURL(currentEmployee._pendingFile) : currentEmployee?.photo_url}
                                                bg="brand.100"
                                                color="brand.700"
                                                border="2px solid"
                                                borderColor="brand.200"
                                            />
                                            <Box flex="1">
                                                <Box
                                                    position="relative"
                                                    height="80px"
                                                    border="2px dashed"
                                                    borderColor={isDragging ? "brand.500" : "gray.200"}
                                                    borderRadius="md"
                                                    bg={isDragging ? "brand.50" : "gray.50"}
                                                    transition="all 0.2s"
                                                    _hover={{ borderColor: "brand.400", bg: "gray.100" }}
                                                    onDragOver={(e) => { e.preventDefault(); setIsDragging(true); }}
                                                    onDragLeave={() => setIsDragging(false)}
                                                    onDrop={async (e) => {
                                                        e.preventDefault();
                                                        setIsDragging(false);
                                                        const file = e.dataTransfer.files[0];
                                                        if (file && file.type.startsWith('image/')) {
                                                            const processedFile = await processImage(file);
                                                            setCurrentEmployee({ ...currentEmployee, _pendingFile: processedFile });
                                                        }
                                                    }}
                                                >
                                                    <VStack
                                                        height="100%"
                                                        justify="center"
                                                        spacing={0}
                                                        cursor="pointer"
                                                        onClick={() => document.getElementById('photo-upload-input').click()}
                                                    >
                                                        <Icon as={FiUploadCloud} boxSize={5} color="brand.500" />
                                                        <Text fontSize="xs" color="gray.600">Clic para subir</Text>
                                                    </VStack>

                                                    <Input
                                                        id="photo-upload-input"
                                                        type="file"
                                                        display="none"
                                                        accept="image/*"
                                                        onChange={async (e) => {
                                                            const file = e.target.files[0];
                                                            if (file) {
                                                                const processedFile = await processImage(file);
                                                                setCurrentEmployee({ ...currentEmployee, _pendingFile: processedFile });
                                                            }
                                                        }}
                                                    />
                                                </Box>
                                            </Box>
                                        </HStack>
                                        <Text fontSize="xs" color="gray.500">
                                            JPG, PNG. 400x400px sugerido
                                        </Text>
                                    </VStack>
                                </FormControl>

                                <FormControl>
                                    <FormLabel fontSize="sm">
                                        Contraseña {currentEmployee?.user_id && '(Opcional)'}
                                    </FormLabel>
                                    <InputGroup size="sm">
                                        <Input
                                            type={showPassword ? "text" : "password"}
                                            placeholder={currentEmployee?.user_id ? '••••••••' : 'Ingresar contraseña'}
                                            value={currentEmployee?.password || ''}
                                            onChange={(e) => setCurrentEmployee({ ...currentEmployee, password: e.target.value })}
                                        />
                                        <InputRightElement h="full" width="3rem">
                                            <Button
                                                variant="ghost"
                                                size="sm"
                                                h="1.75rem"
                                                onMouseDown={() => setShowPassword(true)}
                                                onMouseUp={() => setShowPassword(false)}
                                                onMouseLeave={() => setShowPassword(false)}
                                                onTouchStart={() => setShowPassword(true)}
                                                onTouchEnd={() => setShowPassword(false)}
                                            >
                                                {showPassword ? <FiEyeOff size={16} /> : <FiEye size={16} />}
                                            </Button>
                                        </InputRightElement>
                                    </InputGroup>
                                    {!currentEmployee?.user_id && (
                                        <Text fontSize="xs" color="gray.500" mt={1}>
                                            Requerida para nuevos profesores
                                        </Text>
                                    )}
                                </FormControl>
                            </SimpleGrid>
                        </VStack>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionEmpleados;
