import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Textarea,
    useToast,
    HStack,
    VStack,
    Text,
    Icon
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiPower, FiBookOpen } from 'react-icons/fi';
import { getSpecialties, createSpecialty, updateSpecialty, deleteSpecialty } from '../../services/api';

const GestionEspecialidades = () => {
    const [specialties, setSpecialties] = useState([]);
    const [currentSpecialty, setCurrentSpecialty] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    const fetchSpecialties = async () => {
        try {
            const data = await getSpecialties();
            setSpecialties(data || []);
        } catch (error) {
            console.error('Error fetching specialties:', error);
            toast({
                title: 'Error al cargar especialidades',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchSpecialties();
    }, []);

    const handleCreate = () => {
        setCurrentSpecialty({
            name: '',
            description: '',
            is_active: true
        });
        onOpen();
    };

    const handleEdit = (specialty) => {
        setCurrentSpecialty({ ...specialty });
        onOpen();
    };

    const handleToggleStatus = async (specialty) => {
        try {
            const newStatus = !specialty.is_active;
            await updateSpecialty(specialty.id_specialty || specialty.id, { 
                ...specialty, 
                is_active: newStatus 
            });
            
            setSpecialties(specialties.map(s =>
                (s.id_specialty || s.id) === (specialty.id_specialty || specialty.id) 
                    ? { ...s, is_active: newStatus } 
                    : s
            ));

            toast({
                title: `Especialidad ${newStatus ? 'activada' : 'desactivada'}`,
                status: 'success',
                duration: 2000,
            });
        } catch (error) {
            toast({
                title: 'Error al cambiar estado',
                description: error.message || 'Error desconocido',
                status: 'error',
                duration: 3000,
            });
        }
    };

    const handleDelete = async (specialty) => {
        if (!window.confirm(`¿Estás seguro de eliminar la especialidad "${specialty.name}"?`)) {
            return;
        }

        try {
            await deleteSpecialty(specialty.id_specialty || specialty.id);
            setSpecialties(specialties.filter(s => 
                (s.id_specialty || s.id) !== (specialty.id_specialty || specialty.id)
            ));
            toast({
                title: 'Especialidad eliminada',
                status: 'success',
                duration: 2000,
            });
        } catch (error) {
            toast({
                title: 'Error al eliminar',
                description: error.message || 'No se pudo eliminar la especialidad. Puede estar siendo utilizada por uno o más profesores.',
                status: 'error',
                duration: 5000,
            });
        }
    };

    const handleSave = async () => {
        try {
            if (!currentSpecialty.name || !currentSpecialty.name.trim()) {
                toast({ 
                    title: 'El nombre es requerido', 
                    status: 'warning' 
                });
                return;
            }

            if (currentSpecialty.id_specialty || currentSpecialty.id) {
                await updateSpecialty(currentSpecialty.id_specialty || currentSpecialty.id, currentSpecialty);
                toast({ title: 'Especialidad actualizada', status: 'success' });
            } else {
                await createSpecialty(currentSpecialty);
                toast({ title: 'Especialidad creada', status: 'success' });
            }
            fetchSpecialties();
            onClose();
        } catch (error) {
            toast({ 
                title: 'Error al guardar', 
                description: error.message || 'Error desconocido',
                status: 'error',
                duration: 5000
            });
        }
    };

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <Heading size="lg">Gestión de Especialidades</Heading>
                <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                    Nueva Especialidad
                </Button>
            </HStack>

            {specialties.length === 0 ? (
                <Box 
                    bg="white" 
                    borderRadius="lg" 
                    boxShadow="sm" 
                    p={12}
                    textAlign="center"
                >
                    <VStack spacing={6}>
                        <Box>
                            <Icon 
                                as={FiBookOpen} 
                                boxSize={16} 
                                color="gray.300" 
                                mb={4}
                            />
                            <Heading size="lg" color="gray.600" mb={2}>
                                No hay especialidades registradas
                            </Heading>
                            <VStack spacing={3} align="stretch" maxW="2xl" mx="auto" mt={4}>
                                <Text color="gray.600" fontSize="md" fontWeight="500">
                                    Las especialidades permiten organizar y categorizar a los profesores por su área de conocimiento.
                                </Text>
                                <Box 
                                    bg="blue.50" 
                                    p={4} 
                                    borderRadius="md" 
                                    borderLeft="4px solid" 
                                    borderColor="blue.500"
                                    textAlign="left"
                                >
                                    <Text color="blue.900" fontSize="sm" fontWeight="600" mb={2}>
                                        ¿Por qué es importante tener especialidades?
                                    </Text>
                                    <VStack align="start" spacing={2} fontSize="sm" color="blue.800">
                                        <Text>• Permite organizar a los profesores por área de conocimiento</Text>
                                        <Text>• Facilita la búsqueda y filtrado de profesores</Text>
                                        <Text>• Ayuda a asignar profesores adecuados a materias específicas</Text>
                                        <Text>• Mejora la organización del contenido académico</Text>
                                    </VStack>
                                </Box>
                            </VStack>
                        </Box>
                        <Button
                            leftIcon={<FiPlus />}
                            colorScheme="brand"
                            size="lg"
                            onClick={handleCreate}
                        >
                            Crear Primera Especialidad
                        </Button>
                    </VStack>
                </Box>
            ) : (
                <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                    <Table variant="simple">
                        <Thead bg="gray.50">
                            <Tr>
                                <Th>Nombre</Th>
                                <Th>Descripción</Th>
                                <Th>Estado</Th>
                                <Th>Acciones</Th>
                            </Tr>
                        </Thead>
                        <Tbody>
                            {specialties.map((specialty) => (
                                <Tr key={specialty.id_specialty || specialty.id}>
                                    <Td>
                                        <Text fontWeight="medium">
                                            {specialty.name}
                                        </Text>
                                    </Td>
                                    <Td>
                                        <Text fontSize="sm" color="gray.600" noOfLines={2}>
                                            {specialty.description || 'Sin descripción'}
                                        </Text>
                                    </Td>
                                    <Td>
                                        <Badge colorScheme={specialty.is_active ? 'green' : 'red'}>
                                            {specialty.is_active ? 'ACTIVA' : 'INACTIVA'}
                                        </Badge>
                                    </Td>
                                    <Td>
                                        <HStack spacing={2}>
                                            <IconButton
                                                icon={<FiEdit2 />}
                                                size="sm"
                                                onClick={() => handleEdit(specialty)}
                                                aria-label="Editar"
                                                colorScheme="blue"
                                                variant="ghost"
                                            />
                                            <IconButton
                                                icon={<FiPower />}
                                                size="sm"
                                                onClick={() => handleToggleStatus(specialty)}
                                                aria-label={specialty.is_active ? 'Desactivar' : 'Activar'}
                                                colorScheme={specialty.is_active ? 'red' : 'green'}
                                                variant="ghost"
                                            />
                                            <IconButton
                                                icon={<FiTrash2 />}
                                                size="sm"
                                                onClick={() => handleDelete(specialty)}
                                                aria-label="Eliminar"
                                                colorScheme="red"
                                                variant="ghost"
                                            />
                                        </HStack>
                                    </Td>
                                </Tr>
                            ))}
                        </Tbody>
                    </Table>
                </Box>
            )}

            <Modal isOpen={isOpen} onClose={onClose} size="md">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>
                        {currentSpecialty?.id_specialty || currentSpecialty?.id 
                            ? 'Editar Especialidad' 
                            : 'Nueva Especialidad'}
                    </ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <VStack spacing={4}>
                            <FormControl isRequired>
                                <FormLabel>Nombre</FormLabel>
                                <Input 
                                    value={currentSpecialty?.name || ''}
                                    onChange={(e) => setCurrentSpecialty({
                                        ...currentSpecialty, 
                                        name: e.target.value
                                    })}
                                    placeholder="Ej: Matemáticas, Física, Química..."
                                />
                            </FormControl>
                            <FormControl>
                                <FormLabel>Descripción</FormLabel>
                                <Textarea 
                                    value={currentSpecialty?.description || ''}
                                    onChange={(e) => setCurrentSpecialty({
                                        ...currentSpecialty, 
                                        description: e.target.value
                                    })}
                                    placeholder="Descripción opcional de la especialidad..."
                                    rows={4}
                                />
                            </FormControl>
                        </VStack>
                    </ModalBody>
                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionEspecialidades;


