import { useState } from 'react';
import {
  Box,
  Heading,
  Button,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  TableContainer,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  useDisclosure,
  FormControl,
  FormLabel,
  Input,
  Textarea,
  useToast,
  Flex,
  HStack,
  IconButton,
  Icon,
  Text,
  Select
} from '@chakra-ui/react';
import { BiPencil, BiTrash } from 'react-icons/bi';
import Layout from '../../components/Layout';
import { adminLinks } from './adminLinks';
import { mockMaterias } from '../../data/mockData';

const GestionMaterias = () => {
  const [materias, setMaterias] = useState(mockMaterias);
  const [materiaEditando, setMateriaEditando] = useState(null);
  const { isOpen, onOpen, onClose } = useDisclosure();
  const { isOpen: isEditOpen, onOpen: onEditOpen, onClose: onEditClose } = useDisclosure();
  const toast = useToast();

  // Estado para el selector de periodo
  const [startDay, setStartDay] = useState(1);
  const [endDay, setEndDay] = useState(5);

  const handleCrear = () => {
    setMateriaEditando(null);
    setStartDay(1);
    setEndDay(5);
    onOpen();
  };

  const handleEditar = (materia) => {
    setMateriaEditando(materia);
    // Intentar parsear "1-5 de cada mes"
    const match = materia.periodo_pago ? materia.periodo_pago.match(/(\d+)-(\d+)/) : null;
    if (match) {
      setStartDay(parseInt(match[1]));
      setEndDay(parseInt(match[2]));
    } else {
      setStartDay(1);
      setEndDay(5);
    }
    onEditOpen();
  };

  const handleDesactivar = (id) => {
    toast({
      title: 'Materia desactivada',
      description: 'La materia ha sido marcada como inactiva',
      status: 'info',
      duration: 2000,
      isClosable: true,
    });
  };

  const handleGuardar = (e) => {
    e.preventDefault();
    const form = e.target;
    const nombre = form.elements[0].value; // Asumiendo orden de inputs
    const descripcion = form.elements[1].value;
    const precio = form.elements[2].value;
    const periodo_pago = `${startDay}-${endDay} de cada mes`;

    const nuevaMateria = {
      id: materiaEditando ? materiaEditando.id : Date.now(),
      nombre,
      descripcion,
      precio,
      periodo_pago
    };

    if (materiaEditando) {
      setMaterias(materias.map(m => m.id === materiaEditando.id ? nuevaMateria : m));
    } else {
      setMaterias([...materias, nuevaMateria]);
    }

    toast({
      title: 'Materia guardada',
      description: materiaEditando ? 'Materia actualizada' : 'Materia creada',
      status: 'success',
      duration: 2000,
      isClosable: true,
    });
    onClose();
    onEditClose();
  };

  const days = Array.from({ length: 31 }, (_, i) => i + 1);

  return (
    <Layout links={adminLinks} userType="Administrador" userName="Admin User">
      <Box>

        {/* Instrucciones minimalistas */}
        <Text fontSize="sm" color="gray.600" mb={6}>
          Crea, edita y administra las materias de la escuela. <strong>Las materias desactivadas no estarán disponibles</strong> para asignar nuevas clases.
        </Text>

        {/* Tarjeta con estilo Facit */}
        <Box
          bg="rgba(255, 255, 255, 0.6)"
          backdropFilter="blur(20px) saturate(180%)"
          WebkitBackdropFilter="blur(20px) saturate(180%)"
          borderRadius="xl"
          boxShadow="0 8px 32px 0 rgba(0, 0, 0, 0.1)"
          border="1px solid"
          borderColor="rgba(255, 255, 255, 0.3)"
          overflow="hidden"
        >
          {/* Header de la Tarjeta */}
          <Flex justify="space-between" align="center" p={5} borderBottom="1px" borderColor="rgba(255, 255, 255, 0.2)" direction={{ base: 'column', sm: 'row' }} gap={4}>
            <Heading as="h3" size="md">Lista de Materias</Heading>
            <Button colorScheme="blue" borderRadius="full" onClick={handleCrear} w={{ base: '100%', sm: 'auto' }}>Crear Nuevo</Button>
          </Flex>

          {/* Cuerpo de la Tabla */}
          <TableContainer>
            <Table variant="simple">
              <Thead>
                <Tr>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">ID</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Nombre</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Descripción</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Precio</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Periodo Pago</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Acciones</Th>
                </Tr>
              </Thead>
              <Tbody>
                {materias.map((materia) => (
                  <Tr key={materia.id} _hover={{ bg: "gray.50" }}>
                    <Td verticalAlign="middle">{materia.id}</Td>
                    <Td verticalAlign="middle">{materia.nombre}</Td>
                    <Td verticalAlign="middle">{materia.descripcion}</Td>
                    <Td verticalAlign="middle">${materia.precio}</Td>
                    <Td verticalAlign="middle">{materia.periodo_pago}</Td>
                    <Td verticalAlign="middle">
                      <HStack spacing={2}>
                        <IconButton
                          icon={<Icon as={BiPencil} />}
                          variant="ghost"
                          colorScheme="green"
                          aria-label="Editar"
                          onClick={() => handleEditar(materia)}
                        />
                        <IconButton
                          icon={<Icon as={BiTrash} />}
                          variant="ghost"
                          colorScheme="red"
                          aria-label="Desactivar"
                          onClick={() => handleDesactivar(materia.id)}
                        />
                      </HStack>
                    </Td>
                  </Tr>
                ))}
              </Tbody>
            </Table>
          </TableContainer>
        </Box>

        <Modal isOpen={isOpen} onClose={onClose} isCentered>
          <ModalOverlay bg="blackAlpha.600" backdropFilter="blur(4px)" />
          <ModalContent
            borderRadius="xl"
            bg="rgba(255, 255, 255, 0.8)"
            backdropFilter="blur(20px) saturate(180%)"
            WebkitBackdropFilter="blur(20px) saturate(180%)"
            border="1px solid"
            borderColor="rgba(255, 255, 255, 0.3)"
            boxShadow="0 8px 32px 0 rgba(0, 0, 0, 0.2)"
          >
            <ModalHeader>Crear Nueva Materia</ModalHeader>
            <ModalCloseButton />
            <form onSubmit={handleGuardar}>
              <ModalBody>
                <FormControl mb={4}>
                  <FormLabel>Nombre</FormLabel>
                  <Input type="text" required />
                </FormControl>
                <FormControl mb={4}>
                  <FormLabel>Descripción</FormLabel>
                  <Textarea required />
                </FormControl>
                <FormControl mb={4}>
                  <FormLabel>Precio</FormLabel>
                  <Input type="number" placeholder="0.00" required />
                </FormControl>
                <FormControl mb={4}>
                  <FormLabel>Periodo de Pago (Día del mes)</FormLabel>
                  <HStack>
                    <Select value={startDay} onChange={(e) => setStartDay(e.target.value)}>
                      {days.map(d => <option key={d} value={d}>{d}</option>)}
                    </Select>
                    <Text>al</Text>
                    <Select value={endDay} onChange={(e) => setEndDay(e.target.value)}>
                      {days.map(d => <option key={d} value={d}>{d}</option>)}
                    </Select>
                  </HStack>
                </FormControl>
              </ModalBody>
              <ModalFooter>
                <Button variant="ghost" mr={3} onClick={onClose}>Cancelar</Button>
                <Button colorScheme="brand" type="submit">Guardar</Button>
              </ModalFooter>
            </form>
          </ModalContent>
        </Modal>

        <Modal isOpen={isEditOpen} onClose={onEditClose}>
          <ModalOverlay />
          <ModalContent>
            <ModalHeader>Editar Materia</ModalHeader>
            <ModalCloseButton />
            <form onSubmit={handleGuardar}>
              <ModalBody>
                <FormControl mb={4}>
                  <FormLabel>Nombre</FormLabel>
                  <Input type="text" defaultValue={materiaEditando?.nombre} required />
                </FormControl>
                <FormControl mb={4}>
                  <FormLabel>Descripción</FormLabel>
                  <Textarea defaultValue={materiaEditando?.descripcion} required />
                </FormControl>
                <FormControl mb={4}>
                  <FormLabel>Precio</FormLabel>
                  <Input type="number" defaultValue={materiaEditando?.precio} required />
                </FormControl>
                <FormControl mb={4}>
                  <FormLabel>Periodo de Pago (Día del mes)</FormLabel>
                  <HStack>
                    <Select value={startDay} onChange={(e) => setStartDay(e.target.value)}>
                      {days.map(d => <option key={d} value={d}>{d}</option>)}
                    </Select>
                    <Text>al</Text>
                    <Select value={endDay} onChange={(e) => setEndDay(e.target.value)}>
                      {days.map(d => <option key={d} value={d}>{d}</option>)}
                    </Select>
                  </HStack>
                </FormControl>
              </ModalBody>
              <ModalFooter>
                <Button variant="ghost" mr={3} onClick={onEditClose}>Cancelar</Button>
                <Button colorScheme="brand" type="submit">Guardar</Button>
              </ModalFooter>
            </form>
          </ModalContent>
        </Modal>
      </Box>
    </Layout>
  );
};

export default GestionMaterias;

