import { useState } from 'react';
import {
  Box,
  Heading,
  Tabs,
  TabList,
  TabPanels,
  Tab,
  TabPanel,
  FormControl,
  FormLabel,
  Textarea,
  Input,
  Select,
  Button,
  useToast,
  Text,
  VStack,
  HStack,
  Switch,
  Grid,
  GridItem,
  Card,
  CardBody,
  Container,
  Icon,
  useColorModeValue,
  Stack
} from '@chakra-ui/react';
import { FaPaperPlane, FaHistory, FaEdit, FaEye } from 'react-icons/fa';
import Layout from '../../components/Layout';
import { adminLinks } from './adminLinks';
import NotificacionCard from '../../components/NotificacionCard';
import RichTextEditor from '../../components/RichTextEditor';
import NotificacionesHistorial from './NotificacionesHistorial';
import RecipientSelector from '../../components/RecipientSelector';

const tiposNotificacion = [
  { value: 'default', label: 'General', icon: '🔔' },
  { value: 'promocion', label: 'Promoción', icon: '🎉' },
  { value: 'curso', label: 'Curso', icon: '📚' },
  { value: 'pago', label: 'Pago', icon: '💳' },
  { value: 'recordatorio', label: 'Recordatorio', icon: '⏰' },
  { value: 'bitacora', label: 'Bitácora', icon: '📝' },
  { value: 'meta', label: 'Meta', icon: '🎯' },
  { value: 'alumno', label: 'Alumno', icon: '👤' },
  { value: 'clase', label: 'Clase', icon: '🎨' }
];

const NotificacionesMasivas = () => {
  // Estados para cada tipo de usuario
  const [formDataProfesores, setFormDataProfesores] = useState({
    titulo: '',
    tipo: 'default',
    mensaje: '',
    html: '',
    imagen: '',
    usarHtml: true,
    modoEnvio: 'masivo', // 'masivo' | 'particular' | 'filtro'
    destinatarios: [],
    filtros: { materia: '', semana: '', soloActivos: true }
  });

  const [formDataPadres, setFormDataPadres] = useState({
    titulo: '',
    tipo: 'default',
    mensaje: '',
    html: '',
    imagen: '',
    usarHtml: true,
    modoEnvio: 'masivo',
    destinatarios: [],
    filtros: { materia: '', profesor: '', semana: '', soloActivos: true }
  });

  const [formDataAlumnos, setFormDataAlumnos] = useState({
    titulo: '',
    tipo: 'default',
    mensaje: '',
    html: '',
    imagen: '',
    usarHtml: true,
    modoEnvio: 'masivo',
    destinatarios: [],
    filtros: { materia: '', profesor: '', semana: '', soloActivos: true }
  });

  const toast = useToast();
  const bgColor = useColorModeValue('rgba(255, 255, 255, 0.8)', 'rgba(26, 32, 44, 0.8)');
  const borderColor = useColorModeValue('gray.200', 'gray.700');

  const actualizarFormData = (tipoUsuario, campo, valor) => {
    if (tipoUsuario === 'profesores') {
      setFormDataProfesores(prev => ({ ...prev, [campo]: valor }));
    } else if (tipoUsuario === 'padres') {
      setFormDataPadres(prev => ({ ...prev, [campo]: valor }));
    } else {
      setFormDataAlumnos(prev => ({ ...prev, [campo]: valor }));
    }
  };

  const obtenerFormData = (tipoUsuario) => {
    if (tipoUsuario === 'profesores') return formDataProfesores;
    if (tipoUsuario === 'padres') return formDataPadres;
    return formDataAlumnos;
  };

  const limpiarFormulario = (tipoUsuario) => {
    const datosVacios = {
      titulo: '',
      tipo: 'default',
      mensaje: '',
      html: '',
      imagen: '',
      usarHtml: true,
      modoEnvio: 'masivo',
      destinatarios: [],
      filtros: { materia: '', profesor: '', semana: '', soloActivos: true }
    };

    if (tipoUsuario === 'profesores') {
      setFormDataProfesores(datosVacios);
    } else if (tipoUsuario === 'padres') {
      setFormDataPadres(datosVacios);
    } else {
      setFormDataAlumnos(datosVacios);
    }
  };

  const handleEnviar = (tipoUsuario) => {
    const formData = obtenerFormData(tipoUsuario);

    if (!formData.titulo.trim()) {
      toast({
        title: 'Error',
        description: 'Por favor ingresa un título',
        status: 'error',
        duration: 2000,
        isClosable: true,
      });
      return;
    }

    if (formData.usarHtml && !formData.html.trim()) {
      toast({
        title: 'Error',
        description: 'Por favor ingresa el contenido del mensaje',
        status: 'error',
        duration: 2000,
        isClosable: true,
      });
      return;
    }

    if (!formData.usarHtml && !formData.mensaje.trim()) {
      toast({
        title: 'Error',
        description: 'Por favor ingresa un mensaje',
        status: 'error',
        duration: 2000,
        isClosable: true,
      });
      return;
    }

    if (formData.modoEnvio === 'particular' && formData.destinatarios.length === 0) {
      toast({
        title: 'Error',
        description: 'Debes seleccionar al menos un destinatario para el envío particular',
        status: 'error',
        duration: 3000,
        isClosable: true,
      });
      return;
    }

    if (formData.modoEnvio === 'filtro') {
      // Validación opcional: verificar si se seleccionó algún filtro
      // Por ahora permitimos enviar con filtros vacíos (implica todos los del tipo)
    }

    // Construir objeto de destinatarios para el backend
    let destinatariosInfo = 'todos';
    if (formData.modoEnvio === 'particular') {
      destinatariosInfo = formData.destinatarios.map(d => d.id);
    } else if (formData.modoEnvio === 'filtro') {
      destinatariosInfo = formData.filtros;
    }

    // Aquí se enviaría la notificación al backend
    const notificacion = {
      titulo: formData.titulo,
      tipo: formData.tipo,
      mensaje: formData.usarHtml ? undefined : formData.mensaje,
      html: formData.usarHtml ? formData.html : undefined,
      imagen: formData.imagen || undefined,
      fecha: new Date().toISOString().split('T')[0],
      leida: false,
      modoEnvio: formData.modoEnvio,
      destinatarios: destinatariosInfo
    };

    console.log('Enviando notificación a', tipoUsuario, ':', notificacion);

    let desc = `Se ha enviado la notificación a todos los ${tipoUsuario}`;
    if (formData.modoEnvio === 'particular') {
      desc = `Se ha enviado la notificación a ${formData.destinatarios.length} destinatarios`;
    } else if (formData.modoEnvio === 'filtro') {
      desc = `Se ha enviado la notificación a los ${tipoUsuario} que coinciden con los filtros`;
    }

    toast({
      title: 'Notificación enviada',
      description: desc,
      status: 'success',
      duration: 3000,
      isClosable: true,
    });

    limpiarFormulario(tipoUsuario);
  };

  const obtenerVistaPrevia = (tipoUsuario) => {
    const formData = obtenerFormData(tipoUsuario);

    if (!formData.titulo) return null;

    return {
      id: 'preview',
      titulo: formData.titulo,
      tipo: formData.tipo,
      mensaje: formData.usarHtml ? undefined : formData.mensaje || 'Mensaje de ejemplo...',
      html: formData.usarHtml ? formData.html : undefined,
      imagen: formData.imagen || undefined,
      fecha: new Date().toISOString().split('T')[0],
      leida: false
    };
  };

  const renderFormulario = (tipoUsuario) => {
    const formData = obtenerFormData(tipoUsuario);
    const vistaPrevia = obtenerVistaPrevia(tipoUsuario);

    return (
      <Grid templateColumns={{ base: '1fr', xl: '1.2fr 0.8fr' }} gap={8}>
        <GridItem>
          <VStack align="stretch" spacing={5}>
            {/* Selector de Destinatarios */}
            <Box bg="gray.50" p={4} borderRadius="md" border="1px solid" borderColor="gray.200">
              <RecipientSelector
                tipoUsuario={tipoUsuario}
                mode={formData.modoEnvio}
                onModeChange={(val) => actualizarFormData(tipoUsuario, 'modoEnvio', val)}
                selectedRecipients={formData.destinatarios}
                onSelectionChange={(val) => actualizarFormData(tipoUsuario, 'destinatarios', val)}
                filters={formData.filtros}
                onFiltersChange={(val) => actualizarFormData(tipoUsuario, 'filtros', val)}
              />
            </Box>

            {/* Título */}
            <FormControl isRequired>
              <FormLabel fontWeight="600" color="gray.600">Título de la Notificación</FormLabel>
              <Input
                value={formData.titulo}
                onChange={(e) => actualizarFormData(tipoUsuario, 'titulo', e.target.value)}
                placeholder="Ej: 🎉 Promoción Especial: Descuento en Materiales"
                size="lg"
                bg="white"
              />
            </FormControl>

            <Stack direction={{ base: 'column', md: 'row' }} spacing={4} align="start">
              {/* Tipo de Notificación */}
              <FormControl flex={1}>
                <FormLabel fontWeight="600" color="gray.600">Tipo</FormLabel>
                <Select
                  value={formData.tipo}
                  onChange={(e) => actualizarFormData(tipoUsuario, 'tipo', e.target.value)}
                  size="lg"
                  bg="white"
                >
                  {tiposNotificacion.map(tipo => (
                    <option key={tipo.value} value={tipo.value}>
                      {tipo.icon} {tipo.label}
                    </option>
                  ))}
                </Select>
              </FormControl>

              {/* URL de Imagen */}
              <FormControl flex={1}>
                <FormLabel fontWeight="600" color="gray.600">Imagen (Opcional)</FormLabel>
                <Input
                  value={formData.imagen}
                  onChange={(e) => actualizarFormData(tipoUsuario, 'imagen', e.target.value)}
                  placeholder="https://..."
                  type="url"
                  size="lg"
                  bg="white"
                />
              </FormControl>
            </Stack>

            {/* Switch para HTML */}
            <FormControl display="flex" alignItems="center" justifyContent="space-between" bg="gray.50" p={3} borderRadius="md">
              <FormLabel mb={0} fontWeight="600" color="gray.600">
                Modo de Edición
              </FormLabel>
              <HStack>
                <Text fontSize="sm" color={!formData.usarHtml ? 'brand.500' : 'gray.500'} fontWeight={!formData.usarHtml ? 'bold' : 'normal'}>Texto Simple</Text>
                <Switch
                  isChecked={formData.usarHtml}
                  onChange={(e) => actualizarFormData(tipoUsuario, 'usarHtml', e.target.checked)}
                  colorScheme="brand"
                  size="lg"
                />
                <Text fontSize="sm" color={formData.usarHtml ? 'brand.500' : 'gray.500'} fontWeight={formData.usarHtml ? 'bold' : 'normal'}>Editor Rico (HTML)</Text>
              </HStack>
            </FormControl>

            {/* Mensaje o HTML */}
            <FormControl isRequired>
              <FormLabel fontWeight="600" color="gray.600">Contenido del Mensaje</FormLabel>
              {formData.usarHtml ? (
                <RichTextEditor
                  value={formData.html}
                  onChange={(val) => actualizarFormData(tipoUsuario, 'html', val)}
                  placeholder="Escribe tu mensaje aquí. Puedes usar negritas, listas, y más..."
                />
              ) : (
                <Textarea
                  value={formData.mensaje}
                  onChange={(e) => actualizarFormData(tipoUsuario, 'mensaje', e.target.value)}
                  placeholder="Escribe el mensaje que deseas enviar..."
                  rows={8}
                  bg="white"
                  p={4}
                />
              )}
            </FormControl>

            {/* Botones */}
            <Stack direction={{ base: 'column', md: 'row' }} spacing={4} pt={4}>
              <Button
                colorScheme="brand"
                onClick={() => handleEnviar(tipoUsuario)}
                size="lg"
                flex={1}
                leftIcon={<FaPaperPlane />}
                boxShadow="md"
                _hover={{ transform: 'translateY(-2px)', boxShadow: 'lg' }}
              >
                Enviar Notificación
              </Button>
              <Button
                variant="outline"
                onClick={() => limpiarFormulario(tipoUsuario)}
                size="lg"
                colorScheme="gray"
                w={{ base: '100%', md: 'auto' }}
              >
                Limpiar
              </Button>
            </Stack>
          </VStack>
        </GridItem>

        <GridItem>
          <Box position="sticky" top="100px">
            <Heading size="sm" mb={4} color="gray.600" display="flex" alignItems="center">
              <Icon as={FaEye} mr={2} /> Vista Previa en Vivo
            </Heading>
            <Card
              border="1px solid"
              borderColor={borderColor}
              bg={bgColor}
              backdropFilter="blur(20px)"
              boxShadow="xl"
              borderRadius="xl"
              overflow="hidden"
            >
              <Box bg="brand.500" h="6px" w="100%" />
              <CardBody p={6}>
                {vistaPrevia ? (
                  <NotificacionCard notificacion={vistaPrevia} />
                ) : (
                  <Box
                    p={10}
                    textAlign="center"
                    color="gray.400"
                    border="2px dashed"
                    borderColor="gray.200"
                    borderRadius="xl"
                    bg="whiteAlpha.500"
                  >
                    <Icon as={FaEdit} w={10} h={10} mb={3} color="gray.300" />
                    <Text fontSize="lg" fontWeight="500">
                      Comienza a escribir
                    </Text>
                    <Text fontSize="sm">
                      La vista previa se actualizará automáticamente
                    </Text>
                  </Box>
                )}
              </CardBody>
            </Card>
          </Box>
        </GridItem>
      </Grid>
    );
  };

  return (
    <Layout links={adminLinks} userType="Administrador" userName="Admin User">
      <Box maxW="7xl" mx="auto">
        <Heading size="lg" mb={2} color="brand.500">Centro de Notificaciones</Heading>
        <Text fontSize="md" color="gray.600" mb={8}>
          Gestiona la comunicación con profesores, padres y alumnos.
        </Text>

        <Tabs variant="soft-rounded" colorScheme="brand" isLazy>
          <TabList mb={6} bg="white" p={2} borderRadius="full" boxShadow="sm" display="inline-flex">
            <Tab _selected={{ bg: 'brand.500', color: 'white' }} px={6} py={3}>
              <HStack><Icon as={FaEdit} /> <Text>Redactar Nueva</Text></HStack>
            </Tab>
            <Tab _selected={{ bg: 'brand.500', color: 'white' }} px={6} py={3}>
              <HStack><Icon as={FaHistory} /> <Text>Historial y Estadísticas</Text></HStack>
            </Tab>
          </TabList>

          <TabPanels>
            {/* Panel de Redacción */}
            <TabPanel px={0}>
              <Box
                bg="white"
                borderRadius="2xl"
                boxShadow="sm"
                border="1px solid"
                borderColor="gray.100"
                overflow="hidden"
              >
                <Tabs isFitted variant="enclosed">
                  <TabList mb={0} bg="gray.50" borderBottom="1px solid" borderColor="gray.200">
                    <Tab _selected={{ color: 'brand.600', bg: 'white', borderTop: '3px solid', borderTopColor: 'brand.500' }} fontWeight="600" py={4}>
                      👨‍🏫 Profesores
                    </Tab>
                    <Tab _selected={{ color: 'brand.600', bg: 'white', borderTop: '3px solid', borderTopColor: 'brand.500' }} fontWeight="600" py={4}>
                      👨‍👩‍👧 Padres/Tutores
                    </Tab>
                    <Tab _selected={{ color: 'brand.600', bg: 'white', borderTop: '3px solid', borderTopColor: 'brand.500' }} fontWeight="600" py={4}>
                      🎓 Alumnos
                    </Tab>
                  </TabList>

                  <TabPanels>
                    <TabPanel p={8}>
                      {renderFormulario('profesores')}
                    </TabPanel>
                    <TabPanel p={8}>
                      {renderFormulario('padres')}
                    </TabPanel>
                    <TabPanel p={8}>
                      {renderFormulario('alumnos')}
                    </TabPanel>
                  </TabPanels>
                </Tabs>
              </Box>
            </TabPanel>

            {/* Panel de Historial */}
            <TabPanel px={0}>
              <NotificacionesHistorial />
            </TabPanel>
          </TabPanels>
        </Tabs>
      </Box>
    </Layout>
  );
};

export default NotificacionesMasivas;
