import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Select,
    useToast,
    HStack,
    Text,
    VStack,
    SimpleGrid,
    Image,
    Tabs,
    TabList,
    Tab,
    TabPanel,
    TabPanels,
    Divider,
} from '@chakra-ui/react';
import { FiEdit2, FiPlus, FiEye, FiStar, FiDollarSign } from 'react-icons/fi';
import { ChevronLeftIcon, ChevronRightIcon } from '@chakra-ui/icons';
import { getOrders, getClients, getEmployees, getServicesByCompany, createOrder, updateOrder, getEvidencesByService } from '../../services/api';

const OrdenesServicio = () => {
    const [orders, setOrders] = useState([]);
    const [clients, setClients] = useState([]);
    const [employees, setEmployees] = useState([]);
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const [currentOrder, setCurrentOrder] = useState(null);
    const [isAdmin, setIsAdmin] = useState(false);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    // Disclosure for View Modal
    const { isOpen: isViewOpen, onOpen: onViewOpen, onClose: onViewClose } = useDisclosure();
    const [selectedOrderForView, setSelectedOrderForView] = useState(null);
    const [evidences, setEvidences] = useState([]);

    // Disclosure for Payment Review Modal
    const { isOpen: isPaymentOpen, onOpen: onPaymentOpen, onClose: onPaymentClose } = useDisclosure();
    const [selectedOrderForPayment, setSelectedOrderForPayment] = useState(null);

    // Función para traducir estados a leyendas amigables
    const getStatusLabel = (status) => {
        const statusMap = {
            'PENDING': 'Pendiente',
            'IN_PROGRESS': 'En Progreso',
            'FINISHED': 'Finalizado',
            'CANCELLED': 'Cancelado',
            'CERRADO_ADMIN': 'Cerrado',
            'IN_REVIEW': 'Reapertura'
        };
        return statusMap[status] || status;
    };

    const fetchData = async () => {
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            const companyId = user ? user.company_id : null;

            // Verificar si el usuario es administrador
            setIsAdmin(user?.role === 'admin');

            if (companyId) {
                const [allOrders, allClients, allEmployees, companyServices] = await Promise.all([
                    getOrders(),
                    getClients(),
                    getEmployees(),
                    getServicesByCompany(companyId)
                ]);

                setOrders(allOrders.filter(o => o.company_id === companyId).sort((a, b) => b.id - a.id));
                setClients(allClients.filter(c => c.company_id === companyId));
                setEmployees(allEmployees.filter(e => e.company_id === companyId));
                setServicesCatalog(companyServices);
            }
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar datos',
                status: 'error',
                duration: 3000,
            });
        }
    };

    const [viewMode, setViewMode] = useState('list'); // 'list' | 'calendar'
    const [currentDate, setCurrentDate] = useState(new Date());

    useEffect(() => {
        fetchData();
    }, []);

    const getClientName = (id) => clients.find(c => c.user_id === id)?.name || 'Desconocido';
    const getEmployeeName = (id) => employees.find(e => e.id === id)?.name || 'Sin asignar';
    const getServiceName = (id) => servicesCatalog.find(s => s.id === id)?.name || 'Desconocido';

    const getDeadlineStatusColor = (requestDateStr, deadlineDateStr, status) => {
        if (!deadlineDateStr || !requestDateStr) return 'gray.400';
        if (['FINISHED', 'CANCELLED', 'CERRADO_ADMIN'].includes(status)) return 'gray.400';

        const now = new Date();
        const deadline = new Date(deadlineDateStr);
        const request = new Date(requestDateStr);

        if (deadline < now) return 'red.500'; // Vencido

        const totalDuration = deadline.getTime() - request.getTime();
        const timeRemaining = deadline.getTime() - now.getTime();

        if (totalDuration <= 0) return 'gray.400';

        const percentageRemaining = timeRemaining / totalDuration;

        if (percentageRemaining <= 0.3) return 'yellow.400'; // <= 30% remaining
        return 'green.500'; // Good
    };

    const getTimeMessage = (deadlineDateStr, status) => {
        if (status === 'FINISHED') return "Servicio Finalizado";
        if (!deadlineDateStr) return "Sin fecha límite";

        const now = new Date();
        const deadline = new Date(deadlineDateStr);
        const diff = deadline.getTime() - now.getTime();
        const days = Math.abs(diff) / (1000 * 60 * 60 * 24);

        if (diff < 0) return `Retraso de ${days.toFixed(0)} días`;
        return `${days.toFixed(0)} días para atender`;
    };

    const handleViewDetails = async (order) => {
        setSelectedOrderForView(order);
        try {
            const serviceEvidences = await getEvidencesByService(order.id);
            setEvidences(serviceEvidences);
        } catch (error) {
            console.error("Error fetching evidences:", error);
            setEvidences([]);
        }
        onViewOpen();
    };

    const handleEdit = (order) => {
        setCurrentOrder(order);
        onOpen();
    };

    const handleCreate = () => {
        const user = JSON.parse(sessionStorage.getItem('user'));
        setCurrentOrder({
            service_number: `SRV-${new Date().getFullYear()}-${orders.length + 1}`,
            client_id: '',
            service_catalogo_id: '',
            status: 'PENDING',
            request_date: new Date().toISOString().split('T')[0],
            deadline_date: '',
            employee_id: '',
            company_id: user ? user.company_id : null
        });
        onOpen();
    };

    const handleReviewPayment = (order) => {
        setSelectedOrderForPayment(order);
        onPaymentOpen();
    };

    const handlePaymentAction = async (status) => {
        if (!selectedOrderForPayment) return;

        try {
            // Clean object before sending
            const { employee_name, employee_photo_url, ...cleanOrder } = selectedOrderForPayment;

            await updateOrder(selectedOrderForPayment.id, {
                ...cleanOrder,
                payment_status: status
            });

            toast({
                title: status === 'ACCEPTED' ? 'Pago Aceptado' : 'Pago Rechazado',
                status: status === 'ACCEPTED' ? 'success' : 'warning',
                duration: 2000,
            });
            onPaymentClose();
            fetchData();
        } catch (error) {
            console.error("Error updating payment status:", error);
            toast({
                title: 'Error',
                description: 'No se pudo actualizar el estado del pago.',
                status: 'error',
                duration: 3000,
            });
        }
    };

    const handleSave = async () => {
        try {
            // Clean object before sending
            const { employee_name, employee_photo_url, ...cleanOrder } = currentOrder;

            if (currentOrder.id) {
                await updateOrder(currentOrder.id, cleanOrder);
                toast({
                    title: 'Orden actualizada',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                await createOrder(currentOrder);
                toast({
                    title: 'Orden creada',
                    status: 'success',
                    duration: 2000,
                });
            }
            fetchData();
            onClose();
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    // ---- CALENDAR LOGIC ----
    const getDaysInMonth = (year, month) => {
        return new Date(year, month + 1, 0).getDate();
    };

    const getMonthData = () => {
        const year = currentDate.getFullYear();
        const month = currentDate.getMonth();
        const daysInMonth = getDaysInMonth(year, month);
        let firstDay = new Date(year, month, 1).getDay();
        const startDayIndex = (firstDay + 6) % 7;
        return { year, month, daysInMonth, startDayIndex };
    };

    const changeMonth = (offset) => {
        const newDate = new Date(currentDate);
        newDate.setMonth(newDate.getMonth() + offset);
        setCurrentDate(newDate);
    };

    const renderCalendarView = () => {
        const { year, month, daysInMonth, startDayIndex } = getMonthData();
        const monthName = currentDate.toLocaleString('es-ES', { month: 'long', year: 'numeric' });
        const weekDays = ['Lun', 'Mar', 'Mié', 'Jue', 'Vie', 'Sáb', 'Dom'];

        const cells = [];
        for (let i = 0; i < startDayIndex; i++) {
            cells.push(<Box key={`empty-${i}`} height="120px" bg="gray.100" />);
        }

        for (let day = 1; day <= daysInMonth; day++) {
            const dateStr = `${year}-${String(month + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
            const dayOrders = orders.filter(o =>
                o.deadline_date && o.deadline_date.startsWith(dateStr)
            );
            const isToday = new Date().toDateString() === new Date(year, month, day).toDateString();

            cells.push(
                <Box
                    key={day}
                    height="120px"
                    borderWidth="1px"
                    p={1}
                    bg={isToday ? "blue.50" : "white"}
                    position="relative"
                    overflowY="auto"
                >
                    <Text fontWeight="bold" fontSize="sm" color={isToday ? "blue.600" : "gray.700"}>
                        {day}
                    </Text>
                    <VStack align="start" spacing={1} mt={1}>
                        {dayOrders.map(order => (
                            <Box
                                key={order.id}
                                w="100%"
                                bg="brand.100"
                                p="2px 4px"
                                borderRadius="sm"
                                fontSize="xs"
                                cursor="pointer"
                                onClick={() => handleViewDetails(order)}
                                _hover={{ bg: "brand.200" }}
                            >
                                <Text noOfLines={1} fontWeight="semibold">
                                    {getServiceName(order.service_catalogo_id)}
                                </Text>
                                <Text noOfLines={1} fontSize="10px">
                                    {getClientName(order.client_id)}
                                </Text>
                            </Box>
                        ))}
                    </VStack>
                </Box>
            );
        }

        return (
            <Box>
                <HStack justify="space-between" align="center" mb={4}>
                    <IconButton icon={<ChevronLeftIcon />} onClick={() => changeMonth(-1)} aria-label="Mes anterior" />
                    <Heading size="md" textTransform="capitalize">{monthName}</Heading>
                    <IconButton icon={<ChevronRightIcon />} onClick={() => changeMonth(1)} aria-label="Mes siguiente" />
                </HStack>
                <SimpleGrid columns={7} spacing={1} border="1px solid" borderColor="gray.200" bg="gray.200">
                    {weekDays.map(d => (
                        <Box key={d} textAlign="center" fontWeight="bold" py={2} bg="gray.100">{d}</Box>
                    ))}
                    {cells}
                </SimpleGrid>
            </Box>
        );
    };

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <HStack>
                    <Heading>Órdenes de Servicio</Heading>
                    <HStack ml={4} spacing={0} borderWidth="1px" borderRadius="md">
                        <Button
                            size="sm"
                            variant={viewMode === 'list' ? 'solid' : 'ghost'}
                            colorScheme={viewMode === 'list' ? 'brand' : 'gray'}
                            onClick={() => setViewMode('list')}
                            borderRadius="none"
                            borderLeftRadius="md"
                        >
                            Lista
                        </Button>
                        <Button
                            size="sm"
                            variant={viewMode === 'calendar' ? 'solid' : 'ghost'}
                            colorScheme={viewMode === 'calendar' ? 'brand' : 'gray'}
                            onClick={() => setViewMode('calendar')}
                            borderRadius="none"
                            borderRightRadius="md"
                        >
                            Calendario
                        </Button>
                    </HStack>
                </HStack>
                <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                    Nueva Orden
                </Button>
            </HStack>

            {viewMode === 'list' ? (
                <Tabs variant="enclosed" colorScheme="brand" isLazy>
                    <TabList>
                        <Tab>Activas</Tab>
                        <Tab>Historial</Tab>
                    </TabList>
                    <TabPanels>
                        <TabPanel px={0}>
                            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                                <Table variant="simple">
                                    <Thead bg="gray.50">
                                        <Tr>
                                            <Th>Folio</Th>
                                            <Th>Cliente</Th>
                                            <Th>Servicio</Th>
                                            <Th>Empleado</Th>
                                            <Th>Fecha Límite</Th>
                                            <Th>Estado</Th>
                                            <Th>Acciones</Th>
                                        </Tr>
                                    </Thead>
                                    <Tbody>
                                        {orders.filter(o => ['PENDING', 'IN_PROGRESS', 'IN_REVIEW'].includes(o.status)).map((order) => {
                                            const isClientRequest = order.service_number?.startsWith('REQ-');
                                            const isUnconfirmed = !order.employee_id || !order.deadline_date;
                                            const shouldHighlight = order.status === 'PENDING' && isClientRequest && isUnconfirmed;

                                            return (
                                                <Tr key={order.id} bg={shouldHighlight ? '#aecccc' : 'transparent'}>
                                                    <Td fontWeight="bold">{order.service_number}</Td>
                                                    <Td>{getClientName(order.client_id)}</Td>
                                                    <Td>{getServiceName(order.service_catalogo_id)}</Td>
                                                    <Td>{getEmployeeName(order.employee_id)}</Td>
                                                    <Td>
                                                        <HStack>
                                                            <Box
                                                                w="3"
                                                                h="3"
                                                                borderRadius="full"
                                                                bg={getDeadlineStatusColor(order.request_date, order.deadline_date, order.status)}
                                                            />
                                                            <Text fontSize="sm">
                                                                {order.deadline_date ? order.deadline_date.split('T')[0] : '-'}
                                                            </Text>
                                                        </HStack>
                                                    </Td>
                                                    <Td>
                                                        <Badge
                                                            colorScheme={
                                                                order.status === 'IN_PROGRESS' ? 'blue' :
                                                                    order.status === 'PENDING' ? 'orange' :
                                                                        order.status === 'IN_REVIEW' ? 'purple' : 'gray'
                                                            }
                                                        >
                                                            {getStatusLabel(order.status)}
                                                        </Badge>
                                                    </Td>
                                                    <Td>
                                                        <HStack spacing={2}>
                                                            {order.payment_status === 'REVIEW' && (
                                                                <IconButton
                                                                    icon={<FiDollarSign />}
                                                                    size="sm"
                                                                    colorScheme="purple"
                                                                    aria-label="Revisar Pago"
                                                                    onClick={() => handleReviewPayment(order)}
                                                                />
                                                            )}
                                                            <IconButton
                                                                icon={<FiEdit2 />}
                                                                size="sm"
                                                                aria-label="Editar"
                                                                onClick={() => handleEdit(order)}
                                                            />
                                                            <IconButton
                                                                icon={<FiEye />}
                                                                size="sm"
                                                                aria-label="Ver Detalles"
                                                                onClick={() => handleViewDetails(order)}
                                                            />
                                                        </HStack>
                                                    </Td>
                                                </Tr>
                                            )
                                        })}
                                    </Tbody>
                                </Table>
                            </Box>
                        </TabPanel>

                        <TabPanel px={0}>
                            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                                <Table variant="simple">
                                    <Thead bg="gray.50">
                                        <Tr>
                                            <Th>Folio</Th>
                                            <Th>Cliente</Th>
                                            <Th>Servicio</Th>
                                            <Th>Empleado</Th>
                                            <Th>Fecha Límite</Th>
                                            <Th>Estado</Th>
                                            <Th>Acciones</Th>
                                        </Tr>
                                    </Thead>
                                    <Tbody>
                                        {orders.filter(o => ['FINISHED', 'CANCELLED', 'CERRADO_ADMIN'].includes(o.status)).map((order) => (
                                            <Tr key={order.id}>
                                                <Td fontWeight="bold" color="gray.500">{order.service_number}</Td>
                                                <Td color="gray.500">{getClientName(order.client_id)}</Td>
                                                <Td color="gray.500">{getServiceName(order.service_catalogo_id)}</Td>
                                                <Td color="gray.500">{getEmployeeName(order.employee_id)}</Td>
                                                <Td color="gray.500">
                                                    <HStack>
                                                        <Box w="3" h="3" borderRadius="full" bg="gray.400" />
                                                        <Text fontSize="sm">{order.deadline_date ? order.deadline_date.split('T')[0] : '-'}</Text>
                                                    </HStack>
                                                </Td>
                                                <Td>
                                                    <Badge
                                                        colorScheme={
                                                            order.status === 'FINISHED' ? 'green' :
                                                                order.status === 'CANCELLED' ? 'red' : 'gray'
                                                        }
                                                    >
                                                        {getStatusLabel(order.status)}
                                                    </Badge>
                                                </Td>
                                                <Td>
                                                    <HStack spacing={2}>
                                                        <IconButton
                                                            icon={<FiEye />}
                                                            size="sm"
                                                            aria-label="Ver Detalles"
                                                            onClick={() => handleViewDetails(order)}
                                                        />
                                                    </HStack>
                                                </Td>
                                            </Tr>
                                        ))}
                                    </Tbody>
                                </Table>
                            </Box>
                        </TabPanel>
                    </TabPanels>
                </Tabs>
            ) : (
                renderCalendarView()
            )}

            {/* Modal Edit/Create */}
            <Modal isOpen={isOpen} onClose={onClose} size="lg">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentOrder?.id ? 'Editar Orden' : 'Nueva Orden'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <Box display="flex" flexDirection="column" gap={4}>
                            <FormControl>
                                <FormLabel>Cliente</FormLabel>
                                <Select
                                    value={currentOrder?.client_id || ''}
                                    onChange={(e) => setCurrentOrder({ ...currentOrder, client_id: e.target.value })}
                                >
                                    <option value="">Seleccionar...</option>
                                    {clients.map(c => (
                                        <option key={c.user_id} value={c.user_id}>{c.name}</option>
                                    ))}
                                </Select>
                            </FormControl>

                            <FormControl>
                                <FormLabel>Servicio</FormLabel>
                                <Select
                                    value={currentOrder?.service_catalogo_id || ''}
                                    onChange={(e) => setCurrentOrder({ ...currentOrder, service_catalogo_id: parseInt(e.target.value) })}
                                >
                                    <option value="">Seleccionar...</option>
                                    {servicesCatalog.map(s => (
                                        <option key={s.id} value={s.id}>{s.name}</option>
                                    ))}
                                </Select>
                            </FormControl>

                            <FormControl>
                                <FormLabel>Asignar Empleado</FormLabel>
                                <Select
                                    value={currentOrder?.employee_id || ''}
                                    onChange={(e) => setCurrentOrder({ ...currentOrder, employee_id: e.target.value ? parseInt(e.target.value) : '' })}
                                >
                                    <option value="">Sin asignar</option>
                                    {employees.map(e => (
                                        <option key={e.id} value={e.id}>{e.name} ({e.specialty})</option>
                                    ))}
                                </Select>
                            </FormControl>

                            <HStack>
                                <FormControl>
                                    <FormLabel>Fecha Solicitud</FormLabel>
                                    <Input
                                        type="date"
                                        value={currentOrder?.request_date?.split('T')[0] || ''}
                                        onChange={(e) => setCurrentOrder({ ...currentOrder, request_date: e.target.value })}
                                    />
                                </FormControl>

                                <FormControl>
                                    <FormLabel>Fecha Límite</FormLabel>
                                    <Input
                                        type="date"
                                        value={currentOrder?.deadline_date?.split('T')[0] || ''}
                                        onChange={(e) => setCurrentOrder({ ...currentOrder, deadline_date: e.target.value })}
                                    />
                                </FormControl>
                            </HStack>

                            <FormControl>
                                <FormLabel>Estado</FormLabel>
                                <Select
                                    value={currentOrder?.status || ''}
                                    onChange={(e) => {
                                        if (e.target.value === 'CANCELLED' && !isAdmin) {
                                            toast({
                                                title: 'Permiso denegado',
                                                description: 'Solo los administradores pueden cancelar servicios',
                                                status: 'warning',
                                                duration: 3000,
                                            });
                                            return;
                                        }
                                        setCurrentOrder({ ...currentOrder, status: e.target.value });
                                    }}
                                >
                                    <option value="PENDING">Pendiente</option>
                                    <option value="IN_PROGRESS">En Progreso</option>
                                    <option value="FINISHED">Finalizado</option>
                                    <option value="CANCELLED" disabled={!isAdmin}>
                                        {isAdmin ? 'Cancelado' : 'Cancelado (Solo Admin)'}
                                    </option>
                                    <option value="CERRADO_ADMIN">Cerrado</option>
                                </Select>
                                {!isAdmin && (
                                    <Text fontSize="xs" color="orange.600" mt={1}>
                                        ℹ️ Solo los administradores pueden cancelar servicios
                                    </Text>
                                )}
                            </FormControl>

                            {(currentOrder?.status === 'IN_REVIEW' || currentOrder?.status === 'CERRADO_ADMIN') && (
                                <Box mt={4} p={4} bg="gray.50" borderRadius="md">
                                    <Heading size="sm" mb={2}>Feedback del Cliente</Heading>
                                    {currentOrder.client_rating && (
                                        <HStack>
                                            <Text fontWeight="bold">Calificación:</Text>
                                            <HStack spacing={1}>
                                                {[...Array(5)].map((_, i) => (
                                                    <FiStar key={i} color={i < currentOrder.client_rating ? 'orange' : 'gray'} />
                                                ))}
                                            </HStack>
                                        </HStack>
                                    )}
                                    {currentOrder.client_comment && (
                                        <Box mt={2}>
                                            <Text fontWeight="bold">Comentario:</Text>
                                            <Text>{currentOrder.client_comment}</Text>
                                        </Box>
                                    )}
                                </Box>
                            )}
                        </Box>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>

            {/* View Details Modal */}
            <Modal isOpen={isViewOpen} onClose={onViewClose} size="xl">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Detalles del Servicio</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        {selectedOrderForView && (
                            <VStack align="stretch" spacing={4}>
                                <SimpleGrid columns={2} spacing={4}>
                                    <Box>
                                        <Text fontWeight="bold">Folio:</Text>
                                        <Text>{selectedOrderForView.service_number}</Text>
                                    </Box>
                                    <Box>
                                        <Text fontWeight="bold">Estado:</Text>
                                        <Badge colorScheme={
                                            selectedOrderForView.status === 'FINISHED' ? 'green' :
                                                selectedOrderForView.status === 'IN_PROGRESS' ? 'blue' :
                                                    selectedOrderForView.status === 'PENDING' ? 'orange' :
                                                        selectedOrderForView.status === 'IN_REVIEW' ? 'purple' : 'gray'
                                        }>
                                            {getStatusLabel(selectedOrderForView.status)}
                                        </Badge>
                                    </Box>
                                </SimpleGrid>
                                <Divider />
                                <Tabs variant="enclosed">
                                    <TabList>
                                        <Tab>General</Tab>
                                        <Tab>Evidencias ({evidences.length})</Tab>
                                    </TabList>
                                    <TabPanels>
                                        <TabPanel>
                                            <VStack align="stretch" spacing={3}>
                                                <Box>
                                                    <Text fontWeight="bold">Cliente:</Text>
                                                    <Text>{getClientName(selectedOrderForView.client_id)}</Text>
                                                </Box>
                                                <Box>
                                                    <Text fontWeight="bold">Servicio:</Text>
                                                    <Text>{getServiceName(selectedOrderForView.service_catalogo_id)}</Text>
                                                </Box>
                                                <Box>
                                                    <Text fontWeight="bold">Empleado Asignado:</Text>
                                                    <Text>{getEmployeeName(selectedOrderForView.employee_id)}</Text>
                                                </Box>
                                                <Box>
                                                    <Text fontWeight="bold">Fecha Inicio:</Text>
                                                    <Text>{selectedOrderForView.request_date?.split('T')[0]}</Text>
                                                </Box>
                                                <Box>
                                                    <Text fontWeight="bold">Fecha Límite:</Text>
                                                    <Text>{selectedOrderForView.deadline_date?.split('T')[0] || '-'}</Text>
                                                </Box>
                                                <Box>
                                                    <Text fontWeight="bold">Tiempo:</Text>
                                                    <HStack>
                                                        <Box
                                                            w="3"
                                                            h="3"
                                                            borderRadius="full"
                                                            bg={getDeadlineStatusColor(selectedOrderForView.request_date, selectedOrderForView.deadline_date, selectedOrderForView.status)}
                                                        />
                                                        <Text fontWeight="semibold" fontSize="sm">
                                                            {getTimeMessage(selectedOrderForView.deadline_date, selectedOrderForView.status)}
                                                        </Text>
                                                    </HStack>
                                                </Box>
                                                <Divider />
                                                {(selectedOrderForView.status === 'IN_REVIEW' || selectedOrderForView.status === 'CERRADO_ADMIN') && (
                                                    <Box p={3} bg="gray.50" borderRadius="md">
                                                        <Heading size="sm" mb={2}>Feedback del Cliente</Heading>
                                                        {selectedOrderForView.client_rating && (
                                                            <HStack>
                                                                <Text fontWeight="bold">Calificación:</Text>
                                                                <HStack spacing={1}>
                                                                    {[...Array(5)].map((_, i) => (
                                                                        <FiStar key={i} color={i < selectedOrderForView.client_rating ? 'orange' : 'gray'} />
                                                                    ))}
                                                                </HStack>
                                                            </HStack>
                                                        )}
                                                        {selectedOrderForView.client_comment && (
                                                            <Box mt={2}>
                                                                <Text fontWeight="bold">Comentario:</Text>
                                                                <Text whiteSpace="pre-wrap">{selectedOrderForView.client_comment}</Text>
                                                            </Box>
                                                        )}

                                                        {evidences.filter(e => e.description?.startsWith("Evidencia de reapertura:")).length > 0 && (
                                                            <Box mt={3}>
                                                                <Text fontWeight="bold">Evidencias de Reapertura:</Text>
                                                                <SimpleGrid columns={3} spacing={2} mt={1}>
                                                                    {evidences.filter(e => e.description?.startsWith("Evidencia de reapertura:")).map(ev => (
                                                                        <Image key={ev.id} src={ev.file_url} alt="Evidencia de reapertura" borderRadius="md" />
                                                                    ))}
                                                                </SimpleGrid>
                                                            </Box>
                                                        )}
                                                    </Box>
                                                )}
                                            </VStack>
                                        </TabPanel>
                                        <TabPanel>
                                            {evidences.filter(e => !e.description?.startsWith("Evidencia de reapertura:")).length > 0 ? (
                                                <SimpleGrid columns={2} spacing={4}>
                                                    {evidences.filter(e => !e.description?.startsWith("Evidencia de reapertura:")).map(ev => (
                                                        <Box key={ev.id} borderWidth="1px" borderRadius="lg" p={2}>
                                                            <Image src={ev.file_url} alt={ev.description} borderRadius="md" mb={2} />
                                                            <Text fontSize="sm">{ev.description}</Text>
                                                        </Box>
                                                    ))}
                                                </SimpleGrid>
                                            ) : (
                                                <Text>No hay evidencias generales para este servicio.</Text>
                                            )}
                                        </TabPanel>
                                    </TabPanels>
                                </Tabs>
                            </VStack>
                        )}
                    </ModalBody>
                    <ModalFooter>
                        <Button colorScheme="brand" onClick={onViewClose}>
                            Cerrar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>

            {/* Payment Review Modal */}
            <Modal isOpen={isPaymentOpen} onClose={onPaymentClose} size="lg">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Revisión de Pago</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        {selectedOrderForPayment && (
                            <VStack align="stretch" spacing={4}>
                                <Box p={3} bg="blue.50" borderRadius="md">
                                    <Heading size="sm" mb={2}>Información del Pago</Heading>
                                    <Text><strong>Referencia:</strong> {selectedOrderForPayment.payment_reference}</Text>
                                    <Text><strong>Monto:</strong> {selectedOrderForPayment.cost || 'N/A'}</Text>
                                    <Text><strong>Fecha:</strong> {new Date().toLocaleDateString()}</Text>
                                </Box>

                                <Box>
                                    <Heading size="sm" mb={2}>Comprobante</Heading>
                                    {selectedOrderForPayment.payment_proof_url ? (
                                        <Image
                                            src={selectedOrderForPayment.payment_proof_url}
                                            alt="Comprobante de pago"
                                            borderRadius="md"
                                            maxH="400px"
                                            objectFit="contain"
                                            fallbackSrc="https://via.placeholder.com/400?text=Error+cargando+imagen"
                                        />
                                    ) : (
                                        <Text color="red.500">No se ha subido comprobante.</Text>
                                    )}
                                </Box>
                            </VStack>
                        )}
                    </ModalBody>
                    <ModalFooter>
                        <Button colorScheme="red" variant="outline" mr={3} onClick={() => handlePaymentAction('REJECTED')}>
                            Rechazar
                        </Button>
                        <Button colorScheme="green" onClick={() => handlePaymentAction('ACCEPTED')}>
                            Aceptado
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box >
    );
};

export default OrdenesServicio;
