import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    SimpleGrid,
    Card,
    CardHeader,
    CardBody,
    Text,
    Stat,
    StatLabel,
    StatNumber,
    StatHelpText,
    StatArrow,
    Progress,
    useToast
} from '@chakra-ui/react';
import {
    AreaChart,
    Area,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    ResponsiveContainer,
    Legend,
    PieChart,
    Pie,
    Cell
} from 'recharts';
import { getOrders, getServicesByCompany, getEmployees } from '../../services/api';

const Reportes = () => {
    const [chartData, setChartData] = useState([]);
    const [serviceNames, setServiceNames] = useState([]);

    // Estados para Reportes Dinámicos
    const [satisfactionData, setSatisfactionData] = useState({
        average: 0,
        count5: 0, count4: 0, count3: 0, countLess3: 0,
        pct5: 0, pct4: 0, pct3: 0, pctLess3: 0
    });

    const [efficiencyData, setEfficiencyData] = useState({
        avgCloseTime: 0,
        onTimePct: 0,
        latePct: 0,
        criticalPct: 0
    });

    const [employeeChartData, setEmployeeChartData] = useState([]);

    const toast = useToast();

    // Colores para las diferentes áreas
    const colors = [
        "#8884d8", "#82ca9d", "#ffc658", "#ff7300", "#d0ed57",
        "#0088fe", "#00c49f", "#ffbb28", "#ff8042", "#a4de6c"
    ];

    // Colores para el Pie Chart
    const pieColors = ["#0088FE", "#00C49F", "#FFBB28", "#FF8042", "#8884D8", "#82CA9D"];

    useEffect(() => {
        const fetchData = async () => {
            try {
                const user = JSON.parse(sessionStorage.getItem('user'));
                const institutionId = user ? (user.institution_id || user.company_id) : null;
                if (!institutionId) return;

                const [orders, services, employees] = await Promise.all([
                    getOrders(),
                    getServicesByCompany(institutionId),
                    getEmployees()
                ]);

                processData(orders, services, employees, institutionId);

            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar datos',
                    status: 'error',
                    duration: 3000,
                });
            }
        };

        fetchData();
    }, []);

    const processData = (orders, servicesCatalog, allEmployees, institutionId) => {
        const currentYear = new Date().getFullYear();
        const companyEmployees = allEmployees.filter(e => (e.institution_id === institutionId || e.company_id === institutionId));

        // Filtrar órdenes de la institución
        const companyOrders = orders.filter(o => (o.institution_id === institutionId || o.company_id === institutionId));

        // ---- 1. Datos para Gráfico de Áreas (Año Actual) ----
        const currentYearOrders = companyOrders.filter(o => {
            const orderDate = new Date(o.request_date);
            return orderDate.getFullYear() === currentYear;
        });

        const uniqueServiceIds = [...new Set(currentYearOrders.map(o => o.service_catalogo_id))];
        const relevantServices = servicesCatalog.filter(s => uniqueServiceIds.includes(s.id));
        const relevantServiceNames = relevantServices.map(s => s.name);

        setServiceNames(relevantServiceNames);

        const monthsData = Array.from({ length: 12 }, (_, i) => {
            const date = new Date(currentYear, i, 1);
            const monthName = date.toLocaleString('es-ES', { month: 'short' });
            const initialCounts = relevantServiceNames.reduce((acc, name) => ({ ...acc, [name]: 0 }), {});

            return {
                name: monthName.charAt(0).toUpperCase() + monthName.slice(1),
                monthIndex: i,
                ...initialCounts
            };
        });

        currentYearOrders.forEach(order => {
            const date = new Date(order.request_date);
            const monthIndex = date.getMonth();
            const serviceName = servicesCatalog.find(s => s.id === order.service_catalogo_id)?.name;

            if (serviceName && monthsData[monthIndex]) {
                monthsData[monthIndex][serviceName] = (monthsData[monthIndex][serviceName] || 0) + 1;
            }
        });

        setChartData(monthsData);

        // ---- 2. Satisfacción del Cliente (Todas las órdenes recibidas de la empresa con calificación) ----
        const ratedOrders = companyOrders.filter(o => o.client_rating && o.client_rating > 0);
        const totalRated = ratedOrders.length;

        if (totalRated > 0) {
            const sumRatings = ratedOrders.reduce((acc, o) => acc + o.client_rating, 0);
            const average = (sumRatings / totalRated).toFixed(1);

            const count5 = ratedOrders.filter(o => o.client_rating === 5).length;
            const count4 = ratedOrders.filter(o => o.client_rating === 4).length;
            const count3 = ratedOrders.filter(o => o.client_rating === 3).length;
            const countLess3 = ratedOrders.filter(o => o.client_rating < 3).length;

            setSatisfactionData({
                average,
                count5, count4, count3, countLess3,
                pct5: Math.round((count5 / totalRated) * 100),
                pct4: Math.round((count4 / totalRated) * 100),
                pct3: Math.round((count3 / totalRated) * 100),
                pctLess3: Math.round((countLess3 / totalRated) * 100),
            });
        }

        // ---- 3. Eficiencia de Servicios ----
        const now = new Date();
        const activeOrFinishedOrders = companyOrders.filter(o => o.status !== 'CANCELLED');
        const totalEfficiencyOrders = activeOrFinishedOrders.length;

        if (totalEfficiencyOrders > 0) {
            let onTimeCount = 0;
            let lateCount = 0;
            let totalDurationDays = 0;
            let durationCount = 0;

            activeOrFinishedOrders.forEach(o => {
                const deadline = new Date(o.deadline_date);
                const request = new Date(o.request_date);

                if (o.status === 'FINISHED') {
                    const duration = Math.max(0, (deadline - request) / (1000 * 60 * 60 * 24));
                    totalDurationDays += duration;
                    durationCount++;
                    onTimeCount++;
                } else {
                    if (deadline < now) {
                        lateCount++;
                    } else {
                        onTimeCount++;
                    }
                }
            });

            setEfficiencyData({
                avgCloseTime: durationCount > 0 ? (totalDurationDays / durationCount).toFixed(1) : 0,
                onTimePct: Math.round((onTimeCount / totalEfficiencyOrders) * 100),
                latePct: Math.round((lateCount / totalEfficiencyOrders) * 100),
                criticalPct: 0
            });
        }

        // ---- 4. Gráfico de Pastel: Productividad por empleado (Año Actual) ----
        const employeeCounts = {};

        currentYearOrders.forEach(o => {
            if (o.employee_id) {
                const empName = companyEmployees.find(e => e.id === o.employee_id)?.name || 'Desconocido';
                employeeCounts[empName] = (employeeCounts[empName] || 0) + 1;
            } else {
                employeeCounts['Sin Asignar'] = (employeeCounts['Sin Asignar'] || 0) + 1;
            }
        });

        const pieData = Object.keys(employeeCounts).map(name => ({
            name,
            value: employeeCounts[name]
        }));

        setEmployeeChartData(pieData);
    };

    return (
        <Box>
            <Heading mb={6}>Reportes y Métricas</Heading>

            <SimpleGrid columns={{ base: 1, md: 2 }} spacing={6}>
                <Card>
                    <CardHeader>
                        <Heading size="md">Satisfacción del Cliente</Heading>
                    </CardHeader>
                    <CardBody>
                        <Stat mb={4}>
                            <StatLabel>Calificación Promedio</StatLabel>
                            <StatNumber>{satisfactionData.average} / 5.0</StatNumber>
                            <StatHelpText>Global</StatHelpText>
                        </Stat>
                        <Box>
                            <Text fontSize="sm" mb={1}>5 Estrellas ({satisfactionData.pct5 || 0}%)</Text>
                            <Progress value={satisfactionData.pct5 || 0} colorScheme="green" size="sm" mb={3} />

                            <Text fontSize="sm" mb={1}>4 Estrellas ({satisfactionData.pct4 || 0}%)</Text>
                            <Progress value={satisfactionData.pct4 || 0} colorScheme="blue" size="sm" mb={3} />

                            <Text fontSize="sm" mb={1}>3 Estrellas ({satisfactionData.pct3 || 0}%)</Text>
                            <Progress value={satisfactionData.pct3 || 0} colorScheme="yellow" size="sm" mb={3} />

                            <Text fontSize="sm" mb={1}>Menos de 3 ({satisfactionData.pctLess3 || 0}%)</Text>
                            <Progress value={satisfactionData.pctLess3 || 0} colorScheme="red" size="sm" />
                        </Box>
                    </CardBody>
                </Card>

                <Card>
                    <CardHeader>
                        <Heading size="md">Eficiencia de Servicios</Heading>
                    </CardHeader>
                    <CardBody>
                        <Stat mb={4}>
                            <StatLabel>Tiempo Promedio (Días)</StatLabel>
                            <StatNumber>{efficiencyData.avgCloseTime} Días</StatNumber>
                            <StatHelpText>Estimado</StatHelpText>
                        </Stat>
                        <Box>
                            <Text fontSize="sm" mb={1}>A tiempo ({efficiencyData.onTimePct || 0}%)</Text>
                            <Progress value={efficiencyData.onTimePct || 0} colorScheme="green" size="sm" mb={3} />

                            <Text fontSize="sm" mb={1}>Con retraso ({efficiencyData.latePct || 0}%)</Text>
                            <Progress value={efficiencyData.latePct || 0} colorScheme="red" size="sm" mb={3} />
                        </Box>
                    </CardBody>
                </Card>

                {/* Gráfico de Áreas - Servicios */}
                <Card gridColumn={{ md: "span 2" }}>
                    <CardHeader>
                        <Heading size="md">Servicios</Heading>
                    </CardHeader>
                    <CardBody>
                        <Box height="400px">
                            <ResponsiveContainer width="100%" height="100%">
                                <AreaChart
                                    data={chartData}
                                    margin={{ top: 10, right: 30, left: 0, bottom: 0 }}
                                >
                                    <CartesianGrid strokeDasharray="3 3" />
                                    <XAxis dataKey="name" />
                                    <YAxis />
                                    <Tooltip />
                                    <Legend />
                                    {serviceNames.map((serviceName, index) => (
                                        <Area
                                            key={serviceName}
                                            type="monotone"
                                            dataKey={serviceName}
                                            stackId="1"
                                            stroke={colors[index % colors.length]}
                                            fill={colors[index % colors.length]}
                                        />
                                    ))}
                                </AreaChart>
                            </ResponsiveContainer>
                        </Box>
                    </CardBody>
                </Card>

                {/* Gráfico de Pastel: Productividad por empleado */}
                <Card gridColumn={{ md: "span 2" }}>
                    <CardHeader>
                        <Heading size="md">Productividad por empleado</Heading>
                    </CardHeader>
                    <CardBody>
                        <Box height="400px" width="100%" display="flex" justifyContent="center">
                            <ResponsiveContainer width="100%" height="100%">
                                <PieChart>
                                    <Pie
                                        data={employeeChartData}
                                        cx="50%"
                                        cy="50%"
                                        labelLine={false}
                                        label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(0)}%`}
                                        outerRadius={150}
                                        fill="#8884d8"
                                        dataKey="value"
                                    >
                                        {employeeChartData.map((entry, index) => (
                                            <Cell key={`cell-${index}`} fill={pieColors[index % pieColors.length]} />
                                        ))}
                                    </Pie>
                                    <Tooltip />
                                    <Legend />
                                </PieChart>
                            </ResponsiveContainer>
                        </Box>
                    </CardBody>
                </Card>

            </SimpleGrid>
        </Box>
    );
};

export default Reportes;
