import React, { useState, useMemo } from 'react';
import {
  Box,
  Heading,
  VStack,
  HStack,
  Text,
  Button,
  Badge,
  Image,
  useToast,
  Flex,
  Icon,
  Divider,
  SimpleGrid,
  Card,
  CardBody,
  List,
  ListItem,
  ListIcon,
  Tooltip,
  IconButton,
  useColorModeValue,
  Input,
  FormLabel,
  FormControl,
  Alert,
  AlertIcon,
  AlertTitle,
  AlertDescription,
  Select,
  InputGroup,
  InputLeftElement,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  useDisclosure,
  Textarea
} from '@chakra-ui/react';
import {
  FiCheckCircle,
  FiXCircle,
  FiAlertTriangle,
  FiFileText,
  FiUpload,
  FiEye,
  FiCheck,
  FiX,
  FiSearch,
  FiFilter,
  FiDollarSign
} from 'react-icons/fi';
import Layout from '../../components/Layout';
import { adminLinks } from './adminLinks';
import { mockPagos, mockAlumnos, mockPadres } from '../../data/mockData';

// Mock Bank Statement Data for Simulation
const mockBankStatement = [
  {
    date: '2024-01-05',
    time: '10:35', // 5 mins diff
    amount: 1500.00,
    bank: 'BBVA',
    trackingKey: '1234567890',
    sender: 'Roberto López',
    concept: 'PAGO-ALU-1',
    type: 'Abono'
  },
  {
    date: '2024-01-10',
    time: '14:15', // Exact match
    amount: 1500.00,
    bank: 'Santander',
    trackingKey: '0987654321',
    sender: 'Patricia Ramírez',
    concept: 'PAGO-ALU-2',
    type: 'Abono'
  },
  // Mismatch example
  {
    date: '2024-01-15',
    time: '09:00',
    amount: 1499.50, // Cent difference
    bank: 'Banorte',
    trackingKey: '1122334455',
    sender: 'Fernando Torres',
    concept: 'PAGO-ALU-3',
    type: 'Abono'
  }
];

const ValidacionPagos = () => {
  const [pagos, setPagos] = useState(mockPagos.filter(p => p.estatus === 'Pendiente'));
  const [selectedPago, setSelectedPago] = useState(null);
  const [statementData, setStatementData] = useState([]); // Now holds all transactions from PDF
  const [validationResults, setValidationResults] = useState({}); // Map of pagoId -> result
  const [isUploading, setIsUploading] = useState(false);
  const [isFileUploaded, setIsFileUploaded] = useState(false);

  // Filters
  const [filterName, setFilterName] = useState('');
  const [filterDate, setFilterDate] = useState('');

  // Cash payment modal
  const { isOpen, onOpen, onClose } = useDisclosure();
  const [cashPaymentForm, setCashPaymentForm] = useState({
    id_alumno: '',
    monto: '',
    fecha_pago: new Date().toISOString().split('T')[0],
    concepto: '',
    notas: ''
  });

  const toast = useToast();
  const bgColor = useColorModeValue('white', 'gray.800');
  const borderColor = useColorModeValue('gray.200', 'gray.700');

  const filteredPagos = useMemo(() => {
    return pagos.filter(pago => {
      const alumno = mockAlumnos.find(a => a.id === pago.id_alumno);
      const matchesName = alumno?.nombre.toLowerCase().includes(filterName.toLowerCase());
      const matchesDate = filterDate ? pago.fecha_pago === filterDate : true;
      return matchesName && matchesDate;
    });
  }, [pagos, filterName, filterDate]);

  const handleSelectPago = (pago) => {
    setSelectedPago(pago);
  };

  const handleFileUpload = (e) => {
    const file = e.target.files[0];
    if (!file) return;

    setIsUploading(true);

    // Simulate PDF parsing delay
    setTimeout(() => {
      setIsUploading(false);
      setIsFileUploaded(true);
      setStatementData(mockBankStatement); // Load all mock transactions

      // Run validation for ALL pending payments
      const newResults = {};
      pagos.forEach(pago => {
        const match = mockBankStatement.find(t =>
          t.concept === pago.concepto ||
          (t.amount === pago.monto && t.date === pago.fecha_pago)
        );

        if (match) {
          newResults[pago.id] = {
            matchFound: true,
            data: match,
            checks: {
              date: checkDate(pago.fecha_pago, pago.hora_pago, match.date, match.time),
              amount: pago.monto === match.amount,
              bank: pago.banco_origen?.toLowerCase() === match.bank?.toLowerCase(),
              tracking: pago.clave_rastreo === match.trackingKey,
              sender: pago.nombre_remitente?.toLowerCase() === match.sender?.toLowerCase(),
              concept: pago.concepto === match.concept
            }
          };
        } else {
          newResults[pago.id] = { matchFound: false };
        }
      });

      setValidationResults(newResults);

      toast({
        title: 'Estado de cuenta procesado',
        description: `Se analizaron ${pagos.length} pagos pendientes.`,
        status: 'success',
        duration: 3000,
      });
    }, 1500);
  };

  const checkDate = (pDate, pTime, sDate, sTime) => {
    if (pDate !== sDate) return false;
    if (!pTime || !sTime) return true; // Cannot validate time strictly if missing

    // Simple time diff check (mock)
    const [pH, pM] = pTime.split(':').map(Number);
    const [sH, sM] = sTime.split(':').map(Number);
    const pMinutes = pH * 60 + pM;
    const sMinutes = sH * 60 + sM;

    return Math.abs(pMinutes - sMinutes) <= 60; // Allow 1 hour diff
  };

  const handleAprobar = () => {
    setPagos(pagos.filter(p => p.id !== selectedPago.id));
    setSelectedPago(null);
    toast({ title: 'Pago Aprobado', status: 'success', duration: 2000 });
  };

  const handleRechazar = () => {
    setPagos(pagos.filter(p => p.id !== selectedPago.id));
    setSelectedPago(null);
    toast({ title: 'Pago Rechazado', status: 'error', duration: 2000 });
  };

  const handleCashPaymentSubmit = () => {
    // Validate form
    if (!cashPaymentForm.id_alumno || !cashPaymentForm.monto || !cashPaymentForm.concepto) {
      toast({
        title: 'Campos requeridos',
        description: 'Por favor completa todos los campos obligatorios',
        status: 'warning',
        duration: 3000
      });
      return;
    }

    // Create new cash payment record
    const newPago = {
      id: `CASH-${Date.now()}`,
      id_alumno: parseInt(cashPaymentForm.id_alumno),
      monto: parseFloat(cashPaymentForm.monto),
      fecha_pago: cashPaymentForm.fecha_pago,
      concepto: cashPaymentForm.concepto,
      estatus: 'Aprobado',
      metodo_pago: 'Efectivo',
      notas: cashPaymentForm.notas,
      fecha_validacion: new Date().toISOString().split('T')[0]
    };

    // Here you would normally save to backend
    console.log('Nuevo pago en efectivo:', newPago);

    toast({
      title: 'Pago registrado exitosamente',
      description: `Pago en efectivo de $${cashPaymentForm.monto} registrado para el alumno`,
      status: 'success',
      duration: 3000
    });

    // Reset form and close modal
    setCashPaymentForm({
      id_alumno: '',
      monto: '',
      fecha_pago: new Date().toISOString().split('T')[0],
      concepto: '',
      notas: ''
    });
    onClose();
  };

  const handleCashPaymentFormChange = (field, value) => {
    setCashPaymentForm(prev => ({
      ...prev,
      [field]: value
    }));
  };


  const ValidationItem = ({ label, isValid, value, statementValue }) => (
    <HStack justify="space-between" py={2} borderBottom="1px" borderColor="gray.100">
      <VStack align="start" spacing={0}>
        <Text fontSize="sm" fontWeight="medium" color="gray.600">{label}</Text>
        <Text fontSize="xs" color="gray.500">Captura: {value || 'N/A'}</Text>
        {statementValue && (
          <Text fontSize="xs" color="blue.500">Banco: {statementValue}</Text>
        )}
      </VStack>
      {isValid ? (
        <Icon as={FiCheckCircle} color="green.500" boxSize={5} />
      ) : (
        <Icon as={FiXCircle} color="red.500" boxSize={5} />
      )}
    </HStack>
  );

  const getValidationStatusIcon = (pagoId) => {
    if (!isFileUploaded) return null;
    const result = validationResults[pagoId];

    if (!result || !result.matchFound) {
      return <Icon as={FiAlertTriangle} color="orange.400" />;
    }

    const allChecksPassed = Object.values(result.checks).every(Boolean);
    return allChecksPassed ? (
      <Icon as={FiCheckCircle} color="green.500" />
    ) : (
      <Icon as={FiXCircle} color="red.500" />
    );
  };

  return (
    <Layout links={adminLinks} userType="Administrador" userName="Admin User">
      <Box h="calc(100vh - 100px)">
        <Flex justify="space-between" align="center" mb={4}>
          <Heading size="lg">Validación de Pagos</Heading>

          <HStack spacing={3}>
            {/* Cash Payment Button */}
            <Button
              leftIcon={<FiDollarSign />}
              colorScheme="purple"
              onClick={onOpen}
            >
              Registrar Pago en Efectivo
            </Button>

            {/* Global Upload Button */}
            <Box position="relative">
              <Button
                leftIcon={<FiUpload />}
                colorScheme={isFileUploaded ? "green" : "blue"}
                isLoading={isUploading}
                loadingText="Procesando..."
              >
                {isFileUploaded ? "Estado de Cuenta Cargado" : "Cargar Estado de Cuenta (PDF)"}
              </Button>
              <Input
                type="file"
                accept=".pdf"
                opacity={0}
                position="absolute"
                top={0} left={0} w="100%" h="100%"
                cursor="pointer"
                onChange={handleFileUpload}
                isDisabled={isUploading}
              />
            </Box>
          </HStack>
        </Flex>

        <Flex gap={6} h="full" direction={{ base: 'column', lg: 'row' }}>

          {/* Left Panel: List of Payments */}
          <Box
            w={{ base: '100%', lg: '400px' }}
            bg={bgColor}
            borderRadius="xl"
            boxShadow="sm"
            border="1px"
            borderColor={borderColor}
            display="flex"
            flexDirection="column"
            flexShrink={0}
          >
            {/* Filters Header */}
            <Box p={4} borderBottom="1px" borderColor={borderColor} bg="gray.50">
              <VStack spacing={3} align="stretch">
                <Text fontWeight="bold" color="gray.600">Pagos Pendientes ({filteredPagos.length})</Text>
                <InputGroup size="sm">
                  <InputLeftElement pointerEvents="none">
                    <FiSearch color="gray.300" />
                  </InputLeftElement>
                  <Input
                    placeholder="Buscar alumno..."
                    bg="white"
                    value={filterName}
                    onChange={(e) => setFilterName(e.target.value)}
                  />
                </InputGroup>
                <Input
                  type="date"
                  size="sm"
                  bg="white"
                  value={filterDate}
                  onChange={(e) => setFilterDate(e.target.value)}
                />
              </VStack>
            </Box>

            {/* List Content */}
            <Box overflowY="auto" flex={1}>
              <List spacing={0}>
                {filteredPagos.map(pago => {
                  const alumno = mockAlumnos.find(a => a.id === pago.id_alumno);
                  return (
                    <ListItem
                      key={pago.id}
                      p={4}
                      borderBottom="1px"
                      borderColor={borderColor}
                      cursor="pointer"
                      bg={selectedPago?.id === pago.id ? 'blue.50' : 'transparent'}
                      _hover={{ bg: 'gray.50' }}
                      onClick={() => handleSelectPago(pago)}
                      transition="all 0.2s"
                    >
                      <HStack justify="space-between" mb={1}>
                        <Text fontWeight="bold" fontSize="sm">{alumno?.nombre}</Text>
                        <HStack>
                          {getValidationStatusIcon(pago.id)}
                          <Badge colorScheme="orange" fontSize="xs">Pendiente</Badge>
                        </HStack>
                      </HStack>
                      <Text fontSize="xs" color="gray.500" mb={1}>{pago.fecha_pago}</Text>
                      <Text fontWeight="bold" color="brand.600">${pago.monto.toLocaleString()}</Text>
                    </ListItem>
                  );
                })}
                {filteredPagos.length === 0 && (
                  <Box p={8} textAlign="center" color="gray.500">
                    <Text>No se encontraron pagos.</Text>
                  </Box>
                )}
              </List>
            </Box>
          </Box>

          {/* Right Panel: Detail & Validation */}
          <Box flex={1} bg="transparent" overflowY="auto">
            {selectedPago ? (
              <VStack spacing={6} align="stretch">

                {/* Header Info */}
                <Card borderRadius="xl" boxShadow="sm" border="1px" borderColor={borderColor}>
                  <CardBody>
                    <Flex justify="space-between" align="center">
                      <VStack align="start" spacing={1}>
                        <Heading size="md">
                          {mockAlumnos.find(a => a.id === selectedPago.id_alumno)?.nombre}
                        </Heading>
                        <Text color="gray.500" fontSize="sm">
                          Folio Alumno: {selectedPago.concepto}
                        </Text>
                      </VStack>
                      <HStack>
                        <Button leftIcon={<FiCheck />} colorScheme="green" onClick={handleAprobar}>
                          Aprobar
                        </Button>
                        <Button leftIcon={<FiX />} colorScheme="red" variant="outline" onClick={handleRechazar}>
                          Rechazar
                        </Button>
                      </HStack>
                    </Flex>
                  </CardBody>
                </Card>

                <SimpleGrid columns={{ base: 1, xl: 2 }} spacing={6}>

                  {/* Column 1: Receipt Image */}
                  <Card borderRadius="xl" boxShadow="sm" border="1px" borderColor={borderColor} h="fit-content">
                    <CardBody>
                      <Text fontWeight="bold" mb={4} display="flex" alignItems="center">
                        <Icon as={FiEye} mr={2} /> Comprobante Cargado
                      </Text>
                      <Box
                        borderRadius="lg"
                        overflow="hidden"
                        border="1px"
                        borderColor="gray.200"
                        maxH="500px"
                        overflowY="auto"
                      >
                        <Image
                          src={selectedPago.comprobante_url}
                          alt="Comprobante"
                          w="100%"
                          objectFit="contain"
                        />
                      </Box>
                    </CardBody>
                  </Card>

                  {/* Column 2: Validation Logic */}
                  <VStack spacing={6} align="stretch">

                    {/* Validation Results */}
                    <Card borderRadius="xl" boxShadow="sm" border="1px" borderColor={borderColor}>
                      <CardBody>
                        <Text fontWeight="bold" mb={4} display="flex" alignItems="center">
                          <Icon as={FiFileText} mr={2} /> Resultado de Pre-validación
                        </Text>

                        {!isFileUploaded ? (
                          <Alert status="info" borderRadius="md">
                            <AlertIcon />
                            <Text fontSize="sm">Carga el Estado de Cuenta (PDF) arriba para ver los resultados automáticos.</Text>
                          </Alert>
                        ) : (
                          <Box>
                            {validationResults[selectedPago.id]?.matchFound ? (
                              <Box>
                                <Alert status="success" borderRadius="md" mb={4}>
                                  <AlertIcon />
                                  <Box>
                                    <AlertTitle>Transacción Encontrada</AlertTitle>
                                    <AlertDescription fontSize="sm">
                                      Se ha vinculado con el movimiento del {validationResults[selectedPago.id].data.date}
                                    </AlertDescription>
                                  </Box>
                                </Alert>

                                <VStack align="stretch" spacing={0} divider={<Divider />}>
                                  <ValidationItem
                                    label="Monto Exacto"
                                    isValid={validationResults[selectedPago.id].checks.amount}
                                    value={`$${selectedPago.monto}`}
                                    statementValue={`$${validationResults[selectedPago.id].data.amount}`}
                                  />
                                  <ValidationItem
                                    label="Fecha y Hora"
                                    isValid={validationResults[selectedPago.id].checks.date}
                                    value={`${selectedPago.fecha_pago} ${selectedPago.hora_pago || ''}`}
                                    statementValue={`${validationResults[selectedPago.id].data.date} ${validationResults[selectedPago.id].data.time}`}
                                  />
                                  <ValidationItem
                                    label="Banco de Origen"
                                    isValid={validationResults[selectedPago.id].checks.bank}
                                    value={selectedPago.banco_origen}
                                    statementValue={validationResults[selectedPago.id].data.bank}
                                  />
                                  <ValidationItem
                                    label="Clave de Rastreo"
                                    isValid={validationResults[selectedPago.id].checks.tracking}
                                    value={selectedPago.clave_rastreo}
                                    statementValue={validationResults[selectedPago.id].data.trackingKey}
                                  />
                                  <ValidationItem
                                    label="Remitente"
                                    isValid={validationResults[selectedPago.id].checks.sender}
                                    value={selectedPago.nombre_remitente}
                                    statementValue={validationResults[selectedPago.id].data.sender}
                                  />
                                  <ValidationItem
                                    label="Concepto (Folio)"
                                    isValid={validationResults[selectedPago.id].checks.concept}
                                    value={selectedPago.concepto}
                                    statementValue={validationResults[selectedPago.id].data.concept}
                                  />
                                </VStack>

                                {Object.values(validationResults[selectedPago.id].checks).every(Boolean) ? (
                                  <Alert status="success" mt={4} borderRadius="md">
                                    <AlertIcon />
                                    <Text fontSize="sm" fontWeight="bold">¡Coincidencia Perfecta! El pago parece legítimo.</Text>
                                  </Alert>
                                ) : (
                                  <Alert status="warning" mt={4} borderRadius="md">
                                    <AlertIcon />
                                    <Text fontSize="sm" fontWeight="bold">Atención: Se detectaron discrepancias.</Text>
                                  </Alert>
                                )}
                              </Box>
                            ) : (
                              <Alert status="warning" borderRadius="md">
                                <AlertIcon />
                                <Box>
                                  <AlertTitle>Sin Coincidencias</AlertTitle>
                                  <AlertDescription fontSize="sm">
                                    No se encontró ninguna transacción en el estado de cuenta que coincida con este pago.
                                  </AlertDescription>
                                </Box>
                              </Alert>
                            )}
                          </Box>
                        )}
                      </CardBody>
                    </Card>

                    {/* Manual Data Review */}
                    <Card borderRadius="xl" boxShadow="sm" border="1px" borderColor={borderColor}>
                      <CardBody>
                        <Text fontWeight="bold" mb={4}>Datos Capturados por el Padre</Text>
                        <SimpleGrid columns={{ base: 1, sm: 2 }} spacing={4}>
                          <Box>
                            <Text fontSize="xs" color="gray.500">Banco</Text>
                            <Text fontSize="sm" fontWeight="medium">{selectedPago.banco_origen || 'No especificado'}</Text>
                          </Box>
                          <Box>
                            <Text fontSize="xs" color="gray.500">Clave Rastreo</Text>
                            <Text fontSize="sm" fontWeight="medium">{selectedPago.clave_rastreo || 'No especificado'}</Text>
                          </Box>
                          <Box>
                            <Text fontSize="xs" color="gray.500">Remitente</Text>
                            <Text fontSize="sm" fontWeight="medium">{selectedPago.nombre_remitente || 'No especificado'}</Text>
                          </Box>
                          <Box>
                            <Text fontSize="xs" color="gray.500">Concepto</Text>
                            <Badge colorScheme="purple">{selectedPago.concepto || 'N/A'}</Badge>
                          </Box>
                        </SimpleGrid>
                      </CardBody>
                    </Card>

                  </VStack>
                </SimpleGrid>
              </VStack>
            ) : (
              <Flex h="full" align="center" justify="center" direction="column" color="gray.400">
                <Icon as={FiSearch} boxSize={12} mb={4} />
                <Text fontSize="lg">Selecciona un pago de la lista para validar</Text>
              </Flex>
            )}
          </Box>
        </Flex>

        {/* Cash Payment Modal */}
        <Modal isOpen={isOpen} onClose={onClose} size="xl">
          <ModalOverlay />
          <ModalContent>
            <ModalHeader>
              <HStack>
                <Icon as={FiDollarSign} color="purple.500" />
                <Text>Registrar Pago en Efectivo</Text>
              </HStack>
            </ModalHeader>
            <ModalCloseButton />
            <ModalBody pb={6}>
              <VStack spacing={4} align="stretch">
                <FormControl isRequired>
                  <FormLabel>Alumno</FormLabel>
                  <Select
                    placeholder="Selecciona un alumno"
                    value={cashPaymentForm.id_alumno}
                    onChange={(e) => handleCashPaymentFormChange('id_alumno', e.target.value)}
                  >
                    {mockAlumnos.map(alumno => (
                      <option key={alumno.id} value={alumno.id}>
                        {alumno.nombre} - {alumno.folio}
                      </option>
                    ))}
                  </Select>
                </FormControl>

                <FormControl isRequired>
                  <FormLabel>Monto</FormLabel>
                  <InputGroup>
                    <InputLeftElement pointerEvents="none" color="gray.400">
                      $
                    </InputLeftElement>
                    <Input
                      type="number"
                      step="0.01"
                      placeholder="0.00"
                      value={cashPaymentForm.monto}
                      onChange={(e) => handleCashPaymentFormChange('monto', e.target.value)}
                    />
                  </InputGroup>
                </FormControl>

                <FormControl isRequired>
                  <FormLabel>Fecha de Pago</FormLabel>
                  <Input
                    type="date"
                    value={cashPaymentForm.fecha_pago}
                    onChange={(e) => handleCashPaymentFormChange('fecha_pago', e.target.value)}
                  />
                </FormControl>

                <FormControl isRequired>
                  <FormLabel>Concepto</FormLabel>
                  <Input
                    placeholder="Ej: Mensualidad Enero 2024"
                    value={cashPaymentForm.concepto}
                    onChange={(e) => handleCashPaymentFormChange('concepto', e.target.value)}
                  />
                </FormControl>

                <FormControl>
                  <FormLabel>Notas (Opcional)</FormLabel>
                  <Textarea
                    placeholder="Información adicional sobre el pago..."
                    value={cashPaymentForm.notas}
                    onChange={(e) => handleCashPaymentFormChange('notas', e.target.value)}
                    rows={3}
                  />
                </FormControl>

                <Alert status="info" borderRadius="md">
                  <AlertIcon />
                  <Box>
                    <AlertTitle fontSize="sm">Pago en Efectivo</AlertTitle>
                    <AlertDescription fontSize="xs">
                      Este pago será registrado como aprobado automáticamente.
                    </AlertDescription>
                  </Box>
                </Alert>
              </VStack>
            </ModalBody>

            <ModalFooter>
              <Button variant="ghost" mr={3} onClick={onClose}>
                Cancelar
              </Button>
              <Button
                colorScheme="purple"
                leftIcon={<FiCheck />}
                onClick={handleCashPaymentSubmit}
              >
                Registrar Pago
              </Button>
            </ModalFooter>
          </ModalContent>
        </Modal>
      </Box>
    </Layout>
  );
};

export default ValidacionPagos;
