import React from 'react';
import {
    Box,
    Container,
    Heading,
    Text,
    SimpleGrid,
    Card,
    CardBody,
    Badge,
    VStack,
    HStack,
    Icon,
    Avatar,
    Flex
} from '@chakra-ui/react';
import { FiCalendar, FiClock, FiUser, FiBookOpen } from 'react-icons/fi';
import { motion } from 'framer-motion';
import Layout from '../../components/Layout';
import { mockClases, mockMaterias, mockProfesores } from '../../data/mockData';

const MotionCard = motion(Card);

const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.1
        }
    }
};

const itemVariants = {
    hidden: { y: 20, opacity: 0 },
    visible: {
        y: 0,
        opacity: 1,
        transition: {
            type: "spring",
            stiffness: 100
        }
    }
};

const alumnoLinks = [
    { nombre: 'Dashboard', path: '/alumno' },
    { nombre: 'Mis Clases', path: '/alumno/clases' },
    { nombre: 'Bitácoras', path: '/alumno/bitacoras' },
    { nombre: 'Pagos', path: '/alumno/pagos' },
];

// Simulando alumno ID 1
const alumnoId = 1;
const misClases = mockClases.filter(c => c.id_alumno === alumnoId);

const diasOrdenados = ['Lunes', 'Martes', 'Miércoles', 'Jueves', 'Viernes', 'Sábado', 'Domingo'];

const ClaseCard = ({ clase }) => {
    const materia = mockMaterias.find(m => m.id === clase.id_materia);
    const profesor = mockProfesores.find(p => p.id === clase.id_profesor);

    return (
        <MotionCard
            variants={itemVariants}
            bg="white"
            borderRadius="2xl"
            boxShadow="sm"
            border="1px solid"
            borderColor="gray.100"
            whileHover={{ y: -5, boxShadow: 'lg' }}
            transition={{ duration: 0.2 }}
        >
            <CardBody>
                <Flex direction="column" gap={4}>
                    {/* Header */}
                    <Flex justify="space-between" align="start">
                        <VStack align="start" spacing={1}>
                            <Heading size="md" color="gray.800">
                                {materia?.nombre}
                            </Heading>
                            <Badge colorScheme="purple" borderRadius="full" px={2}>
                                {clase.dia_semana}
                            </Badge>
                        </VStack>
                        <Box
                            p={3}
                            bgGradient="linear(to-br, purple.400, purple.600)"
                            color="white"
                            borderRadius="xl"
                            boxShadow="0 4px 12px rgba(159, 122, 234, 0.3)"
                        >
                            <Icon as={FiBookOpen} boxSize={6} />
                        </Box>
                    </Flex>

                    {/* Time */}
                    <HStack spacing={3} color="gray.600">
                        <Icon as={FiClock} boxSize={5} />
                        <Text fontSize="md" fontWeight="600">
                            {clase.hora_inicio} - {clase.hora_fin}
                        </Text>
                    </HStack>

                    {/* Professor */}
                    <Flex
                        bg="gray.50"
                        p={3}
                        borderRadius="xl"
                        align="center"
                        gap={3}
                    >
                        <Avatar
                            size="sm"
                            src={profesor?.foto_url}
                            name={profesor?.nombre}
                        />
                        <VStack align="start" spacing={0} flex="1">
                            <Text fontSize="xs" color="gray.500" fontWeight="600">
                                Profesor
                            </Text>
                            <Text fontSize="sm" fontWeight="700" color="gray.800">
                                {profesor?.nombre}
                            </Text>
                        </VStack>
                    </Flex>
                </Flex>
            </CardBody>
        </MotionCard>
    );
};

const ClasesAlumno = () => {
    // Agrupar clases por día
    const clasesPorDia = diasOrdenados.map(dia => ({
        dia,
        clases: misClases.filter(c => c.dia_semana === dia)
    })).filter(grupo => grupo.clases.length > 0);

    return (
        <Layout
            links={alumnoLinks}
            userType="Alumno"
            userName="Sofía López"
            userPhoto="https://images.unsplash.com/photo-1517841905240-472988babdf9?ixlib=rb-1.2.1&auto=format&fit=facearea&facepad=2&w=256&h=256&q=80"
        >
            <Container maxW="container.xl" px={0}>
                <motion.div
                    initial="hidden"
                    animate="visible"
                    variants={containerVariants}
                >
                    <Box mb={8}>
                        <Heading size="lg" color="gray.800" mb={2}>
                            Mis Clases
                        </Heading>
                        <Text color="gray.600">
                            Horario semanal de tus clases
                        </Text>
                    </Box>

                    {/* Stats */}
                    <SimpleGrid columns={{ base: 1, md: 2 }} spacing={6} mb={8}>
                        <MotionCard
                            variants={itemVariants}
                            bg="white"
                            borderRadius="2xl"
                            p={6}
                            border="1px solid"
                            borderColor="gray.100"
                        >
                            <HStack spacing={4}>
                                <Box
                                    p={3}
                                    bgGradient="linear(to-br, blue.400, blue.600)"
                                    color="white"
                                    borderRadius="xl"
                                    boxShadow="0 4px 12px rgba(66, 153, 225, 0.3)"
                                >
                                    <Icon as={FiCalendar} boxSize={6} />
                                </Box>
                                <VStack align="start" spacing={0}>
                                    <Text fontSize="3xl" fontWeight="800" color="gray.800">
                                        {misClases.length}
                                    </Text>
                                    <Text fontSize="sm" color="gray.500" fontWeight="600">
                                        Clases por Semana
                                    </Text>
                                </VStack>
                            </HStack>
                        </MotionCard>

                        <MotionCard
                            variants={itemVariants}
                            bg="white"
                            borderRadius="2xl"
                            p={6}
                            border="1px solid"
                            borderColor="gray.100"
                        >
                            <HStack spacing={4}>
                                <Box
                                    p={3}
                                    bgGradient="linear(to-br, purple.400, purple.600)"
                                    color="white"
                                    borderRadius="xl"
                                    boxShadow="0 4px 12px rgba(159, 122, 234, 0.3)"
                                >
                                    <Icon as={FiBookOpen} boxSize={6} />
                                </Box>
                                <VStack align="start" spacing={0}>
                                    <Text fontSize="3xl" fontWeight="800" color="gray.800">
                                        {new Set(misClases.map(c => c.id_materia)).size}
                                    </Text>
                                    <Text fontSize="sm" color="gray.500" fontWeight="600">
                                        Materias Diferentes
                                    </Text>
                                </VStack>
                            </HStack>
                        </MotionCard>
                    </SimpleGrid>

                    {/* Classes by Day */}
                    <VStack align="stretch" spacing={8}>
                        {clasesPorDia.map(({ dia, clases }) => (
                            <Box key={dia}>
                                <Heading size="md" mb={4} color="gray.800">
                                    {dia}
                                </Heading>
                                <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={6}>
                                    {clases.map((clase) => (
                                        <ClaseCard key={clase.id} clase={clase} />
                                    ))}
                                </SimpleGrid>
                            </Box>
                        ))}
                    </VStack>
                </motion.div>
            </Container>
        </Layout>
    );
};

export default ClasesAlumno;
