import { useState, useMemo } from 'react';
import {
  Box,
  VStack,
  HStack,
  Text,
  Heading,
  Tabs,
  TabList,
  TabPanels,
  Tab,
  TabPanel,
  Badge,
  Button,
  Input,
  InputGroup,
  InputLeftElement,
  Select,
  Grid,
  Icon,
  Flex,
  useToast,
  SlideFade,
  IconButton,
  Tooltip
} from '@chakra-ui/react';
import {
  FiSearch,
  FiFilter,
  FiCheckCircle,
  FiMail,
  FiArchive,
  FiX,
  FiInbox
} from 'react-icons/fi';
import Layout from '../../components/Layout';
import NotificacionCard from '../../components/NotificacionCard';

const alumnoLinks = [
  { nombre: 'Dashboard', path: '/alumno' },
  { nombre: 'Mis Clases', path: '/alumno/clases' },
  { nombre: 'Bitácoras', path: '/alumno/bitacoras' },
  { nombre: 'Pagos', path: '/alumno/pagos' },
];

const initialNotificaciones = [
  {
    id: 1,
    titulo: 'Nueva bitácora disponible',
    mensaje: 'Tu profesor ha registrado una nueva bitácora de tu última clase. Revisa tu progreso y los comentarios.',
    fecha: '2024-01-20',
    leida: false,
    tipo: 'bitacora',
    archivada: false
  },
  {
    id: 2,
    titulo: '⏰ Recordatorio de Clase',
    html: `
      <p>¡Hola! Te recordamos tu próxima clase:</p>
      <div style="background: linear-gradient(135deg, #FED7AA 0%, #FCD34D 100%); padding: 16px; border-radius: 8px; margin: 16px 0;">
        <h3 style="color: #92400E; margin: 0 0 8px 0;">🎨 Pintura al Óleo</h3>
        <p style="color: #78350F; margin: 4px 0;"><strong>📅 Fecha:</strong> Lunes, 22 de enero</p>
        <p style="color: #78350F; margin: 4px 0;"><strong>🕐 Hora:</strong> 10:00 AM</p>
        <p style="color: #78350F; margin: 4px 0;"><strong>👤 Profesor:</strong> María González</p>
      </div>
      <p><strong>💡 No olvides traer:</strong> tus materiales de pintura al óleo y el lienzo que comenzaste la semana pasada.</p>
    `,
    fecha: '2024-01-19',
    leida: false,
    tipo: 'clase',
    archivada: false
  },
  {
    id: 3,
    titulo: 'Metas actualizadas',
    mensaje: 'Tu profesor ha actualizado tus metas y objetivos para este semestre. Revisa los nuevos desafíos que tienes por delante.',
    fecha: '2024-01-18',
    leida: true,
    tipo: 'meta',
    archivada: false
  },
  {
    id: 4,
    titulo: '🎉 ¡Felicitaciones! Has Completado tu Primer Proyecto',
    html: `
      <p>¡Excelente trabajo! Has completado tu primer proyecto de <strong>Dibujo a Lápiz</strong>.</p>
      <div style="background: linear-gradient(135deg, #D1FAE5 0%, #A7F3D0 100%); padding: 20px; border-radius: 12px; margin: 16px 0; text-align: center; border: 2px solid #10B981;">
        <h2 style="color: #065F46; margin: 0 0 8px 0; font-size: 22px;">🌟 ¡Felicidades! 🌟</h2>
        <p style="color: #047857; margin: 0; font-size: 16px;">Tu dedicación y esfuerzo están dando frutos</p>
      </div>
      <p><strong>Comentarios de tu profesor:</strong></p>
      <blockquote style="border-left: 4px solid #10B981; padding-left: 16px; margin: 12px 0; color: #047857; font-style: italic;">
        "Has mostrado un gran progreso en el manejo de sombras y proporciones. ¡Sigue así!"
      </blockquote>
      <p style="margin-top: 16px;"><strong>🎯 Próximo paso:</strong> Comenzarás con técnicas de acuarela la próxima semana.</p>
    `,
    fecha: '2024-01-17',
    leida: false,
    tipo: 'promocion',
    imagen: 'https://images.unsplash.com/photo-1513475382585-d06e58bcb0e0?w=800&h=400&fit=crop',
    archivada: false
  },
  {
    id: 5,
    titulo: '📚 Nuevo Taller: Técnicas de Ilustración Digital',
    html: `
      <p>¡Tenemos una excelente oportunidad para ti!</p>
      <h3>🎨 Taller de Ilustración Digital</h3>
      <p><strong>Duración:</strong> 4 sesiones (sábados)</p>
      <p><strong>Horario:</strong> 14:00 a 16:00</p>
      <p><strong>Incluye:</strong></p>
      <ul>
        <li>Uso de tableta gráfica durante el taller</li>
        <li>Software profesional incluido</li>
        <li>Certificado de participación</li>
        <li>Exposición de tu trabajo final</li>
      </ul>
      <div style="background: #E0F2FE; padding: 12px; border-radius: 8px; margin: 16px 0; border-left: 4px solid #0EA5E9;">
        <p style="margin: 0; color: #0C4A6E;"><strong>💰 Precio especial para alumnos actuales:</strong> $800 MXN (precio regular: $1,200)</p>
      </div>
      <p style="margin-top: 12px;"><strong>¡Inscríbete antes del 25 de enero y obtén un 10% adicional de descuento!</strong></p>
    `,
    fecha: '2024-01-16',
    leida: false,
    tipo: 'curso',
    imagen: 'https://images.unsplash.com/photo-1541961017774-22349e4a1262?w=800&h=400&fit=crop',
    archivada: false
  }
];

const NotificacionesAlumno = () => {
  const [notificaciones, setNotificaciones] = useState(initialNotificaciones);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterTipo, setFilterTipo] = useState('todas');
  const toast = useToast();

  // Estadísticas
  const stats = useMemo(() => {
    const noLeidas = notificaciones.filter(n => !n.leida && !n.archivada).length;
    const archivadas = notificaciones.filter(n => n.archivada).length;
    const total = notificaciones.length;
    return { noLeidas, archivadas, total };
  }, [notificaciones]);

  // Filtrar notificaciones
  const filteredNotificaciones = useMemo(() => {
    return notificaciones.filter(notif => {
      const matchesSearch = notif.titulo.toLowerCase().includes(searchTerm.toLowerCase()) ||
        (notif.mensaje && notif.mensaje.toLowerCase().includes(searchTerm.toLowerCase()));
      const matchesTipo = filterTipo === 'todas' || notif.tipo === filterTipo;
      return matchesSearch && matchesTipo;
    });
  }, [notificaciones, searchTerm, filterTipo]);

  // Separar por tabs
  const noLeidas = filteredNotificaciones.filter(n => !n.leida && !n.archivada);
  const archivadas = filteredNotificaciones.filter(n => n.archivada);
  const todas = filteredNotificaciones.filter(n => !n.archivada);

  // Marcar como leída
  const marcarComoLeida = (id) => {
    setNotificaciones(prev =>
      prev.map(n => n.id === id ? { ...n, leida: true } : n)
    );
  };

  // Marcar todas como leídas
  const marcarTodasLeidas = () => {
    setNotificaciones(prev =>
      prev.map(n => ({ ...n, leida: true }))
    );
    toast({
      title: 'Notificaciones marcadas',
      description: 'Todas las notificaciones han sido marcadas como leídas',
      status: 'success',
      duration: 3000,
      isClosable: true,
    });
  };

  // Archivar notificación
  const archivarNotificacion = (id) => {
    setNotificaciones(prev =>
      prev.map(n => n.id === id ? { ...n, archivada: true, leida: true } : n)
    );
    toast({
      title: 'Notificación archivada',
      status: 'info',
      duration: 2000,
      isClosable: true,
    });
  };

  const limpiarFiltros = () => {
    setSearchTerm('');
    setFilterTipo('todas');
  };

  const EmptyState = ({ icon, title, description }) => (
    <Box
      textAlign="center"
      py={16}
      px={6}
      bg="white"
      borderRadius="2xl"
      boxShadow="sm"
      border="1px dashed"
      borderColor="gray.200"
      display="flex"
      flexDirection="column"
      alignItems="center"
      justifyContent="center"
      minH="300px"
    >
      <Box
        p={4}
        bg="gray.50"
        borderRadius="full"
        mb={4}
        color="gray.400"
      >
        <Icon as={icon} boxSize={10} />
      </Box>
      <Text fontSize="xl" fontWeight="bold" color="gray.700" mb={2}>
        {title}
      </Text>
      <Text color="gray.500" maxW="sm">
        {description}
      </Text>
      {(searchTerm || filterTipo !== 'todas') && (
        <Button
          mt={6}
          variant="outline"
          colorScheme="pink"
          size="sm"
          onClick={limpiarFiltros}
          leftIcon={<FiX />}
        >
          Limpiar filtros
        </Button>
      )}
    </Box>
  );

  return (
    <Layout links={alumnoLinks} userType="Alumno" userName="Sofía López">
      <Box maxW="1200px" mx="auto">
        {/* Header con estadísticas */}
        <Box mb={8}>
          <Heading size="lg" mb={2} bgGradient="linear(to-r, pink.500, orange.400)" bgClip="text">
            📬 Centro de Notificaciones
          </Heading>
          <Text color="gray.600" mb={6}>Revisa tus avisos, recordatorios y logros</Text>

          <Grid templateColumns={{ base: '1fr', md: 'repeat(3, 1fr)' }} gap={4} mb={6}>
            {/* Stat: No leídas */}
            <Box
              bg="linear-gradient(135deg, #667eea 0%, #764ba2 100%)"
              p={5}
              borderRadius="2xl"
              color="white"
              boxShadow="lg"
              position="relative"
              overflow="hidden"
              transition="transform 0.2s"
              _hover={{ transform: 'translateY(-2px)' }}
            >
              <Box position="absolute" top={0} right={0} p={4} opacity={0.2}>
                <Icon as={FiMail} boxSize={24} transform="rotate(-15deg)" />
              </Box>
              <HStack spacing={4} position="relative" zIndex={1}>
                <Box p={3} bg="whiteAlpha.200" borderRadius="xl">
                  <Icon as={FiMail} boxSize={6} />
                </Box>
                <Box>
                  <Text fontSize="3xl" fontWeight="800" lineHeight="1">{stats.noLeidas}</Text>
                  <Text fontSize="sm" fontWeight="medium" opacity={0.9}>No leídas</Text>
                </Box>
              </HStack>
            </Box>

            {/* Stat: Total */}
            <Box
              bg="linear-gradient(135deg, #f093fb 0%, #f5576c 100%)"
              p={5}
              borderRadius="2xl"
              color="white"
              boxShadow="lg"
              position="relative"
              overflow="hidden"
              transition="transform 0.2s"
              _hover={{ transform: 'translateY(-2px)' }}
            >
              <Box position="absolute" top={0} right={0} p={4} opacity={0.2}>
                <Icon as={FiInbox} boxSize={24} transform="rotate(-15deg)" />
              </Box>
              <HStack spacing={4} position="relative" zIndex={1}>
                <Box p={3} bg="whiteAlpha.200" borderRadius="xl">
                  <Icon as={FiInbox} boxSize={6} />
                </Box>
                <Box>
                  <Text fontSize="3xl" fontWeight="800" lineHeight="1">{stats.total}</Text>
                  <Text fontSize="sm" fontWeight="medium" opacity={0.9}>Total recibidas</Text>
                </Box>
              </HStack>
            </Box>

            {/* Stat: Archivadas */}
            <Box
              bg="linear-gradient(135deg, #4facfe 0%, #00f2fe 100%)"
              p={5}
              borderRadius="2xl"
              color="white"
              boxShadow="lg"
              position="relative"
              overflow="hidden"
              transition="transform 0.2s"
              _hover={{ transform: 'translateY(-2px)' }}
            >
              <Box position="absolute" top={0} right={0} p={4} opacity={0.2}>
                <Icon as={FiArchive} boxSize={24} transform="rotate(-15deg)" />
              </Box>
              <HStack spacing={4} position="relative" zIndex={1}>
                <Box p={3} bg="whiteAlpha.200" borderRadius="xl">
                  <Icon as={FiArchive} boxSize={6} />
                </Box>
                <Box>
                  <Text fontSize="3xl" fontWeight="800" lineHeight="1">{stats.archivadas}</Text>
                  <Text fontSize="sm" fontWeight="medium" opacity={0.9}>Archivadas</Text>
                </Box>
              </HStack>
            </Box>
          </Grid>

          {/* Filtros y búsqueda */}
          <Flex
            direction={{ base: 'column', md: 'row' }}
            gap={4}
            mb={6}
            bg="white"
            p={4}
            borderRadius="2xl"
            boxShadow="sm"
            border="1px solid"
            borderColor="gray.100"
            alignItems="center"
          >
            <InputGroup flex={1}>
              <InputLeftElement pointerEvents="none">
                <Icon as={FiSearch} color="gray.400" />
              </InputLeftElement>
              <Input
                placeholder="Buscar por título o contenido..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                bg="gray.50"
                border="none"
                _focus={{ bg: 'white', boxShadow: 'outline' }}
                borderRadius="lg"
              />
              {searchTerm && (
                <InputGroup size="sm" width="auto" position="absolute" right={2} top={2} zIndex={2}>
                  <IconButton
                    size="xs"
                    aria-label="Limpiar búsqueda"
                    icon={<FiX />}
                    onClick={() => setSearchTerm('')}
                    variant="ghost"
                    isRound
                  />
                </InputGroup>
              )}
            </InputGroup>

            <HStack spacing={3} w={{ base: 'full', md: 'auto' }}>
              <Select
                value={filterTipo}
                onChange={(e) => setFilterTipo(e.target.value)}
                w={{ base: 'full', md: '200px' }}
                bg="gray.50"
                border="none"
                borderRadius="lg"
                icon={<FiFilter />}
                _focus={{ boxShadow: 'outline' }}
              >
                <option value="todas">Todos los tipos</option>
                <option value="bitacora">📝 Bitácoras</option>
                <option value="clase">📅 Clases</option>
                <option value="meta">🎯 Metas</option>
                <option value="curso">📚 Cursos</option>
                <option value="promocion">🎁 Logros</option>
              </Select>

              {stats.noLeidas > 0 && (
                <Tooltip label="Marcar todas como leídas">
                  <IconButton
                    icon={<FiCheckCircle />}
                    colorScheme="pink"
                    variant="ghost"
                    onClick={marcarTodasLeidas}
                    size="lg"
                    aria-label="Marcar todas como leídas"
                    isRound
                  />
                </Tooltip>
              )}
            </HStack>
          </Flex>
        </Box>

        {/* Tabs */}
        <Tabs colorScheme="pink" variant="soft-rounded" isLazy>
          <TabList mb={6} bg="white" p={2} borderRadius="2xl" boxShadow="sm" display="inline-flex">
            <Tab fontWeight="semibold" _selected={{ bg: 'pink.50', color: 'pink.600' }}>
              <HStack spacing={2}>
                <Icon as={FiMail} />
                <Text>No leídas</Text>
                {stats.noLeidas > 0 && (
                  <Badge colorScheme="pink" borderRadius="full" px={2}>{stats.noLeidas}</Badge>
                )}
              </HStack>
            </Tab>
            <Tab fontWeight="semibold" _selected={{ bg: 'gray.100', color: 'gray.800' }}>
              <HStack spacing={2}>
                <Icon as={FiInbox} />
                <Text>Todas</Text>
                <Badge colorScheme="gray" borderRadius="full" px={2}>{todas.length}</Badge>
              </HStack>
            </Tab>
            <Tab fontWeight="semibold" _selected={{ bg: 'orange.50', color: 'orange.600' }}>
              <HStack spacing={2}>
                <Icon as={FiArchive} />
                <Text>Archivadas</Text>
                {stats.archivadas > 0 && (
                  <Badge colorScheme="orange" borderRadius="full" px={2}>{stats.archivadas}</Badge>
                )}
              </HStack>
            </Tab>
          </TabList>

          <TabPanels>
            {/* Panel: No leídas */}
            <TabPanel px={0} py={0}>
              <SlideFade in={true} offsetY="20px">
                {noLeidas.length === 0 ? (
                  <EmptyState
                    icon={FiCheckCircle}
                    title="¡Todo al día!"
                    description={searchTerm || filterTipo !== 'todas' ? "No hay notificaciones sin leer con los filtros actuales." : "No tienes notificaciones sin leer. ¡Excelente!"}
                  />
                ) : (
                  <VStack align="stretch" spacing={4}>
                    {noLeidas.map((notif) => (
                      <NotificacionCard
                        key={notif.id}
                        notificacion={notif}
                        onMarcarLeida={() => marcarComoLeida(notif.id)}
                        onArchivar={() => archivarNotificacion(notif.id)}
                      />
                    ))}
                  </VStack>
                )}
              </SlideFade>
            </TabPanel>

            {/* Panel: Todas */}
            <TabPanel px={0} py={0}>
              <SlideFade in={true} offsetY="20px">
                {todas.length === 0 ? (
                  <EmptyState
                    icon={FiInbox}
                    title="Bandeja vacía"
                    description={searchTerm || filterTipo !== 'todas' ? "No se encontraron notificaciones con los filtros seleccionados." : "Aún no has recibido ninguna notificación."}
                  />
                ) : (
                  <VStack align="stretch" spacing={4}>
                    {todas.map((notif) => (
                      <NotificacionCard
                        key={notif.id}
                        notificacion={notif}
                        onMarcarLeida={() => marcarComoLeida(notif.id)}
                        onArchivar={() => archivarNotificacion(notif.id)}
                      />
                    ))}
                  </VStack>
                )}
              </SlideFade>
            </TabPanel>

            {/* Panel: Archivadas */}
            <TabPanel px={0} py={0}>
              <SlideFade in={true} offsetY="20px">
                {archivadas.length === 0 ? (
                  <EmptyState
                    icon={FiArchive}
                    title="Archivo vacío"
                    description={searchTerm || filterTipo !== 'todas' ? "No hay notificaciones archivadas que coincidan con tu búsqueda." : "No has archivado ninguna notificación todavía."}
                  />
                ) : (
                  <VStack align="stretch" spacing={4}>
                    {archivadas.map((notif) => (
                      <NotificacionCard
                        key={notif.id}
                        notificacion={notif}
                        onMarcarLeida={() => marcarComoLeida(notif.id)}
                        onArchivar={() => archivarNotificacion(notif.id)}
                      />
                    ))}
                  </VStack>
                )}
              </SlideFade>
            </TabPanel>
          </TabPanels>
        </Tabs>
      </Box>
    </Layout>
  );
};

export default NotificacionesAlumno;
