import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    Text,
    SimpleGrid,
    Card,
    CardHeader,
    CardBody,
    CardFooter,
    Button,
    Badge,
    VStack,
    HStack,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalBody,
    ModalFooter,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Textarea,
    useToast,
    Icon,
    Flex,
    Image
} from '@chakra-ui/react';
import { FiUpload, FiCheck, FiClock, FiFileText } from 'react-icons/fi';
import { getOrders, updateOrder } from '../../services/api';

const RegistroPago = () => {
    const [pendingOrders, setPendingOrders] = useState([]);
    const [selectedOrder, setSelectedOrder] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const [paymentDetails, setPaymentDetails] = useState({
        amount: '',
        reference: '',
        date: new Date().toISOString().split('T')[0],
        notes: '',
        file: null
    });
    const [isSubmitting, setIsSubmitting] = useState(false);
    const toast = useToast();

    useEffect(() => {
        fetchPendingOrders();
    }, []);

    const fetchPendingOrders = async () => {
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            if (user) {
                const allOrders = await getOrders();
                // Filter orders for this client that are PENDING (service status)
                // We show both unpaid and those in review
                const clientOrders = allOrders.filter(o => o.client_id === user.id && o.status === 'PENDING');
                setPendingOrders(clientOrders);
            }
        } catch (error) {
            console.error("Error fetching orders:", error);
            toast({
                title: 'Error al cargar órdenes',
                status: 'error',
                duration: 3000,
            });
        }
    };

    const handleSelectOrder = (order) => {
        setSelectedOrder(order);
        setPaymentDetails({
            amount: '',
            reference: order.payment_reference || '',
            date: new Date().toISOString().split('T')[0],
            notes: '',
            file: null
        });
        onOpen();
    };

    const handleFileChange = (e) => {
        const file = e.target.files[0];
        if (file) {
            setPaymentDetails({ ...paymentDetails, file });
        }
    };

    const handleSubmit = async () => {
        if (!paymentDetails.file || !paymentDetails.reference) {
            toast({
                title: 'Información incompleta',
                description: 'La referencia y el comprobante son obligatorios.',
                status: 'warning',
                duration: 3000,
            });
            return;
        }

        setIsSubmitting(true);

        try {
            // Upload file to backend
            const formData = new FormData();
            formData.append('proof', paymentDetails.file);

            const uploadRes = await fetch(`http://localhost:8081/api/orders/${selectedOrder.id}/payment-proof`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${sessionStorage.getItem('token')}`
                },
                body: formData
            });

            if (!uploadRes.ok) {
                throw new Error('Error al subir el comprobante');
            }

            // Clean object before sending
            // IMPORTANT: Remove payment_proof_url so we don't overwrite the one just saved by the upload endpoint
            const { employee_name, employee_photo_url, payment_proof_url, ...cleanOrder } = selectedOrder;

            // Update order with payment details
            await updateOrder(selectedOrder.id, {
                ...cleanOrder,
                payment_reference: paymentDetails.reference,
                payment_status: 'REVIEW', // Mark for review
            });

            toast({
                title: 'Pago Registrado',
                description: 'La evidencia ha sido enviada para validación.',
                status: 'success',
                duration: 4000,
            });

            onClose();
            fetchPendingOrders(); // Refresh list
        } catch (error) {
            console.error("Error registering payment:", error);
            toast({
                title: 'Error',
                description: 'No se pudo registrar el pago.',
                status: 'error',
                duration: 3000,
            });
        } finally {
            setIsSubmitting(false);
        }
    };

    const getPaymentBadge = (status) => {
        switch (status) {
            case 'REVIEW': return <Badge colorScheme="blue">En Validación</Badge>;
            case 'ACCEPTED': return <Badge colorScheme="green">Pagado</Badge>;
            case 'REJECTED': return <Badge colorScheme="red">Rechazado</Badge>;
            default: return <Badge colorScheme="orange">Pendiente de Pago</Badge>;
        }
    };

    return (
        <Box>
            <Heading mb={6}>Registro de Pagos</Heading>
            <Text mb={6} color="gray.600">
                Administra tus comprobantes de pago.
            </Text>

            {pendingOrders.length === 0 ? (
                <Card>
                    <CardBody textAlign="center" py={10}>
                        <Icon as={FiCheck} boxSize={12} color="green.500" mb={4} />
                        <Heading size="md" mb={2}>¡Todo al día!</Heading>
                        <Text color="gray.500">No tienes pagos pendientes.</Text>
                    </CardBody>
                </Card>
            ) : (
                <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={6}>
                    {pendingOrders.map(order => (
                        <Card key={order.id} borderTop="4px solid" borderColor={order.payment_status === 'REVIEW' ? 'blue.400' : 'orange.400'}>
                            <CardHeader>
                                <HStack justify="space-between">
                                    {getPaymentBadge(order.payment_status)}
                                    <Icon as={FiClock} color="gray.400" />
                                </HStack>
                                <Heading size="sm" mt={3}>{order.service_number}</Heading>
                                <Text fontSize="xs" color="gray.500">
                                    Solicitado: {new Date(order.request_date).toLocaleDateString()}
                                </Text>
                            </CardHeader>
                            <CardBody>
                                <Text fontSize="sm" mb={4}>
                                    {order.payment_status === 'REVIEW'
                                        ? 'Tu pago está siendo revisado por un administrador.'
                                        : 'Por favor sube tu comprobante para que el administrador valide tu servicio.'}
                                </Text>
                            </CardBody>
                            <CardFooter>
                                <Button
                                    leftIcon={<FiUpload />}
                                    colorScheme="brand"
                                    width="full"
                                    onClick={() => handleSelectOrder(order)}
                                    isDisabled={order.payment_status === 'REVIEW' || order.payment_status === 'ACCEPTED'}
                                >
                                    {order.payment_status === 'REJECTED' ? 'Reintentar Subida' : 'Subir Comprobante'}
                                </Button>
                            </CardFooter>
                        </Card>
                    ))}
                </SimpleGrid>
            )}

            {/* Modal de Registro */}
            <Modal isOpen={isOpen} onClose={onClose}>
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Registrar Pago</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <VStack spacing={4}>
                            <Box bg="blue.50" p={4} borderRadius="md" w="full">
                                <Text fontWeight="bold" fontSize="sm">Orden: {selectedOrder?.service_number}</Text>
                                {selectedOrder?.payment_status === 'REJECTED' && (
                                    <Text fontSize="xs" color="red.500" mt={1}>
                                        Su pago anterior fue rechazado. Por favor verifique el motivo y suba un nuevo comprobante.
                                    </Text>
                                )}
                            </Box>

                            <FormControl isRequired>
                                <FormLabel>Referencia de Pago</FormLabel>
                                <Input
                                    placeholder="Ej. Transferencia 123456"
                                    value={paymentDetails.reference}
                                    onChange={(e) => setPaymentDetails({ ...paymentDetails, reference: e.target.value })}
                                />
                            </FormControl>

                            <FormControl isRequired>
                                <FormLabel>Fecha del Pago</FormLabel>
                                <Input
                                    type="date"
                                    value={paymentDetails.date}
                                    onChange={(e) => setPaymentDetails({ ...paymentDetails, date: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Monto Pagado</FormLabel>
                                <Input
                                    type="number"
                                    placeholder="0.00"
                                    value={paymentDetails.amount}
                                    onChange={(e) => setPaymentDetails({ ...paymentDetails, amount: e.target.value })}
                                />
                            </FormControl>

                            <FormControl isRequired>
                                <FormLabel>Comprobante (Imagen/PDF)</FormLabel>
                                <Box
                                    border="2px dashed"
                                    borderColor="gray.300"
                                    borderRadius="md"
                                    p={6}
                                    textAlign="center"
                                    cursor="pointer"
                                    onClick={() => document.getElementById('file-upload').click()}
                                    _hover={{ bg: 'gray.50', borderColor: 'brand.500' }}
                                >
                                    <Input
                                        id="file-upload"
                                        type="file"
                                        display="none"
                                        onChange={handleFileChange}
                                        accept="image/*,.pdf"
                                    />
                                    <Icon as={FiUpload} boxSize={8} color="gray.400" mb={2} />
                                    <Text fontSize="sm" color="gray.600">
                                        {paymentDetails.file ? paymentDetails.file.name : 'Haz clic para seleccionar archivo'}
                                    </Text>
                                </Box>
                            </FormControl>

                            <FormControl>
                                <FormLabel>Notas</FormLabel>
                                <Textarea
                                    placeholder="Detalles adicionales..."
                                    value={paymentDetails.notes}
                                    onChange={(e) => setPaymentDetails({ ...paymentDetails, notes: e.target.value })}
                                />
                            </FormControl>
                        </VStack>
                    </ModalBody>
                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>Cancelar</Button>
                        <Button
                            colorScheme="brand"
                            isLoading={isSubmitting}
                            onClick={handleSubmit}
                            isDisabled={!paymentDetails.file || !paymentDetails.reference}
                        >
                            {selectedOrder?.payment_status === 'REJECTED' ? 'Modificar Comprobante' : 'Enviar Evidencia'}
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default RegistroPago;
