import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    SimpleGrid,
    Card,
    CardBody,
    CardHeader,
    Text,
    Button,
    Badge,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Textarea,
    useToast,
    VStack
} from '@chakra-ui/react';
import { getServicesByCompany, createOrder } from '../../services/api';

const SolicitarServicio = () => {
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const [selectedService, setSelectedService] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();
    const [requestDetails, setRequestDetails] = useState({
        date: '',
        address: '',
        notes: ''
    });

    useEffect(() => {
        const fetchServices = async () => {
            try {
                const user = JSON.parse(sessionStorage.getItem('user'));
                if (user && user.company_id) {
                    const data = await getServicesByCompany(user.company_id);
                    // Filter only active services
                    setServicesCatalog(data.filter(s => s.is_active));
                } else {
                    // Fallback or error handling if no company
                    console.error("No company ID found for user");
                    toast({
                        title: 'Error de sesión',
                        description: 'No se pudo identificar la empresa del usuario.',
                        status: 'error',
                        duration: 3000,
                    });
                }
            } catch (error) {
                console.error("Error fetching services:", error);
                toast({
                    title: 'Error al cargar servicios',
                    description: error.message,
                    status: 'error',
                    duration: 3000,
                });
            }
        };
        fetchServices();
    }, []);

    const handleSelectService = (service) => {
        setSelectedService(service);
        // Default notes with warranty info if applicable
        const warrantyInfo = service.has_warranty ? `Garantía incluida: ${service.warranty_months} meses.` : '';
        setRequestDetails({ ...requestDetails, notes: warrantyInfo });
        onOpen();
    };

    const handleSubmit = async () => {
        if (!requestDetails.date || !requestDetails.address) {
            toast({
                title: 'Error',
                description: 'Por favor completa la fecha y la dirección',
                status: 'error',
                duration: 2000,
            });
            return;
        }

        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            const fullNotes = `Dirección del Servicio: ${requestDetails.address}\n\nNota: ${requestDetails.notes}`;

            await createOrder({
                service_number: `REQ-${Date.now()}`,
                client_id: user.id,
                service_catalogo_id: selectedService.id,
                status: 'PENDING',
                request_date: requestDetails.date, // Using request_date as preferred date
                company_id: user.company_id,
                notes: fullNotes
            });

            toast({
                title: 'Solicitud enviada',
                description: `Hemos recibido tu solicitud para ${selectedService.name}. Será confirmada por el proveedor.`,
                status: 'success',
                duration: 4000,
            });
            onClose();
            setRequestDetails({ date: '', address: '', notes: '' });
        } catch (error) {
            toast({
                title: 'Error al enviar solicitud',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    // Agrupar servicios por categoría
    const categories = [...new Set(servicesCatalog.map(s => s.category))];

    return (
        <Box>
            <Heading mb={6}>Solicitar Nuevo Servicio</Heading>
            <Text mb={8} color="gray.600">
                Selecciona el servicio que necesitas de nuestro catálogo.
            </Text>

            {categories.map(category => (
                <Box key={category} mb={8}>
                    <Heading size="md" mb={4} color="brand.600">{category}</Heading>
                    <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={6}>
                        {servicesCatalog
                            .filter(s => s.category === category)
                            .map(service => (
                                <Card key={service.id} _hover={{ shadow: 'md', transform: 'translateY(-2px)', transition: 'all 0.2s' }}>
                                    <CardHeader pb={2}>
                                        <Heading size="sm">{service.name}</Heading>
                                        {service.has_warranty && (
                                            <Badge colorScheme="green" mt={2} fontSize="xs">
                                                Garantía: {service.warranty_months} Meses
                                            </Badge>
                                        )}
                                    </CardHeader>
                                    <CardBody>
                                        <Text fontSize="sm" color="gray.600" mb={4} noOfLines={3}>
                                            {service.description}
                                        </Text>
                                        <Button
                                            colorScheme="brand"
                                            width="full"
                                            variant="outline"
                                            onClick={() => handleSelectService(service)}
                                        >
                                            Solicitar
                                        </Button>
                                    </CardBody>
                                </Card>
                            ))}
                    </SimpleGrid>
                </Box>
            ))}

            {/* Modal de Solicitud */}
            <Modal isOpen={isOpen} onClose={onClose}>
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Solicitar {selectedService?.name}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <VStack spacing={4}>
                            <Text fontSize="sm" color="gray.600">
                                Completa los detalles para agendar tu servicio. Un asesor confirmará la disponibilidad.
                            </Text>

                            <FormControl isRequired>
                                <FormLabel>Fecha Preferente</FormLabel>
                                <Input
                                    type="date"
                                    min={new Date().toISOString().split('T')[0]}
                                    value={requestDetails.date}
                                    onChange={(e) => setRequestDetails({ ...requestDetails, date: e.target.value })}
                                />
                            </FormControl>

                            <FormControl isRequired>
                                <FormLabel>Dirección del Servicio</FormLabel>
                                <Input
                                    placeholder="Calle, Número, Colonia, CP..."
                                    value={requestDetails.address}
                                    onChange={(e) => setRequestDetails({ ...requestDetails, address: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Notas Adicionales</FormLabel>
                                <Textarea
                                    placeholder="Describe brevemente el problema o requerimientos especiales..."
                                    value={requestDetails.notes}
                                    onChange={(e) => setRequestDetails({ ...requestDetails, notes: e.target.value })}
                                />
                            </FormControl>
                        </VStack>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSubmit}>
                            Enviar Solicitud
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default SolicitarServicio;
