import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    Card,
    CardBody,
    Text,
    Badge,
    VStack,
    HStack,
    Divider,
    useToast,
    ButtonGroup,
    Button,
    SimpleGrid,
    IconButton,
    Flex,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    Tabs,
    TabList,
    TabPanels,
    Tab,
    TabPanel,
    Image,
    FormControl,
    FormLabel,
    Input
} from '@chakra-ui/react';
import { ChevronLeftIcon, ChevronRightIcon } from '@chakra-ui/icons';
import { FiEye, FiCamera } from "react-icons/fi";
import {
    getContractedServicesByEmployee,
    getServicesByCompany,
    getClients,
    updateOrder,
    getEvidencesByService
} from '../../services/api';
import { FilePond, registerPlugin } from "react-filepond";
import FilePondPluginImagePreview from "filepond-plugin-image-preview";
import FilePondPluginFileValidateType from "filepond-plugin-file-validate-type";
import "filepond/dist/filepond.min.css";
import "filepond-plugin-image-preview/dist/filepond-plugin-image-preview.css";

// Register FilePond plugins
registerPlugin(FilePondPluginImagePreview, FilePondPluginFileValidateType);

const CalendarioTrabajo = () => {
    // Calendar & Data State
    const [myServices, setMyServices] = useState([]);
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const [clients, setClients] = useState([]);
    const [viewMode, setViewMode] = useState('list'); // 'list' | 'month'
    const [currentDate, setCurrentDate] = useState(new Date());

    // Modal & Details State
    const { isOpen, onOpen, onClose } = useDisclosure();
    const [selectedService, setSelectedService] = useState(null);
    const [evidences, setEvidences] = useState([]);
    const [newEvidenceDescription, setNewEvidenceDescription] = useState("");
    const [newEvidenceFiles, setNewEvidenceFiles] = useState([]);

    const toast = useToast();

    useEffect(() => {
        const fetchData = async () => {
            try {
                const user = JSON.parse(sessionStorage.getItem('user'));
                if (user) {
                    const { getEmployeeByUserId } = await import('../../services/api');
                    const employee = await getEmployeeByUserId(user.id);

                    if (employee) {
                        const [services, companyServices, allClients] = await Promise.all([
                            getContractedServicesByEmployee(employee.id),
                            getServicesByCompany(user.company_id),
                            getClients()
                        ]);

                        setMyServices(services);
                        setServicesCatalog(companyServices);
                        setClients(allClients);
                    }
                }
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar calendario',
                    status: 'error',
                    duration: 3000,
                });
            }
        };
        fetchData();
    }, []);

    const getServiceName = (id) => servicesCatalog.find(s => s.id === id)?.name || 'Desconocido';
    const getClientName = (id) => clients.find(c => c.user_id === id)?.name || 'Desconocido';

    const getStatusLabel = (status) => {
        const statusMap = {
            'PENDING': 'Pendiente',
            'IN_PROGRESS': 'En Progreso',
            'FINISHED': 'Finalizado',
            'CANCELLED': 'Cancelado',
            'CERRADO_ADMIN': 'Cerrado',
            'IN_REVIEW': 'Reapertura'
        };
        return statusMap[status] || status;
    };

    // ---- DETAILS LOGIC ----
    const handleViewDetails = async (service) => {
        setSelectedService(service);
        try {
            const serviceEvidences = await getEvidencesByService(service.id);
            setEvidences(serviceEvidences);
        } catch (error) {
            console.error(error);
            setEvidences([]);
        }
        onOpen();
    };

    const handleStatusChange = async (newStatus) => {
        try {
            const updatedService = { ...selectedService, status: newStatus };
            await updateOrder(selectedService.id, updatedService);

            setMyServices(myServices.map(s => s.id === selectedService.id ? updatedService : s));
            setSelectedService(updatedService);

            toast({
                title: 'Estado actualizado',
                description: `El servicio ahora está: ${getStatusLabel(newStatus)}`,
                status: 'success',
                duration: 2000,
            });
        } catch (error) {
            toast({
                title: 'Error al actualizar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    const handleAddEvidence = async () => {
        if (newEvidenceFiles.length === 0) {
            toast({ title: "Error", description: "Debe seleccionar al menos una imagen", status: "error", duration: 3000 });
            return;
        }

        try {
            const formData = new FormData();
            formData.append("contracted_service_id", selectedService.id);
            formData.append("evidence_type", "PHOTO");

            newEvidenceFiles.forEach((file) => {
                formData.append("evidences", file.file);
                formData.append("descriptions", newEvidenceDescription || `Evidencia - ${file.file.name}`);
            });

            const response = await fetch("/api/evidences", {
                method: "POST",
                headers: { Authorization: `Bearer ${sessionStorage.getItem("token")}` },
                body: formData,
            });

            if (!response.ok) throw new Error("Error al subir evidencias");

            const updatedEvidences = await getEvidencesByService(selectedService.id);
            setEvidences(updatedEvidences);
            setNewEvidenceDescription("");
            setNewEvidenceFiles([]);

            toast({ title: "Evidencia agregada", status: "success", duration: 2000 });
        } catch (error) {
            toast({ title: "Error", description: error.message, status: "error", duration: 3000 });
        }
    };

    // ---- LOGIC FOR LIST VIEW ----
    const renderListView = () => {
        const today = new Date().toISOString().split('T')[0];
        const todayServices = myServices.filter(s => s.deadline_date && s.deadline_date.startsWith(today));
        const futureServices = myServices.filter(s => s.deadline_date && s.deadline_date > today);

        return (
            <VStack spacing={6} align="stretch">
                <Box>
                    <Heading size="md" mb={4} color="brand.600">Para Hoy ({today})</Heading>
                    {todayServices.length > 0 ? (
                        todayServices.map(service => (
                            <Card key={service.id} mb={3} borderLeft="4px solid" borderColor="brand.500">
                                <CardBody>
                                    <HStack justify="space-between">
                                        <Box>
                                            <Heading size="sm">{getServiceName(service.service_catalogo_id)}</Heading>
                                            <Text fontSize="sm" color="gray.600">Folio: {service.service_number}</Text>
                                        </Box>
                                        <HStack>
                                            <Badge>{getStatusLabel(service.status)}</Badge>
                                            <Button size="sm" leftIcon={<FiEye />} onClick={() => handleViewDetails(service)}>
                                                Detalles
                                            </Button>
                                        </HStack>
                                    </HStack>
                                </CardBody>
                            </Card>
                        ))
                    ) : (
                        <Text color="gray.500">No hay servicios programados para hoy.</Text>
                    )}
                </Box>
                <Divider />
                <Box>
                    <Heading size="md" mb={4}>Próximos Días</Heading>
                    {futureServices.length > 0 ? (
                        futureServices.map(service => (
                            <Card key={service.id} mb={3}>
                                <CardBody>
                                    <HStack justify="space-between">
                                        <Box>
                                            <Text fontSize="xs" fontWeight="bold" color="gray.500">
                                                {new Date(service.deadline_date).toLocaleDateString()}
                                            </Text>
                                            <Heading size="sm">{getServiceName(service.service_catalogo_id)}</Heading>
                                        </Box>
                                        <HStack>
                                            <Badge colorScheme="gray">{getStatusLabel(service.status)}</Badge>
                                            <Button size="sm" onClick={() => handleViewDetails(service)}>Ver</Button>
                                        </HStack>
                                    </HStack>
                                </CardBody>
                            </Card>
                        ))
                    ) : (
                        <Text color="gray.500">No hay servicios futuros asignados.</Text>
                    )}
                </Box>
            </VStack>
        );
    };

    // ---- LOGIC FOR MONTH VIEW ----
    const getDaysInMonth = (year, month) => {
        return new Date(year, month + 1, 0).getDate();
    };

    const getMonthData = () => {
        const year = currentDate.getFullYear();
        const month = currentDate.getMonth();

        const daysInMonth = getDaysInMonth(year, month);

        // Find which day of week the 1st of month is.
        // JS Date: 0 = Sunday, 1 = Monday.
        // We want Monday = 0, Sunday = 6.
        let firstDay = new Date(year, month, 1).getDay();
        // Shift distinct: JS Sun(0) -> We want 6. JS Mon(1) -> We want 0.
        // Formula: (day + 6) % 7
        const startDayIndex = (firstDay + 6) % 7;

        return { year, month, daysInMonth, startDayIndex };
    };

    const changeMonth = (offset) => {
        const newDate = new Date(currentDate);
        newDate.setMonth(newDate.getMonth() + offset);
        setCurrentDate(newDate);
    };

    const renderMonthView = () => {
        const { year, month, daysInMonth, startDayIndex } = getMonthData();
        const monthName = currentDate.toLocaleString('es-ES', { month: 'long', year: 'numeric' });
        const weekDays = ['Lun', 'Mar', 'Mié', 'Jue', 'Vie', 'Sáb', 'Dom'];

        // Cells logic
        const cells = [];
        // Empty cells for start padding
        for (let i = 0; i < startDayIndex; i++) {
            cells.push(<Box key={`empty-${i}`} height="100px" bg="gray.50" />);
        }
        // Day cells
        for (let day = 1; day <= daysInMonth; day++) {
            const dateStr = `${year}-${String(month + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
            // Find tasks for this day
            const tasks = myServices.filter(s => s.deadline_date && s.deadline_date.startsWith(dateStr));
            const isToday = new Date().toDateString() === new Date(year, month, day).toDateString();

            cells.push(
                <Box
                    key={day}
                    height="120px"
                    borderWidth="1px"
                    p={1}
                    bg={isToday ? "blue.50" : "white"}
                    position="relative"
                    overflow="hidden"
                >
                    <Text fontWeight="bold" fontSize="sm" color={isToday ? "blue.600" : "gray.700"}>
                        {day}
                    </Text>
                    <VStack align="start" spacing={0} mt={1}>
                        {tasks.map(task => (
                            <Box
                                key={task.id}
                                w="100%"
                                bg="brand.100"
                                p="2px"
                                borderRadius="sm"
                                fontSize="xs"
                                isTruncated
                                cursor="pointer"
                                onClick={() => handleViewDetails(task)}
                                _hover={{ bg: "brand.200" }}
                            >
                                <Text noOfLines={1} fontSize="10px">
                                    {getServiceName(task.service_catalogo_id)}
                                </Text>
                            </Box>
                        ))}
                    </VStack>
                    {tasks.length > 2 && (
                        <Text fontSize="xs" color="gray.500" position="absolute" bottom="1" right="1">
                            +{tasks.length - 2} más
                        </Text>
                    )}
                </Box>
            );
        }

        return (
            <Box>
                {/* Header Navigation */}
                <Flex justify="space-between" align="center" mb={4}>
                    <IconButton
                        icon={<ChevronLeftIcon />}
                        onClick={() => changeMonth(-1)}
                        aria-label="Mes anterior"
                    />
                    <Heading size="md" textTransform="capitalize">{monthName}</Heading>
                    <IconButton
                        icon={<ChevronRightIcon />}
                        onClick={() => changeMonth(1)}
                        aria-label="Mes siguiente"
                    />
                </Flex>

                {/* Grid */}
                <SimpleGrid columns={7} spacing={1}>
                    {weekDays.map(d => (
                        <Box key={d} textAlign="center" fontWeight="bold" py={2} bg="gray.100">
                            {d}
                        </Box>
                    ))}
                    {cells}
                </SimpleGrid>
            </Box>
        );
    };

    return (
        <Box>
            <Flex justify="space-between" align="center" mb={6}>
                <Heading>Calendario de Trabajo</Heading>
                <ButtonGroup isAttached variant="outline">
                    <Button
                        colorScheme={viewMode === 'list' ? 'brand' : 'gray'}
                        variant={viewMode === 'list' ? 'solid' : 'outline'}
                        onClick={() => setViewMode('list')}
                    >
                        Lista
                    </Button>
                    <Button
                        colorScheme={viewMode === 'month' ? 'brand' : 'gray'}
                        variant={viewMode === 'month' ? 'solid' : 'outline'}
                        onClick={() => setViewMode('month')}
                    >
                        Mes
                    </Button>
                </ButtonGroup>
            </Flex>

            {viewMode === 'list' ? renderListView() : renderMonthView()}

            {/* DETAILS MODAL */}
            <Modal isOpen={isOpen} onClose={onClose} size="xl">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Detalles del Servicio</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        {selectedService && (
                            <Tabs isFitted variant="enclosed">
                                <TabList mb="1em">
                                    <Tab>Información</Tab>
                                    <Tab>Evidencias</Tab>
                                </TabList>
                                <TabPanels>
                                    <TabPanel>
                                        <VStack align="stretch" spacing={4}>
                                            <Box>
                                                <Text fontWeight="bold" color="gray.500" fontSize="sm">Servicio</Text>
                                                <Text fontSize="lg">{getServiceName(selectedService.service_catalogo_id)}</Text>
                                            </Box>
                                            <Box>
                                                <Text fontWeight="bold" color="gray.500" fontSize="sm">Cliente</Text>
                                                <Text fontSize="lg">{getClientName(selectedService.client_id)}</Text>
                                            </Box>
                                            <Box>
                                                <Text fontWeight="bold" color="gray.500" fontSize="sm">Estado Actual</Text>
                                                <Badge fontSize="md" colorScheme={selectedService.status === 'FINISHED' ? 'green' : selectedService.status === 'IN_PROGRESS' ? 'blue' : 'orange'}>
                                                    {getStatusLabel(selectedService.status)}
                                                </Badge>
                                            </Box>

                                            {selectedService.status === 'IN_REVIEW' && (
                                                <Box mt={4} p={3} bg="orange.50" borderRadius="md" borderWidth="1px" borderColor="orange.200">
                                                    <Heading size="sm" color="orange.800" mb={2}>Motivo de Reapertura</Heading>
                                                    {selectedService.client_comment && <Text fontSize="sm" color="orange.700" mb={3}>{selectedService.client_comment.replace("Motivo de reapertura: ", "")}</Text>}
                                                </Box>
                                            )}

                                            <Box pt={4}>
                                                {['CERRADO_ADMIN', 'CANCELLED'].includes(selectedService.status) ? (
                                                    <Box p={3} bg="red.50" borderRadius="md" borderWidth="1px" borderColor="red.200">
                                                        <Text fontSize="sm" color="red.700" fontWeight="bold">⛔ Servicio finalizado/cancelado.</Text>
                                                    </Box>
                                                ) : (
                                                    <HStack>
                                                        <Button colorScheme="blue" isDisabled={selectedService.status === 'IN_PROGRESS'} onClick={() => handleStatusChange('IN_PROGRESS')}>Iniciar</Button>
                                                        <Button colorScheme="green" isDisabled={selectedService.status === 'FINISHED'} onClick={() => handleStatusChange('FINISHED')}>Finalizar</Button>
                                                    </HStack>
                                                )}
                                            </Box>
                                        </VStack>
                                    </TabPanel>
                                    <TabPanel>
                                        <VStack align="stretch" spacing={6}>
                                            <Box>
                                                <Heading size="sm" mb={3}>Evidencias</Heading>
                                                {evidences.length > 0 ? (
                                                    <SimpleGrid columns={2} spacing={4}>
                                                        {evidences.map(ev => (
                                                            <Box key={ev.id} border="1px solid" borderColor="gray.200" borderRadius="md" p={2}>
                                                                <Image src={ev.file_url} alt="Evidencia" borderRadius="sm" mb={2} />
                                                                <Text fontSize="xs">{ev.description}</Text>
                                                            </Box>
                                                        ))}
                                                    </SimpleGrid>
                                                ) : <Text color="gray.500">Sin evidencias.</Text>}
                                            </Box>
                                            <Box borderTop="1px solid" borderColor="gray.200" pt={4}>
                                                {!['CERRADO_ADMIN', 'CANCELLED'].includes(selectedService.status) && (
                                                    <>
                                                        <Heading size="sm" mb={3}>Agregar Evidencia</Heading>
                                                        <VStack spacing={3}>
                                                            <FormControl>
                                                                <FormLabel fontSize="sm">Descripción</FormLabel>
                                                                <Input value={newEvidenceDescription} onChange={(e) => setNewEvidenceDescription(e.target.value)} placeholder="Descripción..." />
                                                            </FormControl>
                                                            <FormControl>
                                                                <FilePond
                                                                    files={newEvidenceFiles}
                                                                    onupdatefiles={setNewEvidenceFiles}
                                                                    allowMultiple={true}
                                                                    maxFiles={4}
                                                                    acceptedFileTypes={["image/*"]}
                                                                    labelIdle='Arrastra fotos aquí'
                                                                    credits={false}
                                                                />
                                                            </FormControl>
                                                            <Button leftIcon={<FiCamera />} colorScheme="brand" width="full" onClick={handleAddEvidence} isDisabled={newEvidenceFiles.length === 0}>Subir Evidencia</Button>
                                                        </VStack>
                                                    </>
                                                )}
                                            </Box>
                                        </VStack>
                                    </TabPanel>
                                </TabPanels>
                            </Tabs>
                        )}
                    </ModalBody>
                    <ModalFooter>
                        <Button onClick={onClose}>Cerrar</Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default CalendarioTrabajo;
