import { useState, useEffect } from 'react';
import {
    Box,
    SimpleGrid,
    Stat,
    StatLabel,
    StatNumber,
    StatHelpText,
    Card,
    CardBody,
    Heading,
    Text,
    Icon,
    Flex,
    Badge,
    VStack,
    HStack,
    useToast
} from '@chakra-ui/react';
import { FiClipboard, FiCheckCircle, FiClock, FiAlertCircle } from 'react-icons/fi';
import { getContractedServicesByEmployee, getClients, getServicesByCompany } from '../../services/api';

const EmployeeDashboard = () => {
    const [myServices, setMyServices] = useState([]);
    const [clients, setClients] = useState([]);
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const toast = useToast();

    useEffect(() => {
        const fetchData = async () => {
            try {
                const user = JSON.parse(sessionStorage.getItem('user'));
                // Assuming user object has user_id or similar that maps to employee_id
                // In LoginPanel, we store user. For Employee, user.id is the user_id (e.g. 'emp-123')
                // But getContractedServicesByEmployee expects employee_id (int) or user_id?
                // The API getContractedServicesByEmployee filters by employee_id (int).
                // We need to get the employee record first to know their ID, or assume user.id IS the employee_id if we stored it that way.
                // Let's check LoginPanel. It stores the whole user object.
                // The backend user model has id, email, role.
                // The employee model has id, user_id, company_id.
                // We need to fetch employee details using user.id to get the employee.id (int).
                // Wait, api.js getContractedServicesByEmployee filters by employee_id (int).
                // Let's look at api.js again.
                // export const getContractedServicesByEmployee = async (employeeId) => { ... return orders.filter(o => o.employee_id === parseInt(employeeId)); };

                // We need to find the employee record for the current logged in user.
                // We can use getEmployeeByUserId(user.id).

                if (user) {
                    const { getEmployeeByUserId } = await import('../../services/api');
                    const employee = await getEmployeeByUserId(user.id);

                    if (employee) {
                        const [services, allClients, companyServices] = await Promise.all([
                            getContractedServicesByEmployee(employee.id), // <-- Corregido: usar employee.id (numérico)
                            getClients(),
                            getServicesByCompany(user.company_id)
                        ]);

                        setMyServices(services);
                        setClients(allClients);
                        setServicesCatalog(companyServices);
                    }
                }
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar dashboard',
                    status: 'error',
                    duration: 3000,
                });
            }
        };
        fetchData();
    }, []);

    const activeServices = myServices.filter(s => s.status === 'IN_PROGRESS').length;
    const pendingServices = myServices.filter(s => s.status === 'PENDING').length;
    const finishedServices = myServices.filter(s => s.status === 'FINISHED').length;

    const getClientName = (id) => clients.find(c => c.user_id === id)?.name || 'Desconocido';
    const getServiceName = (id) => servicesCatalog.find(s => s.id === id)?.name || 'Desconocido';

    return (
        <Box>
            <Heading mb={6}>Mi Panel de Trabajo</Heading>

            {/* Stats Grid */}
            <SimpleGrid columns={{ base: 1, md: 3 }} spacing={6} mb={8}>
                <Card>
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="blue.100" borderRadius="md" mr={3}>
                                    <Icon as={FiClipboard} color="blue.500" boxSize={6} />
                                </Box>
                                <StatLabel fontSize="lg">En Progreso</StatLabel>
                            </Flex>
                            <StatNumber>{activeServices}</StatNumber>
                            <StatHelpText>Tareas activas</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card>
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="orange.100" borderRadius="md" mr={3}>
                                    <Icon as={FiClock} color="orange.500" boxSize={6} />
                                </Box>
                                <StatLabel fontSize="lg">Pendientes</StatLabel>
                            </Flex>
                            <StatNumber>{pendingServices}</StatNumber>
                            <StatHelpText>Por iniciar</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card>
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="green.100" borderRadius="md" mr={3}>
                                    <Icon as={FiCheckCircle} color="green.500" boxSize={6} />
                                </Box>
                                <StatLabel fontSize="lg">Finalizados</StatLabel>
                            </Flex>
                            <StatNumber>{finishedServices}</StatNumber>
                            <StatHelpText>Este mes</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>
            </SimpleGrid>

            {/* Active Tasks */}
            <Heading size="md" mb={4}>Tareas Prioritarias</Heading>
            <VStack spacing={4} align="stretch">
                {myServices
                    .filter(s => s.status === 'IN_PROGRESS' || s.status === 'PENDING')
                    .map((service) => (
                        <Card key={service.id} borderLeft="4px solid" borderColor={service.status === 'IN_PROGRESS' ? 'blue.500' : 'orange.500'}>
                            <CardBody>
                                <Flex justify="space-between" align="center" wrap="wrap" gap={4}>
                                    <Box>
                                        <HStack mb={1}>
                                            <Badge colorScheme={service.status === 'IN_PROGRESS' ? 'blue' : 'orange'}>
                                                {service.status === 'IN_PROGRESS' ? 'EN PROGRESO' : 'PENDIENTE'}
                                            </Badge>
                                            <Text fontSize="sm" color="gray.500">{service.service_number}</Text>
                                        </HStack>
                                        <Heading size="sm" mb={1}>{getServiceName(service.service_catalogo_id)}</Heading>
                                        <Text fontSize="sm" color="gray.600">Cliente: {getClientName(service.client_id)}</Text>
                                    </Box>
                                    <Box textAlign="right">
                                        <Text fontSize="xs" color="gray.500">Fecha Límite</Text>
                                        <Text fontWeight="bold" color={new Date(service.deadline_date) < new Date() ? 'red.500' : 'gray.700'}>
                                            {new Date(service.deadline_date).toLocaleDateString()}
                                        </Text>
                                    </Box>
                                </Flex>
                            </CardBody>
                        </Card>
                    ))}
                {myServices.filter(s => s.status === 'IN_PROGRESS' || s.status === 'PENDING').length === 0 && (
                    <Card>
                        <CardBody textAlign="center" py={8}>
                            <Icon as={FiCheckCircle} boxSize={12} color="green.500" mb={4} />
                            <Heading size="md" mb={2}>¡Todo al día!</Heading>
                            <Text color="gray.500">No tienes tareas pendientes por ahora.</Text>
                        </CardBody>
                    </Card>
                )}
            </VStack>
        </Box>
    );
};

export default EmployeeDashboard;
