import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Select,
    useToast,
    HStack,
    Text,
    Textarea,
    VStack,
    Image,
    Tabs,
    TabList,
    TabPanels,
    Tab,
    TabPanel,
    SimpleGrid,
} from "@chakra-ui/react";
import { FiEye, FiCamera, FiTrash2 } from "react-icons/fi";
import { getContractedServicesByEmployee, getClients, getServicesByCompany, updateOrder, getEvidencesByService, BASE_URL } from '../../services/api';
import { FilePond, registerPlugin } from "react-filepond";
import FilePondPluginImagePreview from "filepond-plugin-image-preview";
import FilePondPluginFileValidateType from "filepond-plugin-file-validate-type";
import "filepond/dist/filepond.min.css";
import "filepond-plugin-image-preview/dist/filepond-plugin-image-preview.css";

// Register FilePond plugins
registerPlugin(FilePondPluginImagePreview, FilePondPluginFileValidateType);

// Función para traducir estados a leyendas amigables
const getStatusLabel = (status) => {
    const statusMap = {
        'PENDING': 'Pendiente',
        'IN_PROGRESS': 'En Progreso',
        'FINISHED': 'Finalizado',
        'CANCELLED': 'Cancelado',
        'CERRADO_ADMIN': 'Cerrado',
        'IN_REVIEW': 'Reapertura'
    };
    return statusMap[status] || status;
};

const MisAsignaciones = () => {
    const [services, setServices] = useState([]);
    const [clients, setClients] = useState([]);
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const [selectedService, setSelectedService] = useState(null);
    const [evidences, setEvidences] = useState([]);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    // Estado para nueva evidencia
    const [newEvidenceDescription, setNewEvidenceDescription] = useState("");
    const [newEvidenceFiles, setNewEvidenceFiles] = useState([]);

    const fetchData = async () => {
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            if (user) {
                const { getEmployeeByUserId } = await import('../../services/api');
                const employee = await getEmployeeByUserId(user.id);

                if (employee) {
                    const [myServices, allClients, companyServices] = await Promise.all([
                        getContractedServicesByEmployee(employee.id),
                        getClients(),
                        getServicesByCompany(user.company_id)
                    ]);

                    // Ordenar los servicios
                    const sortedServices = myServices.sort((a, b) => {
                        const statusOrder = {
                            'PENDING': 1,
                            'IN_REVIEW': 2,
                            'IN_PROGRESS': 3,
                            'FINISHED': 4,
                            'CERRADO_ADMIN': 5,
                            'CANCELLED': 6,
                        };

                        const orderA = statusOrder[a.status] || 99;
                        const orderB = statusOrder[b.status] || 99;

                        if (orderA !== orderB) {
                            return orderA - orderB;
                        }

                        // Si el estatus es el mismo, ordenar por fecha más reciente
                        return new Date(b.request_date) - new Date(a.request_date);
                    });

                    setServices(sortedServices);
                    setClients(allClients);
                    setServicesCatalog(companyServices);
                }
            }
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar asignaciones',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchData();
    }, []);

    const getClientName = (id) => clients.find(c => c.user_id === id)?.name || 'Desconocido';
    const getServiceName = (id) => servicesCatalog.find(s => s.id === id)?.name || 'Desconocido';

    const handleViewDetails = async (service) => {
        setSelectedService(service);
        try {
            const serviceEvidences = await getEvidencesByService(service.id);
            setEvidences(serviceEvidences);
        } catch (error) {
            console.error(error);
            setEvidences([]);
        }
        onOpen();
    };

    const handleStatusChange = async (newStatus) => {
        // Validación: No permitir finalizar sin evidencias
        if (newStatus === 'FINISHED') {
            const hasEvidence = evidences && evidences.length > 0;
            if (!hasEvidence) {
                toast({
                    title: 'Acción requerida',
                    description: 'Debes subir al menos una evidencia (foto) del trabajo realizado antes de finalizar el servicio.',
                    status: 'warning',
                    duration: 5000,
                    isClosable: true,
                });
                return;
            }
        }

        try {
            const updatedService = { ...selectedService, status: newStatus };
            await updateOrder(selectedService.id, updatedService);

            setServices(services.map(s => s.id === selectedService.id ? updatedService : s));
            setSelectedService(updatedService);

            toast({
                title: 'Estado actualizado',
                description: `El servicio ahora está: ${newStatus}`,
                status: 'success',
                duration: 2000,
            });
        } catch (error) {
            toast({
                title: 'Error al actualizar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    const handleAddEvidence = async () => {
        if (newEvidenceFiles.length === 0) {
            toast({
                title: "Error",
                description: "Debe seleccionar al menos una imagen",
                status: "error",
                duration: 3000,
            });
            return;
        }

        try {
            const formData = new FormData();
            formData.append("contracted_service_id", selectedService.id);
            formData.append("evidence_type", "PHOTO");

            newEvidenceFiles.forEach((file) => {
                formData.append("evidences", file.file);
                formData.append(
                    "descriptions",
                    newEvidenceDescription || `Evidencia - ${file.file.name}`,
                );
            });

            const response = await fetch("/api/evidences", {
                method: "POST",
                headers: {
                    Authorization: `Bearer ${sessionStorage.getItem("token")}`,
                },
                body: formData,
            });

            if (!response.ok) {
                const errorText = await response.text();
                try {
                    const errorJson = JSON.parse(errorText);
                    throw new Error(errorJson.error || "Error al subir evidencias");
                } catch (e) {
                    throw new Error(errorText || "Error en el servidor");
                }
            }

            const updatedEvidences = await getEvidencesByService(selectedService.id);
            setEvidences(updatedEvidences);
            setNewEvidenceDescription("");
            setNewEvidenceFiles([]);

            toast({
                title: "Evidencia agregada",
                status: "success",
                duration: 2000,
            });
        } catch (error) {
            toast({
                title: "Error al agregar evidencia",
                description: error.message,
                status: "error",
                duration: 3000,
            });
        }
    };

    const serviceEvidences = evidences; // Already filtered by fetching for specific service

    return (
        <Box>
            <Heading mb={6}>Mis Asignaciones</Heading>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Folio</Th>
                            <Th>Servicio</Th>
                            <Th>Cliente</Th>
                            <Th>Fecha Límite</Th>
                            <Th>Estado</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {services.map((service) => (
                            <Tr key={service.id}>
                                <Td fontWeight="bold">{service.service_number}</Td>
                                <Td>{getServiceName(service.service_catalogo_id)}</Td>
                                <Td>{getClientName(service.client_id)}</Td>
                                <Td>{new Date(service.deadline_date).toLocaleDateString()}</Td>
                                <Td>
                                    <Badge
                                        colorScheme={
                                            service.status === 'FINISHED' ? 'green' :
                                                service.status === 'IN_PROGRESS' ? 'blue' :
                                                    service.status === 'PENDING' ? 'orange' :
                                                        service.status === 'IN_REVIEW' ? 'purple' : 'gray'
                                        }
                                    >
                                        {getStatusLabel(service.status)}
                                    </Badge>
                                </Td>
                                <Td>
                                    <Button
                                        leftIcon={<FiEye />}
                                        size="sm"
                                        colorScheme="brand"
                                        variant="outline"
                                        onClick={() => handleViewDetails(service)}
                                    >
                                        Detalles
                                    </Button>
                                </Td>
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal de Detalles */}
            <Modal isOpen={isOpen} onClose={onClose} size="xl">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Detalles del Servicio</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        {selectedService && (
                            <Tabs isFitted variant="enclosed">
                                <TabList mb="1em">
                                    <Tab>Información</Tab>
                                    <Tab>Evidencias</Tab>
                                </TabList>
                                <TabPanels>
                                    <TabPanel>
                                        <VStack align="stretch" spacing={4}>
                                            <Box>
                                                <Text fontWeight="bold" color="gray.500" fontSize="sm">Servicio</Text>
                                                <Text fontSize="lg">{getServiceName(selectedService.service_catalogo_id)}</Text>
                                            </Box>
                                            <Box>
                                                <Text fontWeight="bold" color="gray.500" fontSize="sm">Cliente</Text>
                                                <Text fontSize="lg">{getClientName(selectedService.client_id)}</Text>
                                            </Box>
                                            <Box>
                                                <Text fontWeight="bold" color="gray.500" fontSize="sm">Estado Actual</Text>
                                                <Badge
                                                    fontSize="md"
                                                    colorScheme={
                                                        selectedService.status === 'FINISHED' ? 'green' :
                                                            selectedService.status === 'IN_PROGRESS' ? 'blue' : 'orange'
                                                    }
                                                >
                                                    {getStatusLabel(selectedService.status)}
                                                </Badge>
                                            </Box>

                                            {selectedService.status === 'IN_REVIEW' && (
                                                <Box mt={4} p={3} bg="orange.50" borderRadius="md" borderWidth="1px" borderColor="orange.200">
                                                    <Heading size="sm" color="orange.800" mb={2}>Motivo de Reapertura (Cliente)</Heading>
                                                    {selectedService.client_comment && <Text fontSize="sm" color="orange.700" mb={3}>{selectedService.client_comment.replace("Motivo de reapertura: ", "")}</Text>}

                                                    {evidences.filter(e => e.description?.startsWith("Evidencia de reapertura:")).length > 0 && (
                                                        <Box>
                                                            <Text fontWeight="bold" fontSize="sm" color="orange.800" mb={1}>Evidencias Adjuntas por el Cliente:</Text>
                                                            <SimpleGrid columns={3} spacing={2}>
                                                                {evidences.filter(e => e.description?.startsWith("Evidencia de reapertura:")).map(ev => (
                                                                    <Image key={ev.id} src={ev.file_url} alt="Evidencia de reapertura" borderRadius="md" />
                                                                ))}
                                                            </SimpleGrid>
                                                        </Box>
                                                    )}
                                                </Box>
                                            )}

                                            <Box pt={4}>
                                                {selectedService.status === 'CERRADO_ADMIN' || selectedService.status === 'CANCELLED' ? (
                                                    <Box p={3} bg="red.50" borderRadius="md" borderWidth="1px" borderColor="red.200">
                                                        <Text fontSize="sm" color="red.700" fontWeight="bold">
                                                            ⛔ Este servicio está finalizado y cerrado por el cliente. No se pueden realizar cambios.
                                                        </Text>
                                                    </Box>
                                                ) : (
                                                    <>
                                                        <Text fontWeight="bold" mb={2}>Actualizar Estado</Text>
                                                        <HStack>
                                                            <Button
                                                                colorScheme="blue"
                                                                isDisabled={selectedService.status === 'IN_PROGRESS'}
                                                                onClick={() => handleStatusChange('IN_PROGRESS')}
                                                            >
                                                                Iniciar
                                                            </Button>
                                                            <Button
                                                                colorScheme="green"
                                                                isDisabled={selectedService.status === 'FINISHED'}
                                                                onClick={() => handleStatusChange('FINISHED')}
                                                            >
                                                                Finalizar
                                                            </Button>
                                                        </HStack>
                                                    </>
                                                )}
                                            </Box>
                                        </VStack>
                                    </TabPanel>
                                    <TabPanel>
                                        <VStack align="stretch" spacing={6}>
                                            {/* Lista de evidencias */}
                                            <Box>
                                                <Heading size="sm" mb={3}>Evidencias Registradas</Heading>
                                                {serviceEvidences.length > 0 ? (
                                                    <SimpleGrid columns={2} spacing={4}>
                                                        {serviceEvidences.map(ev => (
                                                            <Box key={ev.id} border="1px solid" borderColor="gray.200" borderRadius="md" p={2}>
                                                                <Image src={`${BASE_URL}${ev.file_url}`} alt="Evidencia" borderRadius="sm" mb={2} />
                                                                <HStack justify="space-between" mb={1}>
                                                                    <Badge size="xs" colorScheme={ev.uploader_role === 'CLIENT' ? 'purple' : 'blue'}>
                                                                        {ev.uploader_role === 'CLIENT' ? 'Cliente' : 'Empleado'}
                                                                    </Badge>
                                                                    <Text fontSize="xx-small" color="gray.500">
                                                                        {new Date(ev.uploaded_at).toLocaleString()}
                                                                    </Text>
                                                                </HStack>
                                                                <Text fontSize="xs" noOfLines={2}>{ev.description}</Text>
                                                            </Box>
                                                        ))}
                                                    </SimpleGrid>
                                                ) : (
                                                    <Text color="gray.500" fontStyle="italic">No hay evidencias registradas.</Text>
                                                )}
                                            </Box>

                                            {/* Formulario nueva evidencia */}
                                            <Box borderTop="1px solid" borderColor="gray.200" pt={4}>
                                                {selectedService.status === 'CERRADO_ADMIN' || selectedService.status === 'CANCELLED' ? (
                                                    <Box p={3} bg="red.50" borderRadius="md" borderWidth="1px" borderColor="red.200">
                                                        <Text fontSize="sm" color="red.700" fontWeight="bold">
                                                            ⛔ No se pueden agregar evidencias en un servicio finalizado y cerrado.
                                                        </Text>
                                                    </Box>
                                                ) : (
                                                    <>
                                                        <Heading size="sm" mb={3}>Agregar Nueva Evidencia</Heading>
                                                        <VStack spacing={3}>
                                                            <FormControl>
                                                                <FormLabel fontSize="sm">Descripción (Opcional)</FormLabel>
                                                                <Input
                                                                    value={newEvidenceDescription}
                                                                    onChange={(e) =>
                                                                        setNewEvidenceDescription(e.target.value)
                                                                    }
                                                                    placeholder="Describe la evidencia que subes"
                                                                />
                                                            </FormControl>
                                                            <FormControl>
                                                                <FormLabel fontSize="sm">
                                                                    Fotografías (hasta 4)
                                                                </FormLabel>
                                                                <FilePond
                                                                    files={newEvidenceFiles}
                                                                    onupdatefiles={setNewEvidenceFiles}
                                                                    allowMultiple={true}
                                                                    maxFiles={4}
                                                                    acceptedFileTypes={["image/*"]}
                                                                    name="evidences"
                                                                    labelIdle='Arrastra y suelta fotos aquí o <span class="filepond--label-action">Examina</span>'
                                                                    credits={false}
                                                                />
                                                            </FormControl>
                                                            <Button
                                                                leftIcon={<FiCamera />}
                                                                colorScheme="brand"
                                                                width="full"
                                                                onClick={handleAddEvidence}
                                                                isDisabled={newEvidenceFiles.length === 0}
                                                            >
                                                                Subir Evidencia
                                                            </Button>
                                                        </VStack>
                                                    </>
                                                )}
                                            </Box>
                                        </VStack>
                                    </TabPanel>
                                </TabPanels>
                            </Tabs>
                        )}
                    </ModalBody>
                    <ModalFooter>
                        <Button onClick={onClose}>Cerrar</Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default MisAsignaciones;
