import { useState } from 'react';
import {
    Box,
    Heading,
    Card,
    CardBody,
    FormControl,
    FormLabel,
    Input,
    Button,
    VStack,
    Text,
    useToast,
    List,
    ListItem,
    ListIcon,
    Alert,
    AlertIcon
} from '@chakra-ui/react';
import { CheckCircleIcon, WarningIcon } from '@chakra-ui/icons';
import { updateUser } from '../../services/api';

const PerfilEmpleado = () => {
    const [password, setPassword] = useState('');
    const [confirmPassword, setConfirmPassword] = useState('');
    const [loading, setLoading] = useState(false);
    const toast = useToast();

    // Regex for password validation
    // Min 8 chars, 1 upper, 1 lower, 1 number, 1 special
    const passwordRegex = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&._-])[A-Za-z\d@$!%*?&._-]{8,}$/;

    const isValidPassword = passwordRegex.test(password);
    const doPasswordsMatch = password === confirmPassword && password.length > 0;

    const handlePaste = (e) => {
        e.preventDefault();
        toast({
            title: "Acción bloqueada",
            description: "Por seguridad, debes escribir la contraseña manualmente.",
            status: "warning",
            duration: 3000,
            isClosable: true,
        });
    };

    const handleUpdate = async () => {
        if (!isValidPassword) {
            toast({
                title: "Contraseña insegura",
                description: "La contraseña no cumple con los requisitos mínimos.",
                status: "error",
                duration: 4000,
            });
            return;
        }

        if (!doPasswordsMatch) {
            toast({
                title: "Error",
                description: "Las contraseñas no coinciden.",
                status: "error",
                duration: 3000,
            });
            return;
        }

        setLoading(true);
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            if (user) {
                await updateUser(user.id, { password });
                toast({
                    title: "Contraseña actualizada",
                    description: "Tu contraseña ha sido modificada correctamente.",
                    status: "success",
                    duration: 3000,
                });
                setPassword('');
                setConfirmPassword('');
            }
        } catch (error) {
            toast({
                title: "Error",
                description: error.message || "No se pudo actualizar la contraseña.",
                status: "error",
                duration: 4000,
            });
        } finally {
            setLoading(false);
        }
    };

    return (
        <Box maxW="600px" mx="auto" mt={8}>
            <Heading mb={6}>Mi Perfil</Heading>

            <Card>
                <CardBody>
                    <VStack spacing={6} align="stretch">
                        <Box>
                            <Heading size="md" mb={2}>Cambiar Contraseña</Heading>
                            <Text color="gray.600" fontSize="sm">
                                Por seguridad, actualiza tu contraseña periódicamente.
                            </Text>
                        </Box>

                        <Alert status="info" borderRadius="md">
                            <AlertIcon />
                            <Box>
                                <Text fontWeight="bold">Requisitos de seguridad:</Text>
                                <List spacing={1} mt={2} styleType="disc" ml={4} fontSize="sm">
                                    <ListItem>Mínimo 8 caracteres.</ListItem>
                                    <ListItem>Al menos una letra mayúscula y una minúscula.</ListItem>
                                    <ListItem>Al menos un número.</ListItem>
                                    <ListItem>Al menos un carácter especial (@$!%*?&._-).</ListItem>
                                </List>
                                <Text mt={2} fontSize="sm" fontWeight="bold">
                                    Ejemplos válidos: <Badge colorScheme="green">KeServicios2026!</Badge>, <Badge colorScheme="green">Seguridad.123$</Badge>
                                </Text>
                            </Box>
                        </Alert>

                        <FormControl isRequired>
                            <FormLabel>Nueva Contraseña</FormLabel>
                            <Input
                                type="password"
                                value={password}
                                onChange={(e) => setPassword(e.target.value)}
                                onPaste={handlePaste}
                                placeholder="Escribe tu nueva contraseña"
                            />
                            {password && (
                                <Text mt={1} fontSize="xs" color={isValidPassword ? "green.500" : "red.500"}>
                                    {isValidPassword ? <><CheckCircleIcon mr={1} /> Contraseña válida</> : <><WarningIcon mr={1} /> La contraseña no cumple los requisitos</>}
                                </Text>
                            )}
                        </FormControl>

                        <FormControl isRequired>
                            <FormLabel>Confirmar Contraseña</FormLabel>
                            <Input
                                type="password"
                                value={confirmPassword}
                                onChange={(e) => setConfirmPassword(e.target.value)}
                                onPaste={handlePaste}
                                placeholder="Repite la contraseña"
                            />
                            {confirmPassword && !doPasswordsMatch && (
                                <Text mt={1} fontSize="xs" color="red.500">
                                    Las contraseñas no coinciden.
                                </Text>
                            )}
                        </FormControl>

                        <Button
                            colorScheme="brand"
                            onClick={handleUpdate}
                            isLoading={loading}
                            isDisabled={!isValidPassword || !doPasswordsMatch}
                        >
                            Actualizar Contraseña
                        </Button>
                    </VStack>
                </CardBody>
            </Card>
        </Box>
    );
};

// Start of Badge Import hack if needed, or ensuring Badge is imported
import { Badge } from '@chakra-ui/react';

export default PerfilEmpleado;
