import React from 'react';
import {
  Box,
  Text,
  SimpleGrid,
  Card,
  CardBody,
  HStack,
  VStack,
  Heading,
  Icon,
  Button,
  Flex,
  Avatar,
  Badge,
  Container,
  Alert,
  AlertIcon,
  AlertTitle,
  AlertDescription
} from '@chakra-ui/react';
import { FiUsers, FiFileText, FiDollarSign, FiAlertCircle, FiArrowRight, FiCreditCard, FiCheckCircle } from 'react-icons/fi';
import { useNavigate } from 'react-router-dom';
import { motion } from 'framer-motion';
import Layout from '../../components/Layout';
import { mockAlumnos, mockBitacoras, mockPagos, mockMaterias } from '../../data/mockData';

const MotionBox = motion(Box);
const MotionCard = motion(Card);

const containerVariants = {
  hidden: { opacity: 0 },
  visible: {
    opacity: 1,
    transition: {
      staggerChildren: 0.1
    }
  }
};

const itemVariants = {
  hidden: { y: 20, opacity: 0 },
  visible: {
    y: 0,
    opacity: 1,
    transition: {
      type: "spring",
      stiffness: 100
    }
  }
};

const padreLinks = [
  { nombre: 'Dashboard', path: '/padre' },
  { nombre: 'Mis Hijos', path: '/padre/hijos' },
  { nombre: 'Pagos', path: '/padre/pagos' },
  { nombre: 'Reportes', path: '/padre/reportes' },
];

// Simulando Padre ID 1 (Roberto López - padre de Sofía López)
const padreId = 1;
const misHijos = mockAlumnos.filter(a => a.id_padre === padreId);
const misHijosIds = misHijos.map(h => h.id);
const misBitacoras = mockBitacoras.filter(b => misHijosIds.includes(b.id_alumno)).sort((a, b) => new Date(b.fecha) - new Date(a.fecha));
const misPagos = mockPagos.filter(p => misHijosIds.includes(p.id_alumno));
const pagosPendientes = misPagos.filter(p => p.estatus === 'Pendiente');

const StatCard = ({ title, value, icon, color, subtext }) => (
  <MotionCard
    variants={itemVariants}
    bg="white"
    borderRadius="2xl"
    boxShadow="sm"
    border="1px solid"
    borderColor="gray.100"
    whileHover={{ y: -5, boxShadow: 'lg' }}
    transition={{ duration: 0.2 }}
  >
    <CardBody>
      <Flex justify="space-between" align="start" mb={4}>
        <Box
          p={3}
          bgGradient={`linear(to-br, ${color}.400, ${color}.600)`}
          color="white"
          borderRadius="xl"
          boxShadow={`0 4px 12px ${color}.200`}
        >
          <Icon as={icon} boxSize={6} />
        </Box>
      </Flex>
      <VStack align="start" spacing={0}>
        <Text fontSize="3xl" fontWeight="800" color="gray.800" lineHeight="1">
          {value}
        </Text>
        <Text fontSize="sm" color="gray.500" fontWeight="600" mt={1}>
          {title}
        </Text>
        {subtext && (
          <Text fontSize="xs" color={`${color}.500`} fontWeight="700" mt={2}>
            {subtext}
          </Text>
        )}
      </VStack>
    </CardBody>
  </MotionCard>
);

const ChildProgressCard = ({ bitacora }) => {
  const alumno = mockAlumnos.find(a => a.id === bitacora.id_alumno);
  const materia = mockMaterias.find(m => m.id === bitacora.id_materia);

  return (
    <MotionCard
      variants={itemVariants}
      bg="white"
      border="1px solid"
      borderColor="gray.100"
      borderRadius="2xl"
      whileHover={{ scale: 1.02, borderColor: 'blue.200' }}
      transition={{ duration: 0.2 }}
    >
      <CardBody>
        <HStack mb={3} spacing={3}>
          <Avatar size="sm" src={alumno?.foto_url} name={alumno?.nombre} />
          <Box>
            <Text fontSize="sm" fontWeight="700" color="gray.800">{alumno?.nombre}</Text>
            <Text fontSize="xs" color="gray.500">{materia?.nombre}</Text>
          </Box>
        </HStack>
        <Box bg="gray.50" p={3} borderRadius="xl" mb={3}>
          <Text fontSize="sm" color="gray.600" noOfLines={3} fontStyle="italic">
            "{bitacora.actividad}"
          </Text>
        </Box>
        <Flex justify="space-between" align="center">
          <Text fontSize="xs" color="gray.400">{bitacora.fecha}</Text>
          <Badge colorScheme="green" borderRadius="full" px={2} variant="subtle">
            Completado
          </Badge>
        </Flex>
      </CardBody>
    </MotionCard>
  );
};

const PaymentAlert = ({ pagos }) => {
  const total = pagos.reduce((acc, curr) => acc + curr.monto, 0);
  const navigate = useNavigate();

  return (
    <MotionBox
      variants={itemVariants}
      bg="red.50"
      border="1px solid"
      borderColor="red.100"
      borderRadius="2xl"
      p={5}
      mb={6}
      position="relative"
      overflow="hidden"
    >
      <Box position="absolute" right="-10px" top="-10px" opacity={0.1}>
        <Icon as={FiAlertCircle} boxSize="100px" color="red.500" />
      </Box>
      <Flex direction="column" gap={3} position="relative" zIndex={1}>
        <HStack color="red.600">
          <Icon as={FiAlertCircle} boxSize={5} />
          <Text fontWeight="700">Pagos Pendientes</Text>
        </HStack>
        <Text fontSize="sm" color="red.700">
          Tienes {pagos.length} pago(s) pendiente(s) por un total de:
        </Text>
        <Text fontSize="3xl" fontWeight="800" color="red.600">
          ${total.toLocaleString()}
        </Text>
        <Button
          colorScheme="red"
          size="sm"
          rightIcon={<FiArrowRight />}
          onClick={() => navigate('/padre/pagos')}
          w="full"
          borderRadius="xl"
        >
          Realizar Pago
        </Button>
      </Flex>
    </MotionBox>
  );
};

const PadreDashboard = () => {
  const navigate = useNavigate();

  return (
    <Layout links={padreLinks} userType="Padre/Tutor" userName="Roberto López" userPhoto="https://images.unsplash.com/photo-1500648767791-00dcc994a43e?ixlib=rb-1.2.1&auto=format&fit=facearea&facepad=2&w=256&h=256&q=80">
      <Container maxW="container.xl" px={0}>
        <MotionBox
          initial="hidden"
          animate="visible"
          variants={containerVariants}
        >
          <Flex justify="space-between" align="center" mb={8}>
            <Box>
              <Heading size="lg" color="gray.800" mb={2}>Bienvenido, Roberto 👋</Heading>
              <Text color="gray.600">Aquí está el resumen del progreso de tus hijos</Text>
            </Box>
          </Flex>

          {/* Stats Grid */}
          <SimpleGrid columns={{ base: 1, md: 3 }} spacing={6} mb={8}>
            <StatCard
              title="Hijos Registrados"
              value={misHijos.length}
              icon={FiUsers}
              color="blue"
              subtext="Activos en el ciclo"
            />
            <StatCard
              title="Bitácoras Recientes"
              value={misBitacoras.length}
              icon={FiFileText}
              color="green"
              subtext="Último mes"
            />
            <StatCard
              title="Pagos Realizados"
              value={misPagos.length - pagosPendientes.length}
              icon={FiDollarSign}
              color="purple"
              subtext="Historial completo"
            />
          </SimpleGrid>

          <SimpleGrid columns={{ base: 1, lg: 3 }} spacing={8}>
            {/* Main Content - Progress */}
            <Box gridColumn={{ lg: "span 2" }}>
              <Flex justify="space-between" align="center" mb={6}>
                <Heading size="md" color="gray.800">Avance Reciente</Heading>
                <Button variant="ghost" size="sm" rightIcon={<FiArrowRight />} colorScheme="blue" onClick={() => navigate('/padre/bitacoras')}>
                  Ver todo
                </Button>
              </Flex>
              <SimpleGrid columns={{ base: 1, md: 2 }} spacing={4}>
                {misBitacoras.slice(0, 4).map((bitacora) => (
                  <ChildProgressCard key={bitacora.id} bitacora={bitacora} />
                ))}
              </SimpleGrid>
            </Box>

            {/* Sidebar - Payments & Alerts */}
            <Box>
              <Heading size="md" color="gray.800" mb={6}>Estado de Cuenta</Heading>

              {pagosPendientes.length > 0 ? (
                <PaymentAlert pagos={pagosPendientes} />
              ) : (
                <MotionCard
                  variants={itemVariants}
                  bg="green.50"
                  border="1px solid"
                  borderColor="green.200"
                  borderRadius="2xl"
                  mb={6}
                >
                  <CardBody textAlign="center" py={8}>
                    <Icon as={FiCheckCircle} boxSize={10} color="green.500" mb={3} />
                    <Heading size="sm" color="green.700" mb={1}>¡Todo al día!</Heading>
                    <Text fontSize="sm" color="green.600">
                      No tienes pagos pendientes por el momento.
                    </Text>
                  </CardBody>
                </MotionCard>
              )}

              <MotionCard
                variants={itemVariants}
                variant="outline"
                borderRadius="2xl"
                borderColor="gray.200"
              >
                <CardBody>
                  <Heading size="sm" mb={4}>Accesos Rápidos</Heading>
                  <VStack align="stretch" spacing={3}>
                    <Button
                      variant="ghost"
                      justifyContent="flex-start"
                      leftIcon={<FiCreditCard />}
                      h="50px"
                      _hover={{ bg: 'gray.50', transform: 'translateX(5px)' }}
                      transition="all 0.2s"
                    >
                      Historial de Pagos
                    </Button>
                    <Button
                      variant="ghost"
                      justifyContent="flex-start"
                      leftIcon={<FiFileText />}
                      h="50px"
                      _hover={{ bg: 'gray.50', transform: 'translateX(5px)' }}
                      transition="all 0.2s"
                    >
                      Ver Reportes
                    </Button>
                    <Button
                      variant="ghost"
                      justifyContent="flex-start"
                      leftIcon={<FiUsers />}
                      h="50px"
                      _hover={{ bg: 'gray.50', transform: 'translateX(5px)' }}
                      transition="all 0.2s"
                    >
                      Contactar Profesores
                    </Button>
                  </VStack>
                </CardBody>
              </MotionCard>
            </Box>
          </SimpleGrid>
        </MotionBox>
      </Container>
    </Layout>
  );
};

export default PadreDashboard;
