import { useState } from 'react';
import {
  Box,
  Heading,
  VStack,
  HStack,
  Text,
  Button,
  Badge,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  useDisclosure,
  FormControl,
  FormLabel,
  Input,
  useToast,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  TableContainer,
  Flex
} from '@chakra-ui/react';
import Layout from '../../components/Layout';
import { mockPagos, mockAlumnos } from '../../data/mockData';

const padreLinks = [
  { nombre: 'Avance (Bitácoras)', path: '/padre/bitacoras' },
  { nombre: 'Metas', path: '/padre/metas' },
  { nombre: 'Pagos', path: '/padre/pagos' },
  { nombre: 'Calendario', path: '/padre/calendario' },
  { nombre: 'Notificaciones', path: '/padre/notificaciones' }
];

const padreId = 1;
const misPagos = mockPagos.filter(p => p.id_padre === padreId);
const misHijos = mockAlumnos.filter(a => a.id_padre === padreId);

const PagosPadre = () => {
  const { isOpen, onOpen, onClose } = useDisclosure();
  const toast = useToast();

  const getStatusColor = (estatus) => {
    switch (estatus) {
      case 'Validado': return 'green';
      case 'Pendiente': return 'orange';
      case 'Rechazado': return 'red';
      default: return 'gray';
    }
  };

  const handleRegistrarPago = (e) => {
    e.preventDefault();
    const formData = new FormData(e.target);
    const archivo = formData.get('comprobante');

    if (!archivo || !archivo.files[0]) {
      toast({
        title: 'Error',
        description: 'Por favor selecciona un archivo',
        status: 'error',
        duration: 2000,
        isClosable: true,
      });
      return;
    }

    toast({
      title: 'Pago registrado',
      description: 'Tu comprobante ha sido enviado y está pendiente de validación',
      status: 'success',
      duration: 3000,
      isClosable: true,
    });

    e.target.reset();
    onClose();
  };

  return (
    <Layout links={padreLinks} userType="Padre/Tutor" userName="Roberto López">
      <Box>
        <Heading size="lg" mb={4} color="brand.500">Gestión de Pagos</Heading>

        {/* Instrucciones minimalistas */}
        <Text fontSize="sm" color="gray.600" mb={6}>
          Gestiona los pagos de tus hijos. <strong>Sube el comprobante</strong> para que sea validado por el administrador.
        </Text>

        <VStack align="stretch" spacing={6}>
          <Box bg="white" p={6} borderRadius="md" boxShadow="sm">
            <HStack justify="space-between" mb={4}>
              <Text fontWeight="bold">Próximo Pago</Text>
              <Badge colorScheme="orange">Pendiente</Badge>
            </HStack>
            {misHijos.map((hijo) => (
              <Box key={hijo.id} mb={2}>
                <Text color="gray.600">
                  {hijo.nombre} - Fecha de pago: {hijo.fecha_pago}
                </Text>
              </Box>
            ))}
            <Button colorScheme="brand" mt={4} onClick={onOpen}>
              Registrar Pago
            </Button>
          </Box>

          {/* Tarjeta con estilo Facit */}
          <Box bg="white" borderRadius="xl" boxShadow="md" overflow="hidden">
            {/* Header de la Tarjeta */}
            <Flex justify="space-between" align="center" p={5} borderBottom="1px" borderColor="gray.100">
              <Heading as="h3" size="md">Historial de Pagos</Heading>
            </Flex>

            {/* Cuerpo de la Tabla */}
            <TableContainer>
              <Table variant="simple">
                <Thead>
                  <Tr>
                    <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Fecha</Th>
                    <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Alumno</Th>
                    <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Monto</Th>
                    <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Estatus</Th>
                  </Tr>
                </Thead>
                <Tbody>
                  {misPagos.map((pago) => {
                    const alumno = mockAlumnos.find(a => a.id === pago.id_alumno);
                    return (
                      <Tr key={pago.id} _hover={{ bg: "gray.50" }}>
                        <Td verticalAlign="middle">{pago.fecha_pago}</Td>
                        <Td verticalAlign="middle">{alumno?.nombre}</Td>
                        <Td verticalAlign="middle">${pago.monto.toLocaleString()}</Td>
                        <Td verticalAlign="middle">
                          <Badge colorScheme={getStatusColor(pago.estatus)}>
                            {pago.estatus}
                          </Badge>
                        </Td>
                      </Tr>
                    );
                  })}
                </Tbody>
              </Table>
            </TableContainer>
          </Box>
        </VStack>

        <Modal isOpen={isOpen} onClose={onClose}>
          <ModalOverlay />
          <ModalContent>
            <ModalHeader>Registrar Pago</ModalHeader>
            <ModalCloseButton />
            <form onSubmit={handleRegistrarPago}>
              <ModalBody>
                <FormControl mb={4}>
                  <FormLabel>Alumno</FormLabel>
                  <Input
                    value={misHijos[0]?.nombre || ''}
                    readOnly
                    bg="gray.50"
                  />
                </FormControl>
                <HStack mb={4}>
                  <FormControl>
                    <FormLabel>Monto</FormLabel>
                    <Input
                      type="number"
                      defaultValue="1500"
                      readOnly
                      bg="gray.50"
                    />
                  </FormControl>
                  <FormControl>
                    <FormLabel>Concepto (Folio)</FormLabel>
                    <Input
                      value={`PAGO-ALU-${misHijos[0]?.id || 'X'}`}
                      readOnly
                      bg="gray.50"
                      fontWeight="bold"
                      color="brand.500"
                    />
                  </FormControl>
                </HStack>

                <FormControl mb={4} isRequired>
                  <FormLabel>Banco de Origen</FormLabel>
                  <Input placeholder="Ej. BBVA, Santander..." name="banco" />
                </FormControl>

                <FormControl mb={4}>
                  <FormLabel>Clave de Rastreo / No. Operación</FormLabel>
                  <Input placeholder="Ej. 1234567890" name="clave" />
                </FormControl>

                <FormControl mb={4} isRequired>
                  <FormLabel>Nombre del Remitente</FormLabel>
                  <Input placeholder="Nombre del titular de la cuenta" name="remitente" defaultValue="Roberto López" />
                </FormControl>

                <FormControl isRequired>
                  <FormLabel>Subir Comprobante</FormLabel>
                  <Box
                    border="2px dashed"
                    borderColor="gray.300"
                    borderRadius="md"
                    p={4}
                    textAlign="center"
                    _hover={{ borderColor: 'brand.500', bg: 'gray.50' }}
                  >
                    <Input
                      type="file"
                      name="comprobante"
                      accept="image/*,.pdf"
                      p={1}
                    />
                    <Text fontSize="xs" color="gray.500" mt={2}>
                      Formatos aceptados: JPG, PNG, PDF
                    </Text>
                  </Box>
                </FormControl>
              </ModalBody>
              <ModalFooter>
                <Button variant="ghost" mr={3} onClick={onClose}>Cancelar</Button>
                <Button colorScheme="brand" type="submit">Registrar Pago</Button>
              </ModalFooter>
            </form>
          </ModalContent>
        </Modal>
      </Box>
    </Layout>
  );
};

export default PagosPadre;

