
import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    Card,
    CardBody,
    Text,
    Badge,
    VStack,
    HStack,
    Divider,
    useToast,
    ButtonGroup,
    Button,
    SimpleGrid,
    IconButton,
    Flex,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    Spinner,
    Icon,
    Tooltip
} from '@chakra-ui/react';
import { ChevronLeftIcon, ChevronRightIcon } from '@chakra-ui/icons';
import { FiClock, FiUsers } from "react-icons/fi";
import {
    getExams,
    getExamAssignments
} from '../../services/api';

const CalendarioTrabajo = () => {
    const [loading, setLoading] = useState(true);
    const [allAssignments, setAllAssignments] = useState([]);
    const [viewMode, setViewMode] = useState('list'); // 'list' | 'month'
    const [currentDate, setCurrentDate] = useState(new Date());

    const { isOpen, onOpen, onClose } = useDisclosure();
    const [selectedAssignment, setSelectedAssignment] = useState(null);

    const toast = useToast();

    const fetchData = async () => {
        try {
            setLoading(true);
            const user = JSON.parse(sessionStorage.getItem('user'));
            const userId = Number(user?.id || user?.user_id);

            if (userId) {
                // 1. Obtener todos los exámenes
                const exams = await getExams();
                // 2. Filtrar solo los del profesor
                const myExams = exams.filter(e => Number(e.professor_id) === userId);
                
                // 3. Obtener asignaciones para cada examen
                let assignments = [];
                for (const exam of myExams) {
                    try {
                        const examAssignments = await getExamAssignments(exam.id_exam || exam.id);
                        // Obtener duración y tolerancia del examen
                        const settings = typeof exam.settings === 'string' ? JSON.parse(exam.settings) : exam.settings;
                        const timerMinutes = parseInt(settings?.timer_minutes) || 60;
                        const toleranceMinutes = parseInt(exam.tolerance_minutes) || 15;
                        
                        // Enriquecer la asignación con info del examen
                        const enriched = (examAssignments || []).map(a => ({
                            ...a,
                            exam_title: exam.title,
                            exam_description: exam.description,
                            exam_timer_minutes: timerMinutes,
                            exam_tolerance_minutes: toleranceMinutes
                        }));
                        assignments = [...assignments, ...enriched];
                    } catch (eErr) {
                        console.error(`Error fetching assignments for exam ${exam.id}:`, eErr);
                    }
                }

                setAllAssignments(assignments);
            }
        } catch (error) {
            console.error('Error loading calendar data:', error);
            toast({
                title: 'Error al cargar calendario',
                status: 'error',
                duration: 3000,
            });
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchData();
    }, []);

    const getStatusLabel = (status) => {
        const statusMap = {
            'PENDING': 'Pendiente',
            'IN_PROGRESS': 'En Progreso',
            'ACTIVE': 'En Curso',
            'COMPLETED': 'Completado',
            'FINISHED': 'Finalizado',
            'CANCELLED': 'Cancelado'
        };
        return statusMap[status] || status;
    };

    const getStatusColor = (status) => {
        const colorMap = {
            'PENDING': 'orange',
            'ACTIVE': 'blue',
            'FINISHED': 'green',
            'CANCELLED': 'red'
        };
        return colorMap[status] || 'gray';
    };

    // ---- LOGIC FOR LIST VIEW ----
    const renderListView = () => {
        const now = new Date();
        const todayStr = now.toISOString().split('T')[0];
        
        // Ordenar por fecha
        const sorted = [...allAssignments].sort((a, b) => 
            new Date(a.scheduled_at || 0) - new Date(b.scheduled_at || 0)
        );

        const todayTasks = sorted.filter(a => a.scheduled_at && a.scheduled_at.startsWith(todayStr));
        
        // Filtrar próximas aplicaciones: excluir las que ya pasaron (fecha + duración + tolerancia < ahora)
        const upcomingTasks = sorted.filter(a => {
            if (!a.scheduled_at) return false;
            
            const scheduledDate = new Date(a.scheduled_at);
            
            // Calcular fecha de fin: scheduled_at + duración + tolerancia
            const timerMinutes = a.exam_timer_minutes || 60;
            const toleranceMinutes = a.exam_tolerance_minutes || 15;
            const endDate = new Date(scheduledDate);
            endDate.setMinutes(endDate.getMinutes() + timerMinutes + toleranceMinutes);
            
            // Si la fecha de fin ya pasó, no incluir en próximas aplicaciones
            if (endDate < now) {
                return false;
            }
            
            // Solo incluir si la fecha programada es mayor o igual a hoy (pero que aún no haya terminado)
            return a.scheduled_at >= todayStr;
        });

        return (
            <VStack spacing={8} align="stretch">
                <Box>
                    <Box 
                        bg="brand.50" 
                        p={4} 
                        borderRadius="lg" 
                        borderLeft="4px solid" 
                        borderColor="brand.500"
                        mb={4}
                    >
                        <HStack spacing={3} align="center">
                            <Icon as={FiClock} color="brand.600" boxSize={6} />
                            <VStack align="start" spacing={0}>
                                <Heading size="md" color="brand.700" fontWeight="bold">
                                    Aplicaciones de Hoy
                                </Heading>
                                <Text fontSize="sm" color="brand.600" fontWeight="medium">
                                    {new Date().toLocaleDateString('es-ES', { 
                                        weekday: 'long', 
                                        year: 'numeric', 
                                        month: 'long', 
                                        day: 'numeric' 
                                    })}
                                </Text>
                            </VStack>
                        </HStack>
                    </Box>
                    {todayTasks.length > 0 ? (
                        <VStack spacing={3} align="stretch">
                            {todayTasks.map(task => (
                                <Card 
                                    key={task.id_assignment} 
                                    variant="outline" 
                                    borderLeft="4px solid" 
                                    borderColor="brand.500"
                                    _hover={{ shadow: 'md', borderColor: 'brand.600' }}
                                    transition="all 0.2s"
                                >
                                    <CardBody py={4}>
                                        <HStack justify="space-between" align="start">
                                            <VStack align="start" spacing={2} flex={1}>
                                                <Text fontWeight="bold" fontSize="lg" color="gray.800">
                                                    {task.exam_title}
                                                </Text>
                                                <HStack spacing={4} flexWrap="wrap">
                                                    <HStack spacing={1}>
                                                        <Icon as={FiUsers} color="gray.500" boxSize={4} />
                                                        <Text fontSize="sm" color="gray.600" fontWeight="medium">
                                                            {task.student_name || task.student_email}
                                                        </Text>
                                                    </HStack>
                                                    <HStack spacing={1}>
                                                        <Icon as={FiClock} color="brand.500" boxSize={4} />
                                                        <Text fontSize="sm" color="brand.600" fontWeight="semibold">
                                                            {new Date(task.scheduled_at).toLocaleTimeString('es-ES', {
                                                                hour: '2-digit', 
                                                                minute: '2-digit',
                                                                hour12: true
                                                            })}
                                                        </Text>
                                                    </HStack>
                                                </HStack>
                                            </VStack>
                                            <Badge 
                                                colorScheme={getStatusColor(task.status)}
                                                fontSize="sm"
                                                px={3}
                                                py={1}
                                                borderRadius="full"
                                            >
                                                {getStatusLabel(task.status)}
                                            </Badge>
                                        </HStack>
                                    </CardBody>
                                </Card>
                            ))}
                        </VStack>
                    ) : (
                        <Box 
                            bg="gray.50" 
                            p={8} 
                            borderRadius="lg" 
                            border="1px dashed" 
                            borderColor="gray.300"
                            textAlign="center"
                        >
                            <Icon as={FiClock} color="gray.400" boxSize={8} mb={3} />
                            <Text color="gray.600" fontSize="md" fontWeight="medium" mb={1}>
                                No tienes aplicaciones programadas para hoy
                            </Text>
                            <Text color="gray.500" fontSize="sm">
                                Las aplicaciones de hoy aparecerán aquí cuando estén programadas
                            </Text>
                        </Box>
                    )}
                </Box>
                
                <Divider borderColor="gray.300" />
                
                <Box>
                    <Box 
                        bg="blue.50" 
                        p={4} 
                        borderRadius="lg" 
                        borderLeft="4px solid" 
                        borderColor="blue.500"
                        mb={4}
                    >
                        <HStack spacing={3} align="center">
                            <Icon as={FiClock} color="blue.600" boxSize={6} />
                            <VStack align="start" spacing={0}>
                                <Heading size="md" color="blue.700" fontWeight="bold">
                                    Próximas Aplicaciones
                                </Heading>
                                <Text fontSize="sm" color="blue.600" fontWeight="medium">
                                    Exámenes programados para los próximos días
                                </Text>
                            </VStack>
                        </HStack>
                    </Box>
                    {upcomingTasks.length > 0 ? (
                        <VStack spacing={3} align="stretch">
                            {upcomingTasks.map(task => (
                                <Card 
                                    key={task.id_assignment} 
                                    variant="outline"
                                    _hover={{ shadow: 'md', borderColor: 'blue.300' }}
                                    transition="all 0.2s"
                                >
                                    <CardBody py={4}>
                                        <HStack justify="space-between" align="start">
                                            <VStack align="start" spacing={2} flex={1}>
                                                <HStack spacing={2}>
                                                    <Icon as={FiClock} color="blue.500" boxSize={4} />
                                                    <Text fontSize="xs" fontWeight="bold" color="blue.600" textTransform="uppercase">
                                                        {new Date(task.scheduled_at).toLocaleDateString('es-ES', {
                                                            weekday: 'long',
                                                            year: 'numeric',
                                                            month: 'long',
                                                            day: 'numeric'
                                                        })}
                                                    </Text>
                                                </HStack>
                                                <Text fontWeight="bold" fontSize="lg" color="gray.800">
                                                    {task.exam_title}
                                                </Text>
                                                <HStack spacing={1}>
                                                    <Icon as={FiUsers} color="gray.500" boxSize={4} />
                                                    <Text fontSize="sm" color="gray.600" fontWeight="medium">
                                                        {task.student_name || task.student_email}
                                                    </Text>
                                                </HStack>
                                                <Text fontSize="xs" color="blue.600" fontWeight="semibold" mt={1}>
                                                    Hora: {new Date(task.scheduled_at).toLocaleTimeString('es-ES', {
                                                        hour: '2-digit',
                                                        minute: '2-digit',
                                                        hour12: true
                                                    })}
                                                </Text>
                                            </VStack>
                                            <Badge 
                                                variant="subtle" 
                                                colorScheme={getStatusColor(task.status)}
                                                fontSize="sm"
                                                px={3}
                                                py={1}
                                                borderRadius="full"
                                            >
                                                {getStatusLabel(task.status)}
                                            </Badge>
                                        </HStack>
                                    </CardBody>
                                </Card>
                            ))}
                        </VStack>
                    ) : (
                        <Box 
                            bg="gray.50" 
                            p={8} 
                            borderRadius="lg" 
                            border="1px dashed" 
                            borderColor="gray.300"
                            textAlign="center"
                        >
                            <Icon as={FiClock} color="gray.400" boxSize={8} mb={3} />
                            <Text color="gray.600" fontSize="md" fontWeight="medium" mb={1}>
                                No hay aplicaciones futuras programadas
                            </Text>
                            <Text color="gray.500" fontSize="sm">
                                Las próximas aplicaciones aparecerán aquí cuando estén programadas
                            </Text>
                        </Box>
                    )}
                </Box>
            </VStack>
        );
    };

    // ---- LOGIC FOR MONTH VIEW ----
    const getDaysInMonth = (year, month) => new Date(year, month + 1, 0).getDate();

    const getMonthData = () => {
        const year = currentDate.getFullYear();
        const month = currentDate.getMonth();
        const daysInMonth = getDaysInMonth(year, month);
        let firstDay = new Date(year, month, 1).getDay();
        const startDayIndex = (firstDay + 6) % 7;
        return { year, month, daysInMonth, startDayIndex };
    };

    const changeMonth = (offset) => {
        const newDate = new Date(currentDate);
        newDate.setMonth(newDate.getMonth() + offset);
        setCurrentDate(newDate);
    };

    const renderMonthView = () => {
        const { year, month, daysInMonth, startDayIndex } = getMonthData();
        const monthName = currentDate.toLocaleString('es-ES', { month: 'long', year: 'numeric' });
        const weekDays = ['Lun', 'Mar', 'Mié', 'Jue', 'Vie', 'Sáb', 'Dom'];

        const cells = [];
        for (let i = 0; i < startDayIndex; i++) {
            cells.push(<Box key={`empty-${i}`} height="100px" bg="gray.50" border="1px solid" borderColor="gray.100" />);
        }
        
        for (let day = 1; day <= daysInMonth; day++) {
            const dateStr = `${year}-${String(month + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
            const tasks = allAssignments.filter(a => a.scheduled_at && a.scheduled_at.startsWith(dateStr));
            const isToday = new Date().toDateString() === new Date(year, month, day).toDateString();

            cells.push(
                <Box
                    key={day}
                    height="120px"
                    borderWidth="1px"
                    p={1}
                    bg={isToday ? "blue.50" : "white"}
                    position="relative"
                    overflowY="auto"
                >
                    <Text fontWeight="bold" fontSize="xs" color={isToday ? "blue.600" : "gray.700"}>
                        {day}
                    </Text>
                    <VStack align="start" spacing={1} mt={1}>
                        {tasks.map(task => (
                            <Tooltip key={task.id_assignment} label={`${task.exam_title} - ${task.student_name}`}>
                                <Box
                                    w="100%"
                                    bg={`${getStatusColor(task.status)}.100`}
                                    p="2px"
                                    borderRadius="sm"
                                    fontSize="9px"
                                    cursor="pointer"
                                    borderLeft="2px solid"
                                    borderColor={`${getStatusColor(task.status)}.500`}
                                >
                                    <Text noOfLines={1} fontWeight="medium">
                                        {new Date(task.scheduled_at).toLocaleTimeString([], {hour:'2-digit', minute:'2-digit'})} {task.exam_title}
                                    </Text>
                                </Box>
                            </Tooltip>
                        ))}
                    </VStack>
                </Box>
            );
        }

        return (
            <Box>
                <Flex justify="space-between" align="center" mb={4}>
                    <IconButton icon={<ChevronLeftIcon />} onClick={() => changeMonth(-1)} aria-label="Mes anterior" size="sm" />
                    <Heading size="md" textTransform="capitalize">{monthName}</Heading>
                    <IconButton icon={<ChevronRightIcon />} onClick={() => changeMonth(1)} aria-label="Mes siguiente" size="sm" />
                </Flex>
                <SimpleGrid columns={7} spacing={0} border="1px solid" borderColor="gray.200">
                    {weekDays.map(d => (
                        <Box key={d} textAlign="center" fontWeight="bold" py={2} bg="gray.100" fontSize="xs" borderBottom="1px solid" borderColor="gray.200">
                            {d}
                        </Box>
                    ))}
                    {cells}
                </SimpleGrid>
            </Box>
        );
    };

    return (
        <Box p={4}>
            <Flex justify="space-between" align="center" mb={6}>
                <VStack align="start" spacing={0}>
                    <Heading size="lg">Agenda de Aplicaciones</Heading>
                    <Text fontSize="sm" color="gray.500">Consulta cuándo y a quién se aplican tus exámenes</Text>
                </VStack>
                <HStack>
                    {loading && <Spinner size="sm" color="brand.500" mr={2} />}
                    <ButtonGroup isAttached variant="outline" size="sm">
                        <Button
                            colorScheme={viewMode === 'list' ? 'brand' : 'gray'}
                            variant={viewMode === 'list' ? 'solid' : 'outline'}
                            onClick={() => setViewMode('list')}
                        >
                            Lista
                        </Button>
                        <Button
                            colorScheme={viewMode === 'month' ? 'brand' : 'gray'}
                            variant={viewMode === 'month' ? 'solid' : 'outline'}
                            onClick={() => setViewMode('month')}
                        >
                            Calendario
                        </Button>
                    </ButtonGroup>
                </HStack>
            </Flex>

            {loading && allAssignments.length === 0 ? (
                <Flex justify="center" align="center" py={20} direction="column">
                    <Spinner size="xl" color="brand.500" mb={4} />
                    <Text color="gray.500">Cargando aplicaciones programadas...</Text>
                </Flex>
            ) : (
                viewMode === 'list' ? renderListView() : renderMonthView()
            )}
        </Box>
    );
};

export default CalendarioTrabajo;
