import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    Card,
    CardBody,
    VStack,
    HStack,
    Button,
    FormControl,
    FormLabel,
    Input,
    Textarea,
    useToast,
    SimpleGrid,
    IconButton,
    Checkbox,
    Text,
    Badge,
    Divider,
    Spinner,
    Breadcrumb,
    BreadcrumbItem,
    BreadcrumbLink,
    RadioGroup,
    Radio,
    Stack,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    useDisclosure
} from '@chakra-ui/react';
import { FiPlus, FiSave, FiTrash2, FiSettings, FiChevronRight, FiZap } from 'react-icons/fi';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { getQuestions, createExam, getExamById, updateExam } from '../../services/api';

const ExamBuilder = () => {
    const [searchParams] = useSearchParams();
    const examId = searchParams.get('id');
    const navigate = useNavigate();
    const toast = useToast();

    const [loading, setLoading] = useState(false);
    const [title, setTitle] = useState('');
    const [description, setDescription] = useState('');
    const [timer, setTimer] = useState(60);
    const [tolerance, setTolerance] = useState(15);
    const [questions, setQuestions] = useState([]);
    const [selectedQuestions, setSelectedQuestions] = useState([]);
    const { isOpen: isAutoModalOpen, onOpen: onAutoModalOpen, onClose: onAutoModalClose } = useDisclosure();
    const [questionsPerCategory, setQuestionsPerCategory] = useState({});

    useEffect(() => {
        const fetchData = async () => {
            setLoading(true);
            try {
                const [allQuestions, examData] = await Promise.all([
                    getQuestions(),
                    examId ? getExamById(examId) : null
                ]);

                setQuestions(allQuestions);

                if (examData) {
                    setTitle(examData.title || '');
                    setDescription(examData.description || '');
                    setTimer(examData.settings?.timer_minutes || 60);
                    setTolerance(examData.tolerance_minutes || 15);
                    
                    // Mapear solo los IDs de las preguntas ya asignadas
                    if (examData.questions) {
                        setSelectedQuestions(examData.questions.map(q => q.id_question));
                    }
                }
            } catch (error) {
                console.error('Error fetching data:', error);
                toast({ title: 'Error al cargar datos', status: 'error' });
            } finally {
                setLoading(false);
            }
        };
        fetchData();
    }, [examId]);

    const handleSave = async () => {
        if (!title.trim()) {
            toast({ title: 'El título es requerido', status: 'warning' });
            return;
        }

        if (selectedQuestions.length === 0) {
            toast({ title: 'Debes seleccionar al menos una pregunta', status: 'warning' });
            return;
        }

        try {
            const examData = {
                title,
                description,
                settings: { timer_minutes: parseInt(timer), randomization: true },
                tolerance_minutes: parseInt(tolerance),
                questions: selectedQuestions
            };

            if (examId) {
                await updateExam(examId, examData);
                toast({ title: 'Examen actualizado exitosamente', status: 'success' });
            } else {
                await createExam(examData);
                toast({ title: 'Examen creado exitosamente', status: 'success' });
            }
            navigate('/professor/examenes');
        } catch (error) {
            toast({ title: examId ? 'Error al actualizar examen' : 'Error al crear examen', status: 'error' });
        }
    };

    const toggleQuestion = (id) => {
        setSelectedQuestions(prev => 
            prev.includes(id) 
                ? prev.filter(qId => qId !== id) 
                : [...prev, id]
        );
    };

    // Obtener categorías únicas y contar preguntas por categoría
    const getCategoriesWithCounts = () => {
        const categoriesMap = {};
        questions.forEach(q => {
            const category = q.category || 'General';
            if (!categoriesMap[category]) {
                categoriesMap[category] = [];
            }
            categoriesMap[category].push(q);
        });
        return categoriesMap;
    };

    // Función para seleccionar preguntas aleatoriamente
    const handleAutoSelect = () => {
        const categoriesMap = getCategoriesWithCounts();
        const selected = [];
        
        Object.keys(questionsPerCategory).forEach(category => {
            const count = parseInt(questionsPerCategory[category]) || 0;
            if (count > 0) {
                const availableQuestions = categoriesMap[category] || [];
                const maxCount = Math.min(count, availableQuestions.length);
                
                if (maxCount > 0) {
                    // Mezclar aleatoriamente y tomar las primeras 'maxCount'
                    const shuffled = [...availableQuestions].sort(() => Math.random() - 0.5);
                    const selectedFromCategory = shuffled.slice(0, maxCount).map(q => q.id_question);
                    selected.push(...selectedFromCategory);
                }
            }
        });

        setSelectedQuestions(selected);
        onAutoModalClose();
        toast({ 
            title: `Se seleccionaron ${selected.length} preguntas automáticamente`, 
            status: 'success' 
        });
    };

    // Inicializar el estado de preguntas por categoría cuando se abre el modal
    const handleOpenAutoModal = () => {
        const categoriesMap = getCategoriesWithCounts();
        const initialValues = {};
        Object.keys(categoriesMap).forEach(category => {
            initialValues[category] = '';
        });
        setQuestionsPerCategory(initialValues);
        onAutoModalOpen();
    };

    if (loading) {
        return (
            <Box display="flex" justifyContent="center" alignItems="center" height="400px">
                <Spinner size="xl" color="brand.500" thickness="4px" />
            </Box>
        );
    }

    return (
        <Box p={6}>
            <Breadcrumb spacing="8px" separator={<FiChevronRight color="gray.500" />} mb={6}>
                <BreadcrumbItem>
                    <BreadcrumbLink onClick={() => navigate('/professor/examenes')}>Exámenes</BreadcrumbLink>
                </BreadcrumbItem>
                <BreadcrumbItem isCurrentPage>
                    <BreadcrumbLink href="#">{examId ? 'Editar Examen' : 'Nuevo Examen'}</BreadcrumbLink>
                </BreadcrumbItem>
            </Breadcrumb>

            <Heading mb={6}>{examId ? 'Editar Examen' : 'Constructor de Exámenes'}</Heading>
            
            <SimpleGrid columns={{ base: 1, lg: 2 }} spacing={8}>
                {/* Exam Settings */}
                <VStack spacing={6} align="stretch">
                    <Card>
                        <CardBody>
                            <Heading size="md" mb={4}>Configuración General</Heading>
                            <VStack spacing={4}>
                                <FormControl isRequired>
                                    <FormLabel>Título del Examen</FormLabel>
                                    <Input value={title} onChange={(e) => setTitle(e.target.value)} placeholder="Ej: Final de Anatomía" />
                                </FormControl>
                                <FormControl>
                                    <FormLabel>Descripción</FormLabel>
                                    <Textarea value={description} onChange={(e) => setDescription(e.target.value)} placeholder="Instrucciones para el estudiante" />
                                </FormControl>
                                <FormControl>
                                    <FormLabel>Tiempo (minutos)</FormLabel>
                                    <Input type="number" value={timer} onChange={(e) => setTimer(e.target.value)} />
                                </FormControl>
                                <FormControl>
                                    <FormLabel>Tolerancia de Inicio (minutos)</FormLabel>
                                    <Input type="number" value={tolerance} onChange={(e) => setTolerance(e.target.value)} />
                                    <Text fontSize="xs" color="gray.500" mt={1}>
                                        Tiempo máximo que el alumno puede tardar en iniciar después de la hora programada.
                                    </Text>
                                </FormControl>
                            </VStack>
                        </CardBody>
                    </Card>

                    <Button leftIcon={<FiSave />} colorScheme="brand" size="lg" onClick={handleSave}>
                        {examId ? 'Actualizar Examen' : 'Guardar Examen'}
                    </Button>
                </VStack>

                {/* Question Selection */}
                <Card>
                    <CardBody>
                        <HStack justify="space-between" mb={4}>
                            <Heading size="md">Banco de Reactivos ({selectedQuestions.length} seleccionadas)</Heading>
                            <Button 
                                leftIcon={<FiZap />} 
                                colorScheme="purple" 
                                variant="outline"
                                onClick={handleOpenAutoModal}
                                size="sm"
                            >
                                Formación automática
                            </Button>
                        </HStack>
                        <Text fontSize="sm" color="gray.500" mb={4}>
                            Marca las preguntas que deseas incluir en este examen.
                        </Text>
                        <VStack spacing={3} align="stretch" maxH="600px" overflowY="auto" pr={2}>
                            {questions.map(q => (
                                <Box 
                                    key={q.id_question} 
                                    p={3} 
                                    border="1px solid" 
                                    borderColor={selectedQuestions.includes(q.id_question) ? "brand.200" : "gray.200"} 
                                    bg={selectedQuestions.includes(q.id_question) ? "brand.50" : "transparent"}
                                    borderRadius="md"
                                    transition="all 0.2s"
                                    _hover={{ borderColor: "brand.300" }}
                                >
                                    <HStack justify="space-between" align="start">
                                        <Checkbox 
                                            colorScheme="brand"
                                            isChecked={selectedQuestions.includes(q.id_question)} 
                                            onChange={() => toggleQuestion(q.id_question)}
                                            mt={1}
                                        >
                                            <VStack align="start" spacing={1}>
                                                <Text fontWeight="bold" noOfLines={2}>{q.content?.question}</Text>
                                                <HStack spacing={2}>
                                                    <Badge size="xs" colorScheme="purple">{q.category || 'General'}</Badge>
                                                    <Badge size="xs" variant="subtle" colorScheme="blue">
                                                        {q.type === 'MULTIPLE_CHOICE' ? 'Opción Múltiple' :
                                                         q.type === 'TRUE_FALSE' ? 'Verdadero / Falso' :
                                                         q.type === 'OPEN' ? 'Abierta' :
                                                         q.type}
                                                    </Badge>
                                                </HStack>
                                            </VStack>
                                        </Checkbox>
                                    </HStack>
                                </Box>
                            ))}
                            {questions.length === 0 && (
                                <Text textAlign="center" py={10} color="gray.500">
                                    No hay preguntas disponibles en tu banco. Crea algunas primero.
                                </Text>
                            )}
                        </VStack>
                    </CardBody>
                </Card>
            </SimpleGrid>

            {/* Modal de Formación Automática */}
            <Modal isOpen={isAutoModalOpen} onClose={onAutoModalClose} size="lg">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Formación Automática de Examen</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody pb={6}>
                        <Text fontSize="sm" color="gray.600" mb={4}>
                            Indica cuántas preguntas deseas seleccionar por categoría. El sistema seleccionará aleatoriamente las preguntas disponibles.
                        </Text>
                        <VStack spacing={4} align="stretch" maxH="400px" overflowY="auto">
                            {Object.keys(getCategoriesWithCounts()).map(category => {
                                const availableCount = (getCategoriesWithCounts()[category] || []).length;
                                return (
                                    <FormControl key={category}>
                                        <HStack justify="space-between" mb={2}>
                                            <FormLabel mb={0}>
                                                {category}
                                                <Badge ml={2} colorScheme="purple" size="sm">
                                                    {availableCount} disponible{availableCount !== 1 ? 's' : ''}
                                                </Badge>
                                            </FormLabel>
                                        </HStack>
                                        <Input
                                            type="number"
                                            min="0"
                                            max={availableCount}
                                            value={questionsPerCategory[category] || ''}
                                            onChange={(e) => {
                                                const value = e.target.value;
                                                const numValue = parseInt(value) || 0;
                                                if (value === '' || (numValue >= 0 && numValue <= availableCount)) {
                                                    setQuestionsPerCategory({
                                                        ...questionsPerCategory,
                                                        [category]: value === '' ? '' : numValue.toString()
                                                    });
                                                }
                                            }}
                                            placeholder="0"
                                        />
                                        {questionsPerCategory[category] && parseInt(questionsPerCategory[category]) > availableCount && (
                                            <Text fontSize="xs" color="red.500" mt={1}>
                                                El número no puede exceder las {availableCount} preguntas disponibles
                                            </Text>
                                        )}
                                    </FormControl>
                                );
                            })}
                        </VStack>
                    </ModalBody>
                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onAutoModalClose}>
                            Cancelar
                        </Button>
                        <Button 
                            colorScheme="purple" 
                            onClick={handleAutoSelect}
                            leftIcon={<FiZap />}
                        >
                            Seleccionar Preguntas
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default ExamBuilder;


