
import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    useToast,
    HStack,
    VStack,
    Text,
    SimpleGrid,
    Card,
    CardBody,
    Stat,
    StatLabel,
    StatNumber,
    Tooltip,
    Spinner,
    Icon
} from "@chakra-ui/react";
import { FiEye, FiEdit2, FiTrash2, FiPlus, FiBarChart2, FiUsers, FiCheck, FiUserPlus } from "react-icons/fi";
import { useNavigate } from 'react-router-dom';
import { getExams, deleteExam, getStudents, getExamAssignments, assignExam, unassignExam } from '../../services/api';
import { RadioGroup, Radio, Stack, Input, FormControl, FormLabel, Divider } from '@chakra-ui/react';
import useServerTime from '../../hooks/useServerTime';

const GestionExamenes = () => {
    const [exams, setExams] = useState([]);
    const [loading, setLoading] = useState(true);

    // Estados para asignación
    const [students, setStudents] = useState([]);
    const [assignedIds, setAssignedStudentsIds] = useState([]);
    const [currentExam, setCurrentExam] = useState(null);
    const { isOpen: isAssignOpen, onOpen: onAssignOpen, onClose: onAssignClose } = useDisclosure();
    const [isAssigning, setIsAssigning] = useState(false);

    // Estados para programación en el modal de asignación
    const [scheduledAt, setScheduledAt] = useState('');
    const [scheduledEndAt, setScheduledEndAt] = useState('');
    const [schedulingMode, setSchedulingMode] = useState('specific');
    const [assignmentsData, setAssignmentsData] = useState([]);

    const { currentTime, loading: timeLoading } = useServerTime();
    const navigate = useNavigate();
    const toast = useToast();

    const fetchExams = async () => {
        try {
            setLoading(true);
            const data = await getExams();
            const user = JSON.parse(sessionStorage.getItem('user'));
            const isAdmin = user?.role === 'ADMIN';
            const instId = Number(user?.institution_id || user?.company_id);
            const userId = Number(user?.id || user?.user_id);

            console.log('GestionExamenes: user from session:', user);
            console.log('GestionExamenes: instId:', instId, 'userId:', userId);

            // Si es admin ve todo de su institución. Si es profesor ve lo suyo.
            const filtered = data.filter(e => {
                const examInstId = Number(e.institution_id || e.company_id);
                const examProfId = Number(e.professor_id);

                const matchesInst = examInstId === instId;
                if (isAdmin) return matchesInst;
                return examProfId === userId;
            });
            console.log('GestionExamenes: filtered exams:', filtered);
            setExams(filtered);
        } catch (error) {
            console.error('Error fetching exams:', error);
            toast({
                title: 'Error al cargar exámenes',
                status: 'error',
                duration: 3000,
            });
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchExams();
    }, []);

    const handleOpenAssign = async (exam) => {
        setCurrentExam(exam);
        setScheduledAt('');
        setScheduledEndAt('');
        setSchedulingMode('specific');
        onAssignOpen();
        setIsAssigning(true);
        try {
            console.log('Abriendo asignación para examen:', exam.id_exam || exam.id);

            const [allStudents, currentAssignments] = await Promise.all([
                getStudents().catch(err => { console.error('Error fetching students:', err); return []; }),
                getExamAssignments(exam.id_exam || exam.id).catch(err => { console.error('Error fetching assignments:', err); return []; })
            ]);

            const user = JSON.parse(sessionStorage.getItem('user'));
            const instId = Number(user?.institution_id || user?.company_id);

            // Filtrar estudiantes de la misma institución (comparación numérica)
            const filteredStudents = (allStudents || []).filter(s => {
                const studentInstId = Number(s.institution_id || s.company_id);
                return studentInstId === instId;
            });

            setStudents(filteredStudents);
            setAssignmentsData(currentAssignments || []);

            // Mapear IDs asignados para marcar en la lista
            const ids = (currentAssignments || []).map(a => Number(a.student_id));
            setAssignedStudentsIds(ids);
        } catch (error) {
            console.error('Error loading assignment data:', error);
            toast({
                title: 'Error al cargar datos de asignación',
                description: error.message,
                status: 'error'
            });
        } finally {
            setIsAssigning(false);
        }
    };

    const handleAssign = async (studentId) => {
        if (!currentExam) return;
        try {
            const sid = Number(studentId);
            const eid = Number(currentExam.id_exam || currentExam.id);

            const start = scheduledAt ? new Date(scheduledAt).toISOString() : null;
            const end = schedulingMode === 'range' && scheduledEndAt ? new Date(scheduledEndAt).toISOString() : null;

            const assignment = await assignExam(eid, sid, start, end);

            setAssignedStudentsIds(prev => [...prev, sid]);
            setAssignmentsData(prev => [...prev, {
                student_id: sid,
                scheduled_at: start,
                scheduled_end_at: end
            }]);

            toast({
                title: 'Estudiante asignado',
                description: start ? `Programado para ${new Date(start).toLocaleString()}` : '',
                status: 'success',
                duration: 2000,
            });
        } catch (error) {
            toast({
                title: 'Error al asignar',
                description: error.message,
                status: 'error',
            });
        }
    };

    const handleUnassign = async (studentId) => {
        if (!currentExam) return;
        try {
            const sid = Number(studentId);
            const eid = Number(currentExam.id_exam || currentExam.id);

            await unassignExam(eid, sid);

            setAssignedStudentsIds(prev => prev.filter(id => id !== sid));
            setAssignmentsData(prev => prev.filter(a => Number(a.student_id) !== sid));

            toast({
                title: 'Asignación removida',
                status: 'info',
                duration: 2000,
            });
        } catch (error) {
            toast({
                title: 'Error al desasignar',
                description: error.message,
                status: 'error',
            });
        }
    };

    const handleDelete = async (id) => {
        if (!window.confirm('¿Estás seguro de eliminar este examen? Esta acción no se puede deshacer.')) return;

        try {
            await deleteExam(id);
            toast({ title: 'Examen eliminado', status: 'success' });
            fetchExams();
        } catch (error) {
            toast({ title: 'Error al eliminar', description: error.message, status: 'error' });
        }
    };

    const user = JSON.parse(sessionStorage.getItem('user'));
    const isAdmin = user?.role === 'ADMIN';

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <VStack align="start" spacing={0}>
                    <Heading size="lg">{isAdmin ? 'Exámenes de la Institución' : 'Mis Exámenes'}</Heading>
                    <Text color="gray.500" fontSize="sm">
                        {isAdmin ? 'Visualiza y supervisa todas las evaluaciones' : 'Gestiona y monitorea tus evaluaciones'}
                    </Text>
                </VStack>
                {!isAdmin && (
                    <Button
                        leftIcon={<FiPlus />}
                        colorScheme="brand"
                        onClick={() => navigate('/professor/constructor')}
                    >
                        Crear Nuevo Examen
                    </Button>
                )}
            </HStack>

            {/* Stats Rápidas */}
            <SimpleGrid columns={{ base: 1, md: 3 }} spacing={6} mb={8}>
                <Card variant="outline">
                    <CardBody>
                        <Stat>
                            <StatLabel color="gray.500">Total de Exámenes</StatLabel>
                            <StatNumber>{exams.length}</StatNumber>
                        </Stat>
                    </CardBody>
                </Card>
                <Card variant="outline">
                    <CardBody>
                        <Stat>
                            <StatLabel color="gray.500">Exámenes Activos</StatLabel>
                            <StatNumber>{exams.filter(e => e.is_active).length}</StatNumber>
                        </Stat>
                    </CardBody>
                </Card>
            </SimpleGrid>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto" border="1px solid" borderColor="gray.100">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Título del Examen</Th>
                            {isAdmin && <Th>Profesor</Th>}
                            <Th>Fecha de Creación</Th>
                            <Th>Estado</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {loading ? (
                            <Tr>
                                <Td colSpan={isAdmin ? 5 : 4} textAlign="center" py={10}>
                                    <Spinner color="brand.500" />
                                    <Text mt={2} color="gray.500">Cargando exámenes...</Text>
                                </Td>
                            </Tr>
                        ) : (
                            exams.map((exam) => (
                                <Tr key={exam.id_exam}>
                                    <Td>
                                        <VStack align="start" spacing={0}>
                                            <Text fontWeight="bold">{exam.title}</Text>
                                            <Text fontSize="xs" color="gray.500" noOfLines={1}>{exam.description || 'Sin descripción'}</Text>
                                            {exam.scheduled_at && (
                                                <Badge mt={1} colorScheme="blue" variant="subtle" fontSize="10px">
                                                    {exam.scheduled_end_at
                                                        ? `🕒 ${new Date(exam.scheduled_at).toLocaleString()} - ${new Date(exam.scheduled_end_at).toLocaleString()}`
                                                        : `📅 ${new Date(exam.scheduled_at).toLocaleString()}`
                                                    }
                                                </Badge>
                                            )}
                                        </VStack>
                                    </Td>
                                    {isAdmin && (
                                        <Td>
                                            <Text fontSize="sm" fontWeight="bold">{exam.professor_name || 'Desconocido'}</Text>
                                            <Text fontSize="xs" color="gray.500">ID: {exam.professor_id}</Text>
                                        </Td>
                                    )}
                                    <Td fontSize="sm">
                                        {new Date(exam.created_at).toLocaleDateString()}
                                    </Td>
                                    <Td>
                                        <Badge colorScheme={exam.is_active ? 'green' : 'gray'}>
                                            {exam.is_active ? 'ACTIVO' : 'INACTIVO'}
                                        </Badge>
                                    </Td>
                                    <Td>
                                        <HStack spacing={2}>
                                            <Tooltip label="Monitorear en Vivo">
                                                <IconButton
                                                    icon={<FiBarChart2 />}
                                                    size="sm"
                                                    colorScheme="blue"
                                                    variant="ghost"
                                                    onClick={() => navigate(`/professor/monitor/${exam.id_exam}`)}
                                                    aria-label="Monitorear"
                                                />
                                            </Tooltip>
                                            <Tooltip label="Asignar Estudiantes">
                                                <IconButton
                                                    icon={<FiUsers />}
                                                    size="sm"
                                                    colorScheme="purple"
                                                    variant="ghost"
                                                    onClick={() => handleOpenAssign(exam)}
                                                    aria-label="Asignar"
                                                />
                                            </Tooltip>
                                            {!isAdmin && (
                                                <>
                                                    <IconButton
                                                        icon={<FiEdit2 />}
                                                        size="sm"
                                                        variant="ghost"
                                                        onClick={() => navigate(`/professor/constructor?id=${exam.id_exam}`)}
                                                        aria-label="Editar"
                                                    />
                                                    <IconButton
                                                        icon={<FiTrash2 />}
                                                        size="sm"
                                                        colorScheme="red"
                                                        variant="ghost"
                                                        onClick={() => handleDelete(exam.id_exam)}
                                                        aria-label="Eliminar"
                                                    />
                                                </>
                                            )}
                                        </HStack>
                                    </Td>
                                </Tr>
                            ))
                        )}
                        {!loading && exams.length === 0 && (
                            <Tr>
                                <Td colSpan={isAdmin ? 5 : 4} textAlign="center" py={10}>
                                    <VStack spacing={3}>
                                        <Text color="gray.500">Aún no has creado ningún examen.</Text>
                                        <Button
                                            size="sm"
                                            variant="outline"
                                            colorScheme="brand"
                                            onClick={() => navigate('/professor/constructor')}
                                        >
                                            Comenzar ahora
                                        </Button>
                                    </VStack>
                                </Td>
                            </Tr>
                        )}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal de Asignación */}
            <Modal isOpen={isAssignOpen} onClose={onAssignClose} size="xl">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Asignar Estudiantes: {currentExam?.title}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody pb={6}>
                        <VStack align="stretch" spacing={4}>
                            <Box bg="gray.50" p={4} borderRadius="lg" border="1px solid" borderColor="gray.200">
                                <HStack justify="space-between" mb={3}>
                                    <Heading size="xs" textTransform="uppercase" color="gray.600">Programación de la Aplicación</Heading>
                                    <Badge colorScheme="purple" variant="solid">
                                        Hora Servidor: {currentTime.toLocaleTimeString()}
                                    </Badge>
                                </HStack>
                                <RadioGroup onChange={setSchedulingMode} value={schedulingMode} mb={4}>
                                    <Stack direction="row" spacing={5}>
                                        <Radio value="specific" colorScheme="brand">Fecha específica</Radio>
                                        <Radio value="range" colorScheme="brand">Período (Rango)</Radio>
                                    </Stack>
                                </RadioGroup>

                                <SimpleGrid columns={{ base: 1, md: 2 }} spacing={4}>
                                    <FormControl>
                                        <FormLabel fontSize="sm">{schedulingMode === 'specific' ? 'Fecha y Hora de Aplicación' : 'Fecha y Hora de Inicio'}</FormLabel>
                                        <Input
                                            size="sm"
                                            bg="white"
                                            type="datetime-local"
                                            value={scheduledAt}
                                            onChange={(e) => setScheduledAt(e.target.value)}
                                        />
                                    </FormControl>

                                    {schedulingMode === 'range' && (
                                        <FormControl>
                                            <FormLabel fontSize="sm">Fecha y Hora de Fin</FormLabel>
                                            <Input
                                                size="sm"
                                                bg="white"
                                                type="datetime-local"
                                                value={scheduledEndAt}
                                                onChange={(e) => setScheduledEndAt(e.target.value)}
                                            />
                                        </FormControl>
                                    )}
                                </SimpleGrid>
                                <Text fontSize="10px" color="gray.500" mt={2}>
                                    * La fecha configurada arriba se aplicará a los estudiantes que asignes a continuación.
                                </Text>
                            </Box>

                            <Divider />

                            <Text fontSize="sm" color="gray.600">
                                Selecciona los estudiantes que deben realizar esta evaluación con la programación anterior.
                            </Text>

                            <Box maxHeight="400px" overflowY="auto" border="1px solid" borderColor="gray.100" borderRadius="md">
                                <Table variant="simple" size="sm">
                                    <Thead bg="gray.50" position="sticky" top={0} zIndex={1}>
                                        <Tr>
                                            <Th>Estudiante</Th>
                                            <Th>Programación</Th>
                                            <Th textAlign="center">Acción</Th>
                                        </Tr>
                                    </Thead>
                                    <Tbody>
                                        {students.map((student) => {
                                            const studentAssignments = assignmentsData.filter(a => Number(a.student_id) === Number(student.id || student.user_id));
                                            const latestAssignment = studentAssignments.sort((a, b) => b.attempt_number - a.attempt_number)[0];
                                            const isAssigned = !!latestAssignment;

                                            return (
                                                <Tr key={student.id || student.user_id}>
                                                    <Td>
                                                        <VStack align="start" spacing={0}>
                                                            <Text fontWeight="medium">{student.name || 'Sin nombre'}</Text>
                                                            <Text fontSize="xs" color="gray.500">{student.email}</Text>
                                                            {studentAssignments.length > 1 && (
                                                                <Badge variant="ghost" colorScheme="gray" fontSize="9px">
                                                                    {studentAssignments.length} Intentos previos
                                                                </Badge>
                                                            )}
                                                        </VStack>
                                                    </Td>
                                                    <Td>
                                                        {isAssigned && (
                                                            <VStack align="start" spacing={1}>
                                                                <Badge fontSize="9px" colorScheme={latestAssignment.status === 'COMPLETED' ? 'green' : 'blue'} variant="outline">
                                                                    {latestAssignment.status === 'COMPLETED' ? 'Finalizado' : 'Programado'}
                                                                </Badge>
                                                                {latestAssignment.scheduled_at && (
                                                                    <Text fontSize="10px" color="gray.600">
                                                                        {new Date(latestAssignment.scheduled_at).toLocaleString()}
                                                                    </Text>
                                                                )}
                                                            </VStack>
                                                        )}
                                                    </Td>
                                                    <Td textAlign="center">
                                                        <HStack justify="center">
                                                            {isAssigned && latestAssignment.status !== 'COMPLETED' && (
                                                                <Button
                                                                    size="xs"
                                                                    colorScheme="red"
                                                                    variant="ghost"
                                                                    onClick={() => handleUnassign(student.id || student.user_id)}
                                                                >
                                                                    Cancelar
                                                                </Button>
                                                            )}
                                                            <Button
                                                                size="xs"
                                                                colorScheme="brand"
                                                                leftIcon={<FiUserPlus />}
                                                                onClick={() => handleAssign(student.id || student.user_id)}
                                                            >
                                                                {isAssigned ? 'Reprogramar' : 'Asignar'}
                                                            </Button>
                                                        </HStack>
                                                    </Td>
                                                </Tr>
                                            );
                                        })}
                                        {students.length === 0 && !isAssigning && (
                                            <Tr>
                                                <Td colSpan={3} textAlign="center" py={10}>
                                                    <Text color="gray.500" fontSize="sm">No hay estudiantes registrados en la institución.</Text>
                                                </Td>
                                            </Tr>
                                        )}
                                    </Tbody>
                                </Table>
                            </Box>
                        </VStack>
                    </ModalBody>
                    <ModalFooter>
                        <Button onClick={onAssignClose} colorScheme="brand">Finalizar</Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionExamenes;
