
import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Select,
    useToast,
    HStack,
    VStack,
    Text,
    Textarea,
    Icon,
    Divider,
    Spinner,
    Center,
    Checkbox,
    InputGroup,
    InputLeftElement,
    Select as ChakraSelect // Renaming to avoid conflict if Select is already imported or used differently
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiDatabase, FiPower, FiDownload, FiUpload, FiEye, FiSearch } from 'react-icons/fi';
import { getQuestions, createQuestion, updateQuestion, deleteQuestion, deleteQuestionsBulk, getEmployees } from '../../services/api';

const GestionPreguntas = () => {
    const [questions, setQuestions] = useState([]);
    const [categories, setCategories] = useState([]); // Nuevo estado para catálogo de categorías
    const [professors, setProfessors] = useState([]); // Lista de profesores de la institución
    const [isNewCategory, setIsNewCategory] = useState(false);
    const [currentQuestion, setCurrentQuestion] = useState(null);
    const [isReadOnly, setIsReadOnly] = useState(false);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const { isOpen: isImportModalOpen, onOpen: onImportModalOpen, onClose: onImportModalClose } = useDisclosure();
    const [importType, setImportType] = useState('MULTIPLE_CHOICE');
    const [importFile, setImportFile] = useState(null);
    const [importErrors, setImportErrors] = useState([]);
    const [importSuccess, setImportSuccess] = useState(0);
    const [isLoading, setIsLoading] = useState(false);
    const [isDragging, setIsDragging] = useState(false);
    const [selectedIds, setSelectedIds] = useState(new Set());

    // Filtros
    const [searchTerm, setSearchTerm] = useState('');
    const [filterCategory, setFilterCategory] = useState('');
    const [filterType, setFilterType] = useState('');

    const toast = useToast();

    // Lógica de filtrado
    const filteredQuestions = questions.filter(q => {
        const matchesSearch = q.content.question.toLowerCase().includes(searchTerm.toLowerCase());
        const matchesCategory = filterCategory ? q.category === filterCategory : true;
        const matchesType = filterType ? q.type === filterType : true;
        return matchesSearch && matchesCategory && matchesType;
    });

    const fetchQuestions = async () => {
        try {
            const data = await getQuestions();
            const user = JSON.parse(sessionStorage.getItem('user'));
            const isAdmin = user?.role === 'ADMIN';
            const instId = Number(user?.institution_id || user?.company_id);
            const userId = Number(user?.id || user?.user_id);

            console.log('GestionPreguntas: instId:', instId, 'userId:', userId);

            // Si es admin, ve todo lo de su institución. Si es profesor, solo lo propio.
            const filteredQuestions = (data || []).filter(q => {
                const qInstId = Number(q.institution_id || q.company_id);
                const qProfId = Number(q.professor_id);

                const matchesInst = qInstId === instId;
                if (isAdmin) return matchesInst;
                return qProfId === userId;
            });

            // Ordenar por: Profesor, Categoría, Orden de creación (created_at ASC)
            const sortedQuestions = [...filteredQuestions].sort((a, b) => {
                // 1. Ordenar por profesor (professor_id)
                const profA = Number(a.professor_id || 0);
                const profB = Number(b.professor_id || 0);
                if (profA !== profB) {
                    return profA - profB;
                }

                // 2. Ordenar por categoría
                const catA = (a.category || '').toLowerCase();
                const catB = (b.category || '').toLowerCase();
                if (catA !== catB) {
                    return catA.localeCompare(catB);
                }

                // 3. Ordenar por fecha de creación (más antiguas primero)
                const dateA = a.created_at ? new Date(a.created_at).getTime() : 0;
                const dateB = b.created_at ? new Date(b.created_at).getTime() : 0;
                return dateA - dateB;
            });

            console.log('GestionPreguntas: filtered:', sortedQuestions);
            setQuestions(sortedQuestions);

            // Extraer catálogo de categorías únicas
            const uniqueCategories = [...new Set(filteredQuestions.map(q => q.category).filter(Boolean))];
            setCategories(uniqueCategories.sort());
        } catch (error) {
            console.error('Error fetching questions:', error);
        }
    };

    const fetchProfessors = async () => {
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            const isAdmin = user?.role === 'ADMIN';
            const institutionId = user ? (user.institution_id || user.company_id) : null;

            if (isAdmin && institutionId) {
                const allEmployees = await getEmployees();
                // Filtrar por institución y asegurarnos que sean profesores
                const filtered = allEmployees.filter(e => {
                    const matchesInst = Number(e.institution_id) === Number(institutionId) || Number(e.company_id) === Number(institutionId);
                    const isProf = ((e.role || '').toUpperCase() === 'PROFESOR' || (e.role || '').toUpperCase() === 'EMPLOYEE');
                    return matchesInst && isProf && e.is_active;
                });
                setProfessors(filtered);
            }
        } catch (error) {
            console.error('Error fetching professors:', error);
        }
    };

    useEffect(() => {
        fetchQuestions();
        fetchProfessors();
    }, []);

    const handleSelectAll = (e) => {
        if (e.target.checked) {
            const allIds = new Set(questions.map(q => q.id_question));
            setSelectedIds(allIds);
        } else {
            setSelectedIds(new Set());
        }
    };

    const handleSelectOne = (id) => {
        const newSelected = new Set(selectedIds);
        if (newSelected.has(id)) {
            newSelected.delete(id);
        } else {
            newSelected.add(id);
        }
        setSelectedIds(newSelected);
    };

    const handleBulkDelete = async () => {
        if (selectedIds.size === 0) return;

        if (!window.confirm(`¿Estás seguro de que deseas eliminar ${selectedIds.size} preguntas?`)) {
            return;
        }

        try {
            const response = await deleteQuestionsBulk(Array.from(selectedIds));
            // Asumiendo que response es el JSON directo
            const { deleted, deactivated, errors } = response;

            let message = '';
            let status = 'success';

            if (errors > 0) {
                message += `Hubo ${errors} errores. `;
                status = 'warning';
            }
            if (deleted > 0) {
                message += `${deleted} eliminadas. `;
            }
            if (deactivated > 0) {
                message += `${deactivated} desactivadas (en uso). `;
            }

            toast({
                title: 'Operación completada',
                description: message,
                status: status,
                duration: 5000,
            });

            setSelectedIds(new Set());
            fetchQuestions();
        } catch (error) {
            console.error('Error in bulk delete:', error);
            toast({
                title: 'Error',
                description: 'Ocurrió un error al procesar la eliminación masiva.',
                status: 'error',
            });
        }
    };

    const handleCreate = () => {
        const user = JSON.parse(sessionStorage.getItem('user'));
        const isAdmin = user?.role === 'ADMIN';
        setIsNewCategory(false);
        setIsReadOnly(false);
        setCurrentQuestion({
            institution_id: user?.institution_id || user?.company_id,
            professor_id: isAdmin ? null : (user?.id || user?.user_id), // Admin debe seleccionar, profesor usa su ID
            category: categories.length > 0 ? categories[0] : '',
            type: 'MULTIPLE_CHOICE',
            content: {
                question: '',
                options: ['', '', '', '', ''], // 5 opciones
                media_url: ''
            },
            correct_answer: '',
            is_active: true
        });
        onOpen();
    };

    const handleEdit = (question) => {
        setIsNewCategory(false);
        setIsReadOnly(false);
        const editedQuestion = { ...question };

        // Asegurar que si es opción múltiple tenga las 5 opciones para editar
        if (editedQuestion.type === 'MULTIPLE_CHOICE' && editedQuestion.content.options.length < 5) {
            const currentOpts = [...editedQuestion.content.options];
            while (currentOpts.length < 5) currentOpts.push('');
            editedQuestion.content.options = currentOpts;
        }

        setCurrentQuestion(editedQuestion);
        onOpen();
    };

    const handleView = (question) => {
        setIsNewCategory(false);
        setIsReadOnly(true);
        setCurrentQuestion({ ...question });
        onOpen();
    };

    const handleDelete = async (question) => {
        if (!window.confirm('¿Estás seguro de que deseas eliminar esta pregunta?')) {
            return;
        }

        try {
            const response = await deleteQuestion(question.id_question);

            const status = response.status || response.data?.status;

            if (status === 'deactivated') {
                toast({
                    title: 'Pregunta desactivada',
                    description: 'La pregunta no se pudo eliminar porque está siendo utilizada en un examen, pero se ha desactivado para evitar futuros usos.',
                    status: 'warning',
                    duration: 5000,
                });
            } else {
                toast({
                    title: 'Pregunta eliminada',
                    description: 'La pregunta ha sido eliminada correctamente.',
                    status: 'success',
                });
            }
            fetchQuestions();
        } catch (error) {
            console.error('Error deleting question:', error);
            toast({
                title: 'Error al eliminar',
                description: 'Ocurrió un error al intentar eliminar la pregunta.',
                status: 'error',
            });
        }
    };

    const handleToggleStatus = async (question) => {
        try {
            const newStatus = !question.is_active;
            await updateQuestion(question.id_question, { ...question, is_active: newStatus });

            // Actualizar el estado local
            setQuestions(questions.map(q =>
                q.id_question === question.id_question ? { ...q, is_active: newStatus } : q
            ));

            toast({
                title: `Pregunta ${newStatus ? 'activada' : 'desactivada'}`,
                status: 'success',
                duration: 2000,
            });
        } catch (error) {
            toast({
                title: 'Error al cambiar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    // Función para generar y descargar template CSV con codificación UTF-8
    const downloadTemplate = (type) => {
        let csvContent = '';
        let filename = '';

        switch (type) {
            case 'MULTIPLE_CHOICE':
                csvContent = 'Pregunta,Opción 1,Opción 2,Opción 3,Opción 4,Opción 5,Respuesta Correcta,Categoría\n';
                csvContent += '"¿Cuál es la capital de Francia?","París","Londres","Madrid","Berlín","Roma","París","Geografía"\n';
                csvContent += '"¿Cuántos continentes hay en el mundo?","5","6","7","8","9","7","Geografía"\n';
                filename = 'template_opcion_multiple.csv';
                break;
            case 'TRUE_FALSE':
                csvContent = 'Pregunta,Respuesta Correcta,Categoría\n';
                csvContent += '"El sol es una estrella","Verdadero","Ciencias"\n';
                csvContent += '"La Tierra es plana","Falso","Ciencias"\n';
                csvContent += '"El agua hierve a 100 grados Celsius","Verdadero","Ciencias"\n';
                filename = 'template_verdadero_falso.csv';
                break;
            case 'OPEN':
                csvContent = 'Pregunta,Respuesta Correcta (Ejemplo),Categoría\n';
                csvContent += '"Explica el proceso de fotosíntesis","La fotosíntesis es el proceso mediante el cual las plantas convierten la luz solar en energía química...","Ciencias"\n';
                csvContent += '"Describe las causas de la Primera Guerra Mundial","Las causas principales fueron el imperialismo, el nacionalismo, las alianzas militares...","Historia"\n';
                filename = 'template_abierta.csv';
                break;
            default:
                return;
        }

        // Agregar BOM UTF-8 para asegurar codificación correcta en Excel y otros programas
        const BOM = '\uFEFF';
        const csvWithBOM = BOM + csvContent;

        // Crear blob con codificación UTF-8 explícita
        const blob = new Blob([csvWithBOM], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);
        link.setAttribute('href', url);
        link.setAttribute('download', filename);
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);

        toast({
            title: 'Template descargado',
            description: `Se descargó el template para preguntas de tipo ${type === 'MULTIPLE_CHOICE' ? 'Opción Múltiple' : type === 'TRUE_FALSE' ? 'Verdadero/Falso' : 'Abierta'}`,
            status: 'success',
            duration: 3000
        });
    };

    // Función para parsear CSV
    const parseCSV = (text) => {
        const lines = text.split('\n').filter(line => line.trim());
        if (lines.length < 2) return [];

        const headers = lines[0].split(',').map(h => h.trim().replace(/^"|"$/g, ''));
        const rows = [];

        for (let i = 1; i < lines.length; i++) {
            const values = [];
            let currentValue = '';
            let inQuotes = false;

            for (let j = 0; j < lines[i].length; j++) {
                const char = lines[i][j];
                if (char === '"') {
                    inQuotes = !inQuotes;
                } else if (char === ',' && !inQuotes) {
                    values.push(currentValue.trim());
                    currentValue = '';
                } else {
                    currentValue += char;
                }
            }
            values.push(currentValue.trim());

            if (values.length === headers.length) {
                const row = {};
                headers.forEach((header, idx) => {
                    row[header] = values[idx] || '';
                });
                rows.push(row);
            }
        }

        return rows;
    };

    // Función para validar y procesar preguntas importadas (validación completa antes de insertar)
    const validateAndProcessQuestions = async (rows, type) => {
        const errors = [];
        const validQuestions = [];
        const user = JSON.parse(sessionStorage.getItem('user'));
        const isAdmin = user?.role === 'ADMIN';
        const institutionId = user?.institution_id || user?.company_id;
        const defaultProfessorId = isAdmin ? null : (user?.id || user?.user_id);

        // Validar que si es admin, haya profesores disponibles
        if (isAdmin && professors.length === 0) {
            errors.push({
                row: 'Sistema',
                question: 'Validación general',
                errors: ['Debes tener al menos un profesor registrado para importar preguntas como administrador.']
            });
            return { validQuestions, errors };
        }

        // Obtener preguntas existentes para verificar duplicados
        let existingQuestions = [];
        try {
            const allQuestions = await getQuestions();
            const instId = Number(institutionId);
            const userId = Number(defaultProfessorId);

            // Filtrar preguntas según el rol
            // Para admin: todas las preguntas de la institución
            // Para profesor: solo sus propias preguntas
            existingQuestions = (allQuestions || []).filter(q => {
                const qInstId = Number(q.institution_id || q.company_id);
                const qProfId = Number(q.professor_id);

                if (isAdmin) {
                    return qInstId === instId;
                } else {
                    return qProfId === userId;
                }
            });
        } catch (error) {
            console.error('Error fetching existing questions:', error);
            // Continuar sin validación de duplicados si hay error
        }

        // Crear un mapa de preguntas existentes por profesor para búsqueda rápida
        const questionsByProfessor = {};
        existingQuestions.forEach(q => {
            const profId = Number(q.professor_id);
            if (!questionsByProfessor[profId]) {
                questionsByProfessor[profId] = [];
            }
            questionsByProfessor[profId].push({
                question: (q.content?.question || '').trim(),
                answer: (q.correct_answer || '').trim()
            });
        });

        rows.forEach((row, index) => {
            const rowNum = index + 2; // +2 porque la primera fila es el header y el índice empieza en 0
            const questionErrors = [];

            // Validar pregunta
            const question = (row['Pregunta'] || '').trim();
            if (!question) {
                questionErrors.push('La pregunta no puede estar vacía');
            } else {
                // Validación de longitud según tipo
                const minLength = type === 'OPEN' ? 5 : 10;
                if (question.length < minLength) {
                    questionErrors.push(`La pregunta debe tener al menos ${minLength} caracteres`);
                }
            }

            // Validar categoría
            const category = (row['Categoría'] || '').trim();
            if (!category) {
                questionErrors.push('La categoría no puede estar vacía');
            } else if (category.length < 2) {
                questionErrors.push('La categoría debe tener al menos 2 caracteres');
            }

            // Validar según el tipo
            if (type === 'MULTIPLE_CHOICE') {
                const options = [
                    (row['Opción 1'] || '').trim(),
                    (row['Opción 2'] || '').trim(),
                    (row['Opción 3'] || '').trim(),
                    (row['Opción 4'] || '').trim(),
                    (row['Opción 5'] || '').trim()
                ];

                const filledOptions = options.filter(opt => opt && opt.length > 0);

                if (filledOptions.length < 2) {
                    questionErrors.push('Debe haber al menos 2 opciones con contenido');
                }

                // Validar que no haya opciones duplicadas
                const uniqueOptions = [...new Set(filledOptions)];
                if (uniqueOptions.length !== filledOptions.length) {
                    questionErrors.push('No puede haber opciones duplicadas');
                }

                // Validar que cada opción tenga contenido válido
                filledOptions.forEach((opt, idx) => {
                    if (opt.length < 1) {
                        questionErrors.push(`La opción ${idx + 1} no puede estar vacía`);
                    }
                });

                const correctAnswer = (row['Respuesta Correcta'] || '').trim();
                if (!correctAnswer) {
                    questionErrors.push('La respuesta correcta no puede estar vacía');
                } else if (!filledOptions.includes(correctAnswer)) {
                    questionErrors.push(`La respuesta correcta "${correctAnswer}" no coincide con ninguna de las opciones proporcionadas`);
                }

                // Validar duplicado: pregunta y respuesta correcta para el profesor
                if (questionErrors.length === 0) {
                    let isDuplicate = false;
                    let duplicateMessage = '';

                    if (isAdmin) {
                        // Para admin: verificar contra TODOS los profesores de la institución
                        // porque no sabemos a cuál se asignará
                        const duplicateProfessors = [];
                        Object.keys(questionsByProfessor).forEach(profId => {
                            const profQuestions = questionsByProfessor[profId];
                            const duplicate = profQuestions.find(existing =>
                                existing.question === question && existing.answer === correctAnswer
                            );
                            if (duplicate) {
                                duplicateProfessors.push(profId);
                            }
                        });

                        if (duplicateProfessors.length > 0) {
                            isDuplicate = true;
                            duplicateMessage = `Ya existe una pregunta idéntica con la misma respuesta correcta para ${duplicateProfessors.length} profesor(es) de la institución (IDs: ${duplicateProfessors.join(', ')})`;
                        }
                    } else {
                        // Para profesor: verificar solo contra sus propias preguntas
                        const profId = Number(defaultProfessorId);
                        const profQuestions = questionsByProfessor[profId] || [];
                        const duplicate = profQuestions.find(existing =>
                            existing.question === question && existing.answer === correctAnswer
                        );

                        if (duplicate) {
                            isDuplicate = true;
                            duplicateMessage = `Ya tienes una pregunta idéntica con la misma respuesta correcta en tu banco de reactivos`;
                        }
                    }

                    if (isDuplicate) {
                        questionErrors.push(duplicateMessage);
                    } else {
                        validQuestions.push({
                            institution_id: institutionId,
                            professor_id: defaultProfessorId,
                            category,
                            type: 'MULTIPLE_CHOICE',
                            content: {
                                question,
                                options: options,
                                media_url: ''
                            },
                            correct_answer: correctAnswer,
                            is_active: true
                        });
                    }
                }
            } else if (type === 'TRUE_FALSE') {
                const correctAnswer = (row['Respuesta Correcta'] || '').trim();
                if (!correctAnswer) {
                    questionErrors.push('La respuesta correcta no puede estar vacía');
                } else if (correctAnswer !== 'Verdadero' && correctAnswer !== 'Falso') {
                    questionErrors.push(`La respuesta correcta debe ser exactamente "Verdadero" o "Falso" (se encontró: "${correctAnswer}")`);
                }

                // Validar duplicado: pregunta y respuesta correcta para el profesor
                if (questionErrors.length === 0) {
                    let isDuplicate = false;
                    let duplicateMessage = '';

                    if (isAdmin) {
                        // Para admin: verificar contra TODOS los profesores de la institución
                        const duplicateProfessors = [];
                        Object.keys(questionsByProfessor).forEach(profId => {
                            const profQuestions = questionsByProfessor[profId];
                            const duplicate = profQuestions.find(existing =>
                                existing.question === question && existing.answer === correctAnswer
                            );
                            if (duplicate) {
                                duplicateProfessors.push(profId);
                            }
                        });

                        if (duplicateProfessors.length > 0) {
                            isDuplicate = true;
                            duplicateMessage = `Ya existe una pregunta idéntica con la misma respuesta correcta para ${duplicateProfessors.length} profesor(es) de la institución (IDs: ${duplicateProfessors.join(', ')})`;
                        }
                    } else {
                        // Para profesor: verificar solo contra sus propias preguntas
                        const profId = Number(defaultProfessorId);
                        const profQuestions = questionsByProfessor[profId] || [];
                        const duplicate = profQuestions.find(existing =>
                            existing.question === question && existing.answer === correctAnswer
                        );

                        if (duplicate) {
                            isDuplicate = true;
                            duplicateMessage = `Ya tienes una pregunta idéntica con la misma respuesta correcta en tu banco de reactivos`;
                        }
                    }

                    if (isDuplicate) {
                        questionErrors.push(duplicateMessage);
                    } else {
                        validQuestions.push({
                            institution_id: institutionId,
                            professor_id: defaultProfessorId,
                            category,
                            type: 'TRUE_FALSE',
                            content: {
                                question,
                                options: ['Verdadero', 'Falso'],
                                media_url: ''
                            },
                            correct_answer: correctAnswer,
                            is_active: true
                        });
                    }
                }
            } else if (type === 'OPEN') {
                const correctAnswer = (row['Respuesta Correcta (Ejemplo)'] || '').trim();
                if (!correctAnswer) {
                    questionErrors.push('Debe proporcionar un ejemplo de respuesta correcta');
                } else if (correctAnswer.length < 1) {
                    questionErrors.push('El ejemplo de respuesta correcta debe tener al menos 1 carácter');
                }

                // Validar duplicado: pregunta y respuesta correcta para el profesor
                if (questionErrors.length === 0) {
                    let isDuplicate = false;
                    let duplicateMessage = '';

                    if (isAdmin) {
                        // Para admin: verificar contra TODOS los profesores de la institución
                        const duplicateProfessors = [];
                        Object.keys(questionsByProfessor).forEach(profId => {
                            const profQuestions = questionsByProfessor[profId];
                            const duplicate = profQuestions.find(existing =>
                                existing.question === question && existing.answer === correctAnswer
                            );
                            if (duplicate) {
                                duplicateProfessors.push(profId);
                            }
                        });

                        if (duplicateProfessors.length > 0) {
                            isDuplicate = true;
                            duplicateMessage = `Ya existe una pregunta idéntica con la misma respuesta correcta para ${duplicateProfessors.length} profesor(es) de la institución (IDs: ${duplicateProfessors.join(', ')})`;
                        }
                    } else {
                        // Para profesor: verificar solo contra sus propias preguntas
                        const profId = Number(defaultProfessorId);
                        const profQuestions = questionsByProfessor[profId] || [];
                        const duplicate = profQuestions.find(existing =>
                            existing.question === question && existing.answer === correctAnswer
                        );

                        if (duplicate) {
                            isDuplicate = true;
                            duplicateMessage = `Ya tienes una pregunta idéntica con la misma respuesta correcta en tu banco de reactivos`;
                        }
                    }

                    if (isDuplicate) {
                        questionErrors.push(duplicateMessage);
                    } else {
                        validQuestions.push({
                            institution_id: institutionId,
                            professor_id: defaultProfessorId,
                            category,
                            type: 'OPEN',
                            content: {
                                question,
                                options: [],
                                media_url: ''
                            },
                            correct_answer: correctAnswer,
                            is_active: true
                        });
                    }
                }
            }

            if (questionErrors.length > 0) {
                errors.push({
                    row: rowNum,
                    question: question || '(sin pregunta)',
                    errors: questionErrors
                });
            }
        });

        // Validar que si es admin, todas las preguntas válidas tengan profesor asignado
        if (isAdmin && validQuestions.length > 0) {
            validQuestions.forEach((q, idx) => {
                if (!q.professor_id) {
                    // Asignar el primer profesor disponible
                    if (professors.length > 0) {
                        q.professor_id = professors[0].id || professors[0].user_id;
                    } else {
                        errors.push({
                            row: `Pregunta ${idx + 1}`,
                            question: q.content.question,
                            errors: ['No hay profesores disponibles para asignar']
                        });
                    }
                }
            });
        }

        return { validQuestions, errors };
    };

    // Función para manejar la importación
    const handleImport = async () => {
        if (!importFile) {
            toast({
                title: 'Archivo requerido',
                description: 'Debes seleccionar un archivo CSV para importar',
                status: 'warning'
            });
            return;
        }

        setIsLoading(true);
        const reader = new FileReader();
        // Especificar codificación UTF-8 explícita
        reader.readAsText(importFile, 'UTF-8');

        reader.onload = async (e) => {
            try {
                let text = e.target.result;

                // Remover BOM UTF-8 si existe (algunos editores lo agregan)
                if (text.charCodeAt(0) === 0xFEFF) {
                    text = text.slice(1);
                }

                const rows = parseCSV(text);

                if (rows.length === 0) {
                    toast({
                        title: 'Archivo vacío',
                        description: 'El archivo CSV no contiene datos válidos',
                        status: 'error'
                    });
                    setIsLoading(false);
                    return;
                }

                // VALIDACIÓN COMPLETA ANTES DE INSERTAR NADA (incluye verificación de duplicados)
                const { validQuestions, errors } = await validateAndProcessQuestions(rows, importType);

                // Si hay errores, mostrar bitácora completa y NO insertar nada
                if (errors.length > 0) {
                    setImportErrors(errors);
                    setImportSuccess(0);
                    toast({
                        title: 'Errores de validación encontrados',
                        description: `Se encontraron ${errors.length} error(es) en ${errors.length} fila(s). Revisa la bitácora de incidencias antes de continuar.`,
                        status: 'error',
                        duration: 7000
                    });
                    setIsLoading(false);
                    return; // NO INSERTAR NADA SI HAY ERRORES
                }

                // Si no hay preguntas válidas después de la validación
                if (validQuestions.length === 0) {
                    toast({
                        title: 'No hay preguntas válidas',
                        description: 'No se encontraron preguntas válidas para importar después de la validación.',
                        status: 'error',
                        duration: 5000
                    });
                    setIsLoading(false);
                    return;
                }

                // SOLO SI TODAS LAS VALIDACIONES PASARON, PROCEDER CON LAS INSERCIONES
                let successCount = 0;
                const failedQuestions = [];

                for (const question of validQuestions) {
                    try {
                        await createQuestion(question);
                        successCount++;
                    } catch (error) {
                        console.error('Error importing question:', error);
                        failedQuestions.push({
                            question: question.content.question,
                            error: error.message || 'Error desconocido al insertar en la base de datos'
                        });
                    }
                }

                // Si hubo errores durante la inserción, agregarlos a la bitácora
                if (failedQuestions.length > 0) {
                    const insertionErrors = failedQuestions.map((fq, idx) => ({
                        row: `Inserción ${idx + 1}`,
                        question: fq.question,
                        errors: [fq.error]
                    }));
                    setImportErrors(insertionErrors);
                    toast({
                        title: 'Errores durante la inserción',
                        description: `${successCount} pregunta(s) se importaron exitosamente, pero ${failedQuestions.length} fallaron durante la inserción. Revisa la bitácora.`,
                        status: 'warning',
                        duration: 7000
                    });
                } else {
                    // Éxito total
                    setImportSuccess(successCount);
                    toast({
                        title: 'Importación completada exitosamente',
                        description: `Se importaron ${successCount} pregunta(s) correctamente.`,
                        status: 'success',
                        duration: 5000
                    });
                    fetchQuestions();
                    setTimeout(() => {
                        onImportModalClose();
                        setImportFile(null);
                        setImportErrors([]);
                        setImportSuccess(0);
                    }, 2000);
                }
            } catch (error) {
                console.error('Error processing file:', error);
                toast({
                    title: 'Error al procesar archivo',
                    description: error.message || 'Error desconocido al procesar el archivo CSV',
                    status: 'error'
                });
            } finally {
                setIsLoading(false);
            }
        };

        reader.onerror = () => {
            toast({
                title: 'Error al leer archivo',
                description: 'No se pudo leer el archivo. Asegúrate de que el archivo esté en formato CSV con codificación UTF-8.',
                status: 'error',
                duration: 5000
            });
            setIsLoading(false);
        };
    };

    const handleSave = async () => {
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            const isAdmin = user?.role === 'ADMIN';

            // Validar que no esté vacía la pregunta y tenga la longitud mínima
            const questionText = currentQuestion.content.question.trim();
            if (!questionText) {
                toast({ title: 'La pregunta no puede estar vacía', status: 'warning' });
                return;
            }

            // Validar longitud mínima según tipo de pregunta
            const minQuestionLength = currentQuestion.type === 'OPEN' ? 5 : 10;
            if (questionText.length < minQuestionLength) {
                toast({
                    title: 'Pregunta muy corta',
                    description: `La pregunta debe tener al menos ${minQuestionLength} caracteres`,
                    status: 'warning'
                });
                return;
            }

            // Validar que se haya seleccionado un profesor si es admin creando nueva pregunta
            if (isAdmin && !currentQuestion.id_question && !currentQuestion.professor_id) {
                toast({
                    title: 'Profesor requerido',
                    description: 'Debes seleccionar un profesor para asignar la pregunta.',
                    status: 'warning'
                });
                return;
            }

            // Validar que si es admin editando, tenga un profesor asignado
            if (isAdmin && currentQuestion.id_question && !currentQuestion.professor_id) {
                toast({
                    title: 'Profesor requerido',
                    description: 'La pregunta debe tener un profesor asignado.',
                    status: 'warning'
                });
                return;
            }

            // Validar que la respuesta correcta esté en las opciones (solo para MULTIPLE_CHOICE y TRUE_FALSE)
            if (currentQuestion.type === 'MULTIPLE_CHOICE' || currentQuestion.type === 'TRUE_FALSE') {
                const options = currentQuestion.content?.options || [];
                const correctAnswer = currentQuestion.correct_answer?.trim();

                if (!correctAnswer) {
                    toast({
                        title: 'Respuesta correcta requerida',
                        description: 'Debes especificar la respuesta correcta para este tipo de pregunta.',
                        status: 'warning'
                    });
                    return;
                }

                // Verificar que la respuesta correcta esté en las opciones
                const answerExists = options.some(opt => opt.trim() === correctAnswer);

                if (!answerExists) {
                    toast({
                        title: 'Respuesta correcta inválida',
                        description: `La respuesta correcta "${correctAnswer}" no coincide con ninguna de las opciones proporcionadas. Por favor, verifica que la respuesta correcta sea exactamente igual a una de las opciones.`,
                        status: 'error',
                        duration: 6000
                    });
                    return;
                }
            }

            // Validar preguntas abiertas: respuesta correcta debe tener al menos 1 carácter
            if (currentQuestion.type === 'OPEN') {
                const correctAnswer = currentQuestion.correct_answer?.trim();
                if (!correctAnswer || correctAnswer.length < 1) {
                    toast({
                        title: 'Respuesta correcta requerida',
                        description: 'Debes proporcionar un ejemplo de respuesta correcta (mínimo 1 carácter).',
                        status: 'warning'
                    });
                    return;
                }
            }

            if (currentQuestion.id_question) {
                // Preparar datos para actualización, asegurando que professor_id se envíe correctamente
                const updateData = {
                    ...currentQuestion,
                    professor_id: currentQuestion.professor_id ? Number(currentQuestion.professor_id) : null
                };
                // Eliminar campos que no deben enviarse
                delete updateData.id_question;
                delete updateData.professor_name;
                delete updateData.created_at;
                delete updateData.updated_at;

                await updateQuestion(currentQuestion.id_question, updateData);
                toast({ title: 'Pregunta actualizada', status: 'success' });
            } else {
                await createQuestion(currentQuestion);
                toast({ title: 'Pregunta creada', status: 'success' });
            }
            fetchQuestions();
            onClose();
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message || 'Posible pregunta duplicada',
                status: 'error',
                duration: 5000
            });
        }
    };

    const user = JSON.parse(sessionStorage.getItem('user'));
    const isAdmin = user?.role === 'ADMIN';

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <Heading size="lg">{isAdmin ? 'Banco de Reactivos Institucional' : 'Mi Banco de Reactivos'}</Heading>
                <HStack spacing={3}>
                    <Button
                        leftIcon={<FiDownload />}
                        colorScheme="purple"
                        variant="outline"
                        onClick={onImportModalOpen}
                    >
                        Importar CSV
                    </Button>
                    <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                        Nueva Pregunta
                    </Button>
                </HStack>
            </HStack>

            {questions.length === 0 ? (
                <Box
                    bg="white"
                    borderRadius="lg"
                    boxShadow="sm"
                    p={12}
                    textAlign="center"
                >
                    <VStack spacing={6}>
                        <Box>
                            <Icon
                                as={FiDatabase}
                                boxSize={16}
                                color="gray.300"
                                mb={4}
                            />
                            <Heading size="lg" color="gray.600" mb={2}>
                                No hay preguntas registradas
                            </Heading>
                            <VStack spacing={3} align="stretch" maxW="2xl" mx="auto" mt={4}>
                                <Text color="gray.600" fontSize="md" fontWeight="500">
                                    El banco de preguntas es fundamental para crear exámenes efectivos y personalizados.
                                </Text>
                                <Box
                                    bg="blue.50"
                                    p={4}
                                    borderRadius="md"
                                    borderLeft="4px solid"
                                    borderColor="blue.500"
                                    textAlign="left"
                                >
                                    <Text color="blue.900" fontSize="sm" fontWeight="600" mb={2}>
                                        ¿Por qué es importante tener preguntas registradas?
                                    </Text>
                                    <VStack align="start" spacing={2} fontSize="sm" color="blue.800">
                                        <Text>• Permite crear exámenes rápidamente seleccionando preguntas de tu banco</Text>
                                        <Text>• Facilita la reutilización de preguntas en diferentes evaluaciones</Text>
                                        <Text>• Organiza tus reactivos por categorías para mejor gestión</Text>
                                        <Text>• Ahorra tiempo al no tener que escribir preguntas desde cero cada vez</Text>
                                        <Text>• Mantiene la consistencia en la calidad de tus evaluaciones</Text>
                                    </VStack>
                                </Box>
                            </VStack>
                        </Box>
                        <Button
                            leftIcon={<FiPlus />}
                            colorScheme="brand"
                            size="lg"
                            onClick={handleCreate}
                        >
                            Crear Primera Pregunta
                        </Button>
                    </VStack>
                </Box>
            ) : (
                <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                    <VStack spacing={4} align="stretch" p={4} bg="gray.50" borderRadius="md" mb={2}>
                        <HStack justifyContent="space-between" alignItems="center">
                            <Badge colorScheme="blue" p={2} borderRadius="md" fontSize="md">
                                Total: {filteredQuestions.length} preguntas
                            </Badge>
                            {selectedIds.size > 0 && (
                                <Button
                                    leftIcon={<FiTrash2 />}
                                    colorScheme="red"
                                    size="sm"
                                    onClick={handleBulkDelete}
                                >
                                    Eliminar seleccionadas ({selectedIds.size})
                                </Button>
                            )}
                        </HStack>

                        <HStack spacing={4}>
                            <InputGroup>
                                <InputLeftElement pointerEvents="none">
                                    <Icon as={FiSearch} color="gray.400" />
                                </InputLeftElement>
                                <Input
                                    placeholder="Buscar pregunta..."
                                    bg="white"
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                />
                            </InputGroup>

                            <ChakraSelect
                                placeholder="Todas las categorías"
                                bg="white"
                                w="250px"
                                value={filterCategory}
                                onChange={(e) => setFilterCategory(e.target.value)}
                            >
                                {categories.map(cat => (
                                    <option key={cat} value={cat}>{cat}</option>
                                ))}
                            </ChakraSelect>

                            <ChakraSelect
                                placeholder="Todos los tipos"
                                bg="white"
                                w="200px"
                                value={filterType}
                                onChange={(e) => setFilterType(e.target.value)}
                            >
                                <option value="MULTIPLE_CHOICE">Opción Múltiple</option>
                                <option value="TRUE_FALSE">Verdadero/Falso</option>
                                <option value="OPEN">Abierta</option>
                            </ChakraSelect>
                        </HStack>
                    </VStack>

                    <Table variant="simple">
                        <Thead bg="gray.50">
                            <Tr>
                                <Th width="50px">
                                    <Checkbox
                                        isChecked={filteredQuestions.length > 0 && selectedIds.size === filteredQuestions.length}
                                        isIndeterminate={selectedIds.size > 0 && selectedIds.size < filteredQuestions.length}
                                        onChange={(e) => {
                                            if (e.target.checked) {
                                                const allIds = new Set(filteredQuestions.map(q => q.id_question));
                                                setSelectedIds(allIds);
                                            } else {
                                                setSelectedIds(new Set());
                                            }
                                        }}
                                        colorScheme="brand"
                                    />
                                </Th>
                                <Th width="50px">#</Th>
                                <Th>Pregunta</Th>
                                {isAdmin && <Th>Profesor</Th>}
                                <Th>Categoría</Th>
                                <Th>Tipo</Th>
                                <Th>Estado</Th>
                                <Th>Acciones</Th>
                            </Tr>
                        </Thead>
                        <Tbody>
                            {filteredQuestions.map((q, index) => (
                                <Tr key={q.id_question}>
                                    <Td>
                                        <Checkbox
                                            isChecked={selectedIds.has(q.id_question)}
                                            onChange={() => handleSelectOne(q.id_question)}
                                            colorScheme="brand"
                                        />
                                    </Td>
                                    <Td>{index + 1}</Td>
                                    <Td maxW="400px">
                                        <Text noOfLines={2} fontWeight="medium">
                                            {q.content?.question}
                                        </Text>
                                    </Td>
                                    {isAdmin && (
                                        <Td>
                                            {q.professor_name && q.professor_name.trim() ? (
                                                <>
                                                    <Text fontSize="sm" fontWeight="bold">
                                                        {q.professor_name.trim()}
                                                    </Text>
                                                    <Text fontSize="xs" color="gray.500">ID: {q.professor_id}</Text>
                                                </>
                                            ) : (
                                                <Text fontSize="sm" color="gray.500">
                                                    {q.professor_id ? `Sin nombre (ID: ${q.professor_id})` : 'Sin asignar'}
                                                </Text>
                                            )}
                                        </Td>
                                    )}
                                    <Td>
                                        <Badge colorScheme="purple">{q.category || 'Sin Categoría'}</Badge>
                                    </Td>
                                    <Td>
                                        <Text fontSize="sm">
                                            {q.type === 'MULTIPLE_CHOICE' ? 'Opción Múltiple' :
                                                q.type === 'TRUE_FALSE' ? 'Verdadero / Falso' :
                                                    q.type === 'OPEN' ? 'Abierta' :
                                                        q.type}
                                        </Text>
                                    </Td>
                                    <Td>
                                        <Badge colorScheme={q.is_active ? 'green' : 'red'}>
                                            {q.is_active ? 'ACTIVA' : 'INACTIVA'}
                                        </Badge>
                                    </Td>
                                    <Td>
                                        <HStack spacing={2}>
                                            {isAdmin && (
                                                <>
                                                    <IconButton
                                                        icon={<FiEye />}
                                                        size="sm"
                                                        onClick={() => handleView(q)}
                                                        aria-label="Ver"
                                                        colorScheme="teal"
                                                        variant="ghost"
                                                    />
                                                    <IconButton
                                                        icon={<FiEdit2 />}
                                                        size="sm"
                                                        onClick={() => handleEdit(q)}
                                                        aria-label="Editar"
                                                        colorScheme="blue"
                                                        variant="ghost"
                                                    />
                                                    <IconButton
                                                        icon={<FiPower />}
                                                        size="sm"
                                                        onClick={() => handleToggleStatus(q)}
                                                        aria-label={q.is_active ? 'Desactivar' : 'Activar'}
                                                        colorScheme={q.is_active ? 'red' : 'green'}
                                                        variant="ghost"
                                                    />
                                                    <IconButton
                                                        icon={<FiTrash2 />}
                                                        size="sm"
                                                        onClick={() => handleDelete(q)}
                                                        aria-label="Eliminar"
                                                        colorScheme="red"
                                                        variant="ghost"
                                                    />
                                                </>
                                            )}
                                            {!isAdmin && (
                                                <>
                                                    <IconButton
                                                        icon={<FiEye />}
                                                        size="sm"
                                                        onClick={() => handleView(q)}
                                                        aria-label="Ver"
                                                        colorScheme="teal"
                                                        variant="ghost"
                                                    />
                                                    <IconButton
                                                        icon={<FiEdit2 />}
                                                        size="sm"
                                                        onClick={() => handleEdit(q)}
                                                        aria-label="Editar"
                                                    />
                                                    <IconButton
                                                        icon={<FiTrash2 />}
                                                        size="sm"
                                                        onClick={() => handleDelete(q)}
                                                        aria-label="Eliminar"
                                                        colorScheme="red"
                                                        variant="ghost"
                                                    />
                                                </>
                                            )}
                                        </HStack>
                                    </Td>
                                </Tr>
                            ))}
                        </Tbody>
                    </Table>
                </Box>
            )}

            <Modal isOpen={isOpen} onClose={onClose} size="xl">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>
                        {isReadOnly ? 'Detalles de la Pregunta' : (currentQuestion?.id_question ? 'Editar Pregunta' : 'Nueva Pregunta')}
                    </ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <VStack spacing={4}>
                            <FormControl isRequired>
                                <FormLabel>Categoría</FormLabel>
                                <HStack>
                                    {!isNewCategory ? (
                                        <Select
                                            value={currentQuestion?.category || ''}
                                            onChange={(e) => setCurrentQuestion({ ...currentQuestion, category: e.target.value })}
                                            isDisabled={isReadOnly}
                                        >
                                            {categories.map(cat => (
                                                <option key={cat} value={cat}>{cat}</option>
                                            ))}
                                            <option value="">-- Sin Categoría --</option>
                                        </Select>
                                    ) : (
                                        <Input
                                            placeholder="Escribe la nueva categoría..."
                                            value={currentQuestion?.category || ''}
                                            onChange={(e) => setCurrentQuestion({ ...currentQuestion, category: e.target.value })}
                                            autoFocus
                                            isReadOnly={isReadOnly}
                                        />
                                    )}
                                    {!isReadOnly && (
                                        <Button
                                            size="sm"
                                            variant="outline"
                                            onClick={() => {
                                                const nextState = !isNewCategory;
                                                setIsNewCategory(nextState);
                                                // Si pasamos a "Nueva", limpiamos el campo para escribir
                                                // Si volvemos a "Lista", ponemos la primera opción por defecto
                                                setCurrentQuestion({
                                                    ...currentQuestion,
                                                    category: nextState ? '' : (categories[0] || '')
                                                });
                                            }}
                                            colorScheme="blue"
                                        >
                                            {isNewCategory ? "Ver Lista" : "Nueva +"}
                                        </Button>
                                    )}
                                </HStack>
                                <Text fontSize="xs" color="gray.500" mt={1}>
                                    {!isNewCategory ? (
                                        <>Si no existe la categoría que necesitas, presiona <strong>"Nueva +"</strong> para dar de alta una nueva categoría.</>
                                    ) : (
                                        <>Escribe el nombre de la nueva categoría y presiona <strong>"Ver Lista"</strong> para volver a la lista de categorías existentes.</>
                                    )}
                                </Text>
                            </FormControl>

                            {isAdmin && (
                                <FormControl isRequired={!currentQuestion?.id_question}>
                                    <FormLabel>Asignar a Profesor</FormLabel>
                                    {professors.length > 0 ? (
                                        <>
                                            <Select
                                                value={currentQuestion?.professor_id ? String(currentQuestion.professor_id) : ''}
                                                onChange={(e) => {
                                                    const newProfessorId = e.target.value ? parseInt(e.target.value) : null;
                                                    setCurrentQuestion({ ...currentQuestion, professor_id: newProfessorId });
                                                }}
                                                placeholder="Selecciona un profesor..."
                                                isDisabled={isReadOnly}
                                            >
                                                {professors.map(prof => (
                                                    <option key={prof.id || prof.user_id} value={prof.id || prof.user_id}>
                                                        {prof.name || prof.email} {prof.specialty ? `- ${prof.specialty}` : ''}
                                                    </option>
                                                ))}
                                            </Select>
                                            <Text fontSize="xs" color="gray.500" mt={1}>
                                                {currentQuestion?.id_question
                                                    ? 'Puedes cambiar el profesor asignado a esta pregunta. Solo se muestran profesores activos de tu institución.'
                                                    : 'Selecciona el profesor al que se asignará esta pregunta. Solo se muestran profesores activos de tu institución.'
                                                }
                                            </Text>
                                        </>
                                    ) : (
                                        <>
                                            <Select isDisabled placeholder="No hay profesores disponibles">
                                                <option value="">No hay profesores disponibles</option>
                                            </Select>
                                            <Text fontSize="xs" color="orange.500" mt={1}>
                                                No hay profesores activos en tu institución. Debes crear profesores antes de asignar preguntas.
                                            </Text>
                                        </>
                                    )}
                                </FormControl>
                            )}

                            <FormControl isRequired>
                                <FormLabel>Enunciado del Reactivo</FormLabel>
                                <Textarea
                                    placeholder="Escribe el enunciado de la pregunta..."
                                    value={currentQuestion?.content?.question || ''}
                                    onChange={(e) => setCurrentQuestion({
                                        ...currentQuestion,
                                        content: { ...currentQuestion.content, question: e.target.value }
                                    })}
                                    isReadOnly={isReadOnly}
                                />
                            </FormControl>

                            <FormControl isRequired>
                                <FormLabel>Tipo de Pregunta</FormLabel>
                                <Select
                                    value={currentQuestion?.type || ''}
                                    onChange={(e) => {
                                        const newType = e.target.value;
                                        let newContent = { ...currentQuestion.content };

                                        if (newType === 'MULTIPLE_CHOICE' && (!newContent.options || newContent.options.length < 5)) {
                                            newContent.options = ['', '', '', '', ''];
                                        } else if (newType === 'TRUE_FALSE') {
                                            newContent.options = ['Verdadero', 'Falso'];
                                        } else {
                                            newContent.options = [];
                                        }

                                        setCurrentQuestion({
                                            ...currentQuestion,
                                            type: newType,
                                            content: newContent
                                        });
                                    }}
                                    isDisabled={isReadOnly}
                                >
                                    <option value="MULTIPLE_CHOICE">Opción Múltiple</option>
                                    <option value="TRUE_FALSE">Verdadero / Falso</option>
                                    <option value="OPEN">Abierta</option>
                                </Select>
                            </FormControl>

                            {currentQuestion?.type === 'MULTIPLE_CHOICE' && (
                                <Box w="100%" p={4} bg="gray.50" borderRadius="md">
                                    <Text fontWeight="bold" mb={3}>Opciones de Respuesta:</Text>
                                    {currentQuestion.content.options.map((opt, idx) => (
                                        <FormControl key={idx} mb={2} isRequired>
                                            <HStack>
                                                <Text fontWeight="bold">{idx + 1}.</Text>
                                                <Input
                                                    bg="white"
                                                    value={opt}
                                                    onChange={(e) => {
                                                        const newOpts = [...currentQuestion.content.options];
                                                        newOpts[idx] = e.target.value;
                                                        setCurrentQuestion({
                                                            ...currentQuestion,
                                                            content: { ...currentQuestion.content, options: newOpts }
                                                        });
                                                    }}
                                                    isReadOnly={isReadOnly}
                                                />
                                            </HStack>
                                        </FormControl>
                                    ))}
                                </Box>
                            )}

                            <FormControl isRequired>
                                <FormLabel>Respuesta Correcta</FormLabel>
                                {currentQuestion?.type === 'TRUE_FALSE' ? (
                                    <Select
                                        value={currentQuestion?.correct_answer || ''}
                                        onChange={(e) => setCurrentQuestion({ ...currentQuestion, correct_answer: e.target.value })}
                                        placeholder="Selecciona la respuesta correcta..."
                                        isDisabled={isReadOnly}
                                    >
                                        <option value="Verdadero">Verdadero</option>
                                        <option value="Falso">Falso</option>
                                    </Select>
                                ) : (
                                    <>
                                        <Input
                                            placeholder={
                                                currentQuestion?.type === 'MULTIPLE_CHOICE'
                                                    ? "Escribe el texto exacto de la opción correcta..."
                                                    : "Indica la respuesta correcta o ejemplo de respuesta..."
                                            }
                                            value={currentQuestion?.correct_answer || ''}
                                            onChange={(e) => setCurrentQuestion({ ...currentQuestion, correct_answer: e.target.value })}
                                        />
                                        <Text fontSize="xs" color="gray.500" mt={1}>
                                            {currentQuestion?.type === 'MULTIPLE_CHOICE' && (
                                                <>El texto debe coincidir <strong>exactamente</strong> con una de las opciones de arriba (incluyendo mayúsculas, minúsculas y espacios).</>
                                            )}
                                            {currentQuestion?.type === 'OPEN' && (
                                                <>Para preguntas abiertas, puedes indicar un ejemplo de respuesta esperada o criterios de evaluación.</>
                                            )}
                                        </Text>
                                    </>
                                )}
                            </FormControl>
                        </VStack>
                    </ModalBody>
                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            {isReadOnly ? 'Cerrar' : 'Cancelar'}
                        </Button>
                        {!isReadOnly && (
                            <Button colorScheme="brand" onClick={handleSave}>Guardar Pregunta</Button>
                        )}
                    </ModalFooter>
                </ModalContent>
            </Modal>

            {/* Modal de Importación CSV */}
            <Modal isOpen={isImportModalOpen} onClose={isLoading ? undefined : onImportModalClose} size="xl" closeOnOverlayClick={!isLoading}>
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Importar Preguntas desde CSV</ModalHeader>
                    {!isLoading && <ModalCloseButton />}
                    <ModalBody pb={6}>
                        {isLoading ? (
                            <Center py={10} flexDirection="column">
                                <Spinner size="xl" color="brand.500" thickness="4px" />
                                <Text mt={4} fontWeight="bold" fontSize="xl" color="brand.600">
                                    Kemarketing
                                </Text>
                                <Text mt={2} color="gray.500" fontSize="sm">
                                    Procesando preguntas...
                                </Text>
                            </Center>
                        ) : (
                            <VStack spacing={6} align="stretch">
                                <Box>
                                    <Text fontSize="sm" color="gray.600" mb={4}>
                                        Selecciona el tipo de pregunta que deseas importar y descarga el template correspondiente para llenarlo con tus datos.
                                    </Text>

                                    <FormControl>
                                        <FormLabel>Tipo de Pregunta</FormLabel>
                                        <Select
                                            value={importType}
                                            onChange={(e) => setImportType(e.target.value)}
                                        >
                                            <option value="MULTIPLE_CHOICE">Opción Múltiple</option>
                                            <option value="TRUE_FALSE">Verdadero / Falso</option>
                                            <option value="OPEN">Abierta</option>
                                        </Select>
                                    </FormControl>

                                    <Button
                                        leftIcon={<FiDownload />}
                                        colorScheme="blue"
                                        variant="outline"
                                        size="sm"
                                        mt={2}
                                        onClick={() => downloadTemplate(importType)}
                                    >
                                        Descargar Template CSV
                                    </Button>
                                </Box>

                                <Divider />

                                <FormControl>
                                    <FormLabel>Seleccionar Archivo CSV</FormLabel>
                                    <Box
                                        border="2px dashed"
                                        borderColor={isDragging ? "brand.500" : (importFile ? "green.300" : "gray.300")}
                                        bg={isDragging ? "blue.50" : (importFile ? "green.50" : "gray.50")}
                                        borderRadius="md"
                                        p={6}
                                        textAlign="center"
                                        cursor="pointer"
                                        transition="all 0.2s"
                                        _hover={{ borderColor: "brand.400", bg: "gray.100" }}
                                        onDragOver={(e) => {
                                            e.preventDefault();
                                            setIsDragging(true);
                                        }}
                                        onDragLeave={() => setIsDragging(false)}
                                        onDrop={(e) => {
                                            e.preventDefault();
                                            setIsDragging(false);
                                            const file = e.dataTransfer.files[0];
                                            if (file && file.name.endsWith('.csv')) {
                                                setImportFile(file);
                                                setImportErrors([]);
                                                setImportSuccess(0);
                                                toast({
                                                    title: 'Archivo seleccionado',
                                                    description: file.name,
                                                    status: 'info',
                                                    duration: 2000
                                                });
                                            } else {
                                                toast({
                                                    title: 'Formato incorrecto',
                                                    description: 'Por favor arrastra un archivo CSV válido',
                                                    status: 'warning'
                                                });
                                            }
                                        }}
                                        onClick={() => document.getElementById('fileInput').click()}
                                    >
                                        <input
                                            id="fileInput"
                                            type="file"
                                            accept=".csv"
                                            style={{ display: 'none' }}
                                            onChange={(e) => {
                                                if (e.target.files[0]) {
                                                    setImportFile(e.target.files[0]);
                                                    setImportErrors([]);
                                                    setImportSuccess(0);
                                                }
                                            }}
                                        />
                                        <Icon as={importFile ? FiDatabase : FiUpload} boxSize={8} color={importFile ? "green.500" : "gray.400"} mb={2} />
                                        <Text fontWeight="bold" color={importFile ? "green.700" : "gray.600"}>
                                            {importFile ? importFile.name : "Arrastra tu archivo CSV aquí"}
                                        </Text>
                                        <Text fontSize="xs" color="gray.500" mt={1}>
                                            {importFile ? "Haz clic o arrastra otro archivo para cambiar" : "o haz clic para buscar en tu equipo"}
                                        </Text>
                                    </Box>
                                </FormControl>

                                {importErrors.length > 0 && (
                                    <Box
                                        bg="red.50"
                                        border="2px solid"
                                        borderColor="red.300"
                                        borderRadius="md"
                                        p={4}
                                        maxH="400px"
                                        overflowY="auto"
                                    >
                                        <HStack justify="space-between" mb={3}>
                                            <Text fontWeight="bold" color="red.700" fontSize="md">
                                                📋 Bitácora de Incidencias ({importErrors.length} error{importErrors.length !== 1 ? 'es' : ''})
                                            </Text>
                                            <Badge colorScheme="red" fontSize="sm">
                                                {importErrors.length} {importErrors.length === 1 ? 'incidencia' : 'incidencias'}
                                            </Badge>
                                        </HStack>
                                        <Text fontSize="xs" color="red.600" mb={3} fontStyle="italic">
                                            Corrige los siguientes errores antes de intentar importar nuevamente:
                                        </Text>
                                        <VStack align="stretch" spacing={3}>
                                            {importErrors.map((error, idx) => (
                                                <Box
                                                    key={idx}
                                                    bg="white"
                                                    p={3}
                                                    borderRadius="md"
                                                    border="1px solid"
                                                    borderColor="red.200"
                                                    fontSize="sm"
                                                >
                                                    <HStack mb={2}>
                                                        <Badge colorScheme="red" variant="solid" fontSize="xs">
                                                            Fila {error.row}
                                                        </Badge>
                                                        <Text fontWeight="semibold" color="red.700" flex={1} noOfLines={1}>
                                                            {error.question}
                                                        </Text>
                                                    </HStack>
                                                    <VStack align="start" spacing={1} ml={2} pl={2} borderLeft="2px solid" borderColor="red.300">
                                                        {error.errors.map((err, errIdx) => (
                                                            <HStack key={errIdx} spacing={2} align="start">
                                                                <Text color="red.600" fontSize="xs" fontWeight="bold">✗</Text>
                                                                <Text color="red.600" fontSize="xs" flex={1}>
                                                                    {err}
                                                                </Text>
                                                            </HStack>
                                                        ))}
                                                    </VStack>
                                                </Box>
                                            ))}
                                        </VStack>
                                        <Box mt={4} p={3} bg="yellow.50" borderRadius="md" border="1px solid" borderColor="yellow.200">
                                            <Text fontSize="xs" color="yellow.800" fontWeight="semibold">
                                                ⚠️ Importante: No se insertó ninguna pregunta. Corrige todos los errores y vuelve a intentar.
                                            </Text>
                                        </Box>
                                    </Box>
                                )}

                                {importSuccess > 0 && (
                                    <Box
                                        bg="green.50"
                                        border="1px solid"
                                        borderColor="green.200"
                                        borderRadius="md"
                                        p={4}
                                    >
                                        <Text fontWeight="bold" color="green.700">
                                            ✓ Se importaron {importSuccess} pregunta(s) exitosamente
                                        </Text>
                                    </Box>
                                )}
                            </VStack>
                        )}
                    </ModalBody>
                    <ModalFooter>
                        <Button
                            variant="ghost"
                            mr={3}
                            onClick={() => {
                                onImportModalClose();
                                setImportFile(null);
                                setImportErrors([]);
                                setImportSuccess(0);
                            }}
                            isDisabled={isLoading}
                        >
                            {importErrors.length > 0 ? 'Cerrar' : 'Cancelar'}
                        </Button>
                        <Button
                            colorScheme="brand"
                            onClick={handleImport}
                            leftIcon={!isLoading && <FiUpload />}
                            isDisabled={isLoading || !importFile || importErrors.length > 0}
                            isLoading={isLoading}
                            loadingText="Importando..."
                        >
                            {importErrors.length > 0 ? 'Corrige los errores primero' : 'Importar Preguntas'}
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionPreguntas;

