
import { useState, useEffect, useRef } from 'react';
import {
    Box,
    Heading,
    SimpleGrid,
    Card,
    CardBody,
    Text,
    Badge,
    VStack,
    HStack,
    Icon,
    Avatar,
    useToast,
    Flex,
    Spinner,
    Breadcrumb,
    BreadcrumbItem,
    BreadcrumbLink,
    Center
} from '@chakra-ui/react';
import { FiAlertCircle, FiActivity, FiChevronRight, FiUser } from 'react-icons/fi';
import { io } from 'socket.io-client';
import { useParams, Link } from 'react-router-dom';
import { BASE_URL, getExamById, getStudents } from '../../services/api';

const LiveMonitor = () => {
    const { examId } = useParams();
    const [exam, setExam] = useState(null);
    const [alerts, setAlerts] = useState([]);
    const [activeStudents, setActiveStudents] = useState({});
    const [studentsCatalog, setStudentsCatalog] = useState([]);
    const [loading, setLoading] = useState(true);
    const toast = useToast();
    const studentsCatalogRef = useRef([]);

    // Mantener referencia actualizada
    useEffect(() => {
        studentsCatalogRef.current = studentsCatalog;
    }, [studentsCatalog]);

    const getStudentName = (id) => {
        const student = studentsCatalog.find(s => Number(s.id || s.user_id) === Number(id));
        return student ? student.name : `ID: ${id}`;
    };

    useEffect(() => {
        const fetchInitialData = async () => {
            try {
                setLoading(true);
                const [examData, studentsData] = await Promise.all([
                    getExamById(examId),
                    getStudents()
                ]);
                setExam(examData);
                setStudentsCatalog(studentsData || []);
            } catch (error) {
                console.error('Error loading monitor data:', error);
                toast({
                    title: 'Error al cargar datos',
                    description: error.message || 'No se pudieron cargar los datos del examen',
                    status: 'error',
                    duration: 3000,
                });
            } finally {
                setLoading(false);
            }
        };

        if (examId) {
            fetchInitialData();
            const socket = io(BASE_URL);
            
            socket.on('connect', () => {
                console.log('Socket conectado al monitor');
                socket.emit('join-exam', examId);
            });

            socket.on('connect_error', (error) => {
                console.error('Error de conexión socket:', error);
                toast({
                    title: 'Error de conexión',
                    description: 'No se pudo conectar al servidor de monitoreo',
                    status: 'error',
                    duration: 3000,
                });
            });

            socket.on('monitor-alert', (data) => {
                // data: { examId, studentId, type: 'FOCUS_LOST' | 'TAB_CHANGE' | 'CONECTADO' | 'HEARTBEAT' | 'EXAMEN_INICIADO' | 'EXAMEN_FINALIZADO', timestamp }
                console.log('Alerta recibida:', data);
                
                const isAlert = data.type === 'FOCUS_LOST' || data.type === 'TAB_CHANGE';
                const isStartEvent = data.type === 'EXAMEN_INICIADO';
                const isFinishEvent = data.type === 'EXAMEN_FINALIZADO';
                
                // Obtener nombre del estudiante para el toast usando la referencia actualizada
                const student = studentsCatalogRef.current.find(s => Number(s.id || s.user_id) === Number(data.studentId));
                const studentName = student ? student.name : `ID: ${data.studentId}`;
                
                // Notificar inicio de examen
                if (isStartEvent) {
                    toast({
                        title: `📝 Examen Iniciado`,
                        description: `El estudiante ${studentName} ha comenzado a aplicar el examen`,
                        status: 'info',
                        duration: 5000,
                        isClosable: true,
                        position: 'top-right'
                    });
                }
                
                // Notificar finalización de examen
                if (isFinishEvent) {
                    const score = data.score !== undefined ? ` con calificación: ${Number(parseFloat(data.score).toFixed(2))} / 10` : '';
                    toast({
                        title: `✅ Examen Finalizado`,
                        description: `El estudiante ${studentName} ha completado el examen${score}`,
                        status: 'success',
                        duration: 5000,
                        isClosable: true,
                        position: 'top-right'
                    });
                }
                
                if (isAlert) {
                    setAlerts(prev => [data, ...prev].slice(0, 50));
                    
                    toast({
                        title: `¡Incidencia Detectada!`,
                        description: `Estudiante: ${studentName} - ${data.type === 'TAB_CHANGE' ? 'Cambio de pestaña' : 'Pérdida de foco'}`,
                        status: 'warning',
                        duration: 5000,
                        isClosable: true,
                        position: 'top-right'
                    });
                }

                // Update student status in real-time list (para todos los eventos, no solo alertas)
                setActiveStudents(prev => {
                    const currentInfo = prev[data.studentId] || {};
                    let lastEvent = currentInfo.lastEvent || 'Estable';
                    
                    if (isStartEvent) {
                        lastEvent = 'Iniciado';
                    } else if (isFinishEvent) {
                        lastEvent = 'Finalizado';
                    } else if (isAlert) {
                        lastEvent = data.type;
                    } else if (data.type === 'CONECTADO') {
                        lastEvent = 'Conectado';
                    }
                    
                    return {
                        ...prev,
                        [data.studentId]: {
                            ...currentInfo,
                            id: data.studentId,
                            lastEvent: lastEvent,
                            lastTimestamp: data.timestamp || new Date(),
                            isOnline: !isFinishEvent, // Marcar como offline si finalizó
                            score: isFinishEvent && data.score !== undefined ? data.score : currentInfo.score
                        }
                    };
                });
            });

            return () => {
                console.log('Desconectando socket del monitor');
                socket.disconnect();
            };
        }
    }, [examId, toast]);

    if (loading) {
        return (
            <Center h="60vh">
                <Spinner size="xl" color="brand.500" thickness="4px" />
            </Center>
        );
    }

    return (
        <Box p={4}>
            <Breadcrumb spacing="8px" separator={<FiChevronRight color="gray.500" />} mb={4}>
                <BreadcrumbItem>
                    <BreadcrumbLink as={Link} to="/professor/examenes">Exámenes</BreadcrumbLink>
                </BreadcrumbItem>
                <BreadcrumbItem isCurrentPage>
                    <BreadcrumbLink color="gray.500">Monitor en Vivo</BreadcrumbLink>
                </BreadcrumbItem>
            </Breadcrumb>

            <HStack mb={6} justify="space-between" wrap="wrap" gap={4}>
                <VStack align="start" spacing={0}>
                    <Heading size="lg">Monitor: {exam?.title || 'Examen'}</Heading>
                    <Text color="gray.500">Seguimiento de actividad en tiempo real</Text>
                </VStack>
                <Badge colorScheme="green" p={2} borderRadius="md" variant="subtle" border="1px solid" borderColor="green.200">
                    <HStack><Icon as={FiActivity} /> <Text>Transmisión Activa</Text></HStack>
                </Badge>
            </HStack>

            <SimpleGrid columns={{ base: 1, lg: 3 }} spacing={6}>
                {/* Active Students List */}
                <Box gridColumn={{ lg: "span 2" }}>
                    <Heading size="md" mb={4} display="flex" alignItems="center">
                        <Icon as={FiUser} mr={2} /> Alumnos en Aplicación
                    </Heading>
                    <SimpleGrid columns={{ base: 1, md: 2 }} spacing={4}>
                        {Object.entries(activeStudents).map(([id, info]) => {
                            const getBorderColor = () => {
                                if (info.lastEvent === 'Finalizado') return "gray.400";
                                if (info.lastEvent === 'Iniciado') return "blue.400";
                                if (info.lastEvent === 'TAB_CHANGE' || info.lastEvent === 'FOCUS_LOST') return "orange.400";
                                return "green.400";
                            };
                            
                            const getBadgeColor = () => {
                                if (info.lastEvent === 'Finalizado') return "gray";
                                if (info.lastEvent === 'Iniciado') return "blue";
                                if (info.lastEvent === 'TAB_CHANGE' || info.lastEvent === 'FOCUS_LOST') return "orange";
                                return "green";
                            };
                            
                            const getBadgeText = () => {
                                if (info.lastEvent === 'Finalizado') return 'FINALIZADO';
                                if (info.lastEvent === 'Iniciado') return 'EN PROGRESO';
                                if (info.lastEvent === 'TAB_CHANGE') return 'FUERA DE PESTAÑA';
                                if (info.lastEvent === 'FOCUS_LOST') return 'FOCO PERDIDO';
                                return 'ESTABLE';
                            };
                            
                            return (
                                <Card key={id} variant="outline" borderLeft="4px solid" borderColor={getBorderColor()} boxShadow="sm">
                                    <CardBody py={4}>
                                        <HStack spacing={4}>
                                            <Avatar size="sm" name={getStudentName(id)} bg={info.lastEvent === 'Finalizado' ? "gray.100" : info.lastEvent === 'Iniciado' ? "blue.100" : info.lastEvent !== 'Estable' ? "orange.100" : "green.100"} color={info.lastEvent === 'Finalizado' ? "gray.700" : info.lastEvent === 'Iniciado' ? "blue.700" : info.lastEvent !== 'Estable' ? "orange.700" : "green.700"} />
                                            <VStack align="start" spacing={0} flex="1">
                                                <Text fontWeight="bold" fontSize="sm">{getStudentName(id)}</Text>
                                                <Text fontSize="xs" color="gray.500">Actividad: {new Date(info.lastTimestamp).toLocaleTimeString()}</Text>
                                                {info.score !== undefined && (
                                                    <Text fontSize="xs" color="gray.600" fontWeight="semibold" mt={1}>
                                                        Calificación: {Number(parseFloat(info.score).toFixed(2))} / 10
                                                    </Text>
                                                )}
                                            </VStack>
                                            <Badge colorScheme={getBadgeColor()} variant="solid" size="sm">
                                                {getBadgeText()}
                                            </Badge>
                                        </HStack>
                                    </CardBody>
                                </Card>
                            );
                        })}
                        {Object.keys(activeStudents).length === 0 && (
                            <Center p={10} bg="gray.50" borderRadius="lg" border="1px dashed" borderColor="gray.300" w="full" gridColumn="span 2">
                                <VStack spacing={2}>
                                    <Icon as={FiActivity} boxSize={8} color="gray.300" />
                                    <Text color="gray.500" fontSize="sm">Esperando inicio de examen por los alumnos...</Text>
                                </VStack>
                            </Center>
                        )}
                    </SimpleGrid>
                </Box>

                {/* Live Alerts Log */}
                <Box>
                    <Heading size="md" mb={4} display="flex" alignItems="center">
                        <Icon as={FiAlertCircle} mr={2} /> Registro de Incidencias
                    </Heading>
                    <VStack spacing={3} align="stretch" maxH="600px" overflowY="auto">
                        {alerts.length > 0 ? (
                            alerts.map((alert, index) => (
                                <Box key={index} p={3} bg="orange.50" border="1px solid" borderColor="orange.200" borderRadius="md">
                                    <HStack align="start">
                                        <Icon as={FiAlertCircle} color="orange.500" mt={1} />
                                        <VStack align="start" spacing={0}>
                                            <Text fontSize="sm" fontWeight="bold" color="orange.800">
                                                {alert.type === 'TAB_CHANGE' ? 'Cambio de Pestaña' : 'Pérdida de Foco'}
                                            </Text>
                                            <Text fontSize="xs" color="orange.700">Alumno: {getStudentName(alert.studentId)}</Text>
                                            <Text fontSize="xx-small" color="orange.600" mt={1}>
                                                {new Date(alert.timestamp).toLocaleTimeString()}
                                            </Text>
                                        </VStack>
                                    </HStack>
                                </Box>
                            ))
                        ) : (
                            <Center py={10} color="gray.400" bg="gray.50" borderRadius="lg" border="1px solid" borderColor="gray.100">
                                <Text fontSize="sm">No hay incidencias registradas.</Text>
                            </Center>
                        )}
                    </VStack>
                </Box>
            </SimpleGrid>
        </Box>
    );
};

export default LiveMonitor;
