
import { useState, useEffect } from 'react';
import {
    Box,
    SimpleGrid,
    Stat,
    StatLabel,
    StatNumber,
    StatHelpText,
    Card,
    CardBody,
    Heading,
    Text,
    Icon,
    Flex,
    Badge,
    VStack,
    HStack,
    useToast,
    Spinner,
    Button
} from '@chakra-ui/react';
import { FiBookOpen, FiClock, FiUsers, FiArrowRight } from 'react-icons/fi';
import { useNavigate } from 'react-router-dom';
import { getExams, getStudents, getExamAssignments } from '../../services/api'; 

const ProfessorDashboard = () => {
    const [exams, setExams] = useState([]);
    const [students, setStudents] = useState([]);
    const [upcomingAssignments, setUpcomingAssignments] = useState([]);
    const [loading, setLoading] = useState(true);
    const toast = useToast();
    const navigate = useNavigate();

    useEffect(() => {
        const fetchData = async () => {
            try {
                setLoading(true);
                const user = JSON.parse(sessionStorage.getItem('user'));
                const instId = Number(user?.institution_id || user?.company_id);
                const userId = Number(user?.id || user?.user_id);
                
                if (user) {
                    const [allExams, allStudents] = await Promise.all([
                        getExams(),
                        getStudents()
                    ]);

                    // Filtrar exámenes creados por el profesor
                    const myExams = allExams.filter(e => Number(e.professor_id) === userId);
                    setExams(myExams);

                    // Filtrar estudiantes de su institución
                    const myStudents = allStudents.filter(s => Number(s.institution_id || s.company_id) === instId);
                    setStudents(myStudents);

                    // Obtener todas las asignaciones de sus exámenes
                    let allAssignments = [];
                    for (const exam of myExams) {
                        try {
                            const assignments = await getExamAssignments(exam.id_exam || exam.id);
                            const enriched = (assignments || []).map(a => ({
                                ...a,
                                exam_title: exam.title
                            }));
                            allAssignments = [...allAssignments, ...enriched];
                        } catch (e) {
                            console.error(`Error fetching assignments for exam ${exam.id_exam}:`, e);
                        }
                    }
                    
                    // Ordenar por fecha y filtrar solo las pendientes o activas
                    const upcoming = allAssignments
                        .filter(a => a.status === 'PENDING' || a.status === 'ACTIVE')
                        .sort((a, b) => new Date(a.scheduled_at || 0) - new Date(b.scheduled_at || 0));
                    
                    setUpcomingAssignments(upcoming);
                }
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar dashboard',
                    status: 'error',
                    duration: 3000,
                });
            } finally {
                setLoading(false);
            }
        };
        fetchData();
    }, []);

    if (loading) {
        return (
            <Flex justify="center" align="center" h="60vh">
                <Spinner size="xl" color="brand.500" thickness="4px" />
            </Flex>
        );
    }

    return (
        <Box>
            <Heading mb={6}>Panel del Profesor</Heading>

            {/* Stats Grid */}
            <SimpleGrid columns={{ base: 1, md: 3 }} spacing={6} mb={8}>
                <Card variant="outline" boxShadow="sm">
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="blue.100" borderRadius="md" mr={3}>
                                    <Icon as={FiBookOpen} color="blue.500" boxSize={5} />
                                </Box>
                                <StatLabel fontSize="md" fontWeight="medium">Exámenes</StatLabel>
                            </Flex>
                            <StatNumber fontSize="3xl">{exams.length}</StatNumber>
                            <StatHelpText>Creados en tu banco</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card variant="outline" boxShadow="sm">
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="green.100" borderRadius="md" mr={3}>
                                    <Icon as={FiUsers} color="green.500" boxSize={5} />
                                </Box>
                                <StatLabel fontSize="md" fontWeight="medium">Estudiantes</StatLabel>
                            </Flex>
                            <StatNumber fontSize="3xl">{students.length}</StatNumber>
                            <StatHelpText>En tu institución</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card variant="outline" boxShadow="sm">
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="orange.100" borderRadius="md" mr={3}>
                                    <Icon as={FiClock} color="orange.500" boxSize={5} />
                                </Box>
                                <StatLabel fontSize="md" fontWeight="medium">Aplicaciones</StatLabel>
                            </Flex>
                            <StatNumber fontSize="3xl">{upcomingAssignments.length}</StatNumber>
                            <StatHelpText>Exámenes programados</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>
            </SimpleGrid>

            <Flex justify="space-between" align="center" mb={4}>
                <Heading size="md">Próximas Aplicaciones</Heading>
                <Button variant="ghost" size="sm" rightIcon={<FiArrowRight />} onClick={() => navigate('/professor/calendario')}>
                    Ver Agenda Completa
                </Button>
            </Flex>

            <VStack spacing={4} align="stretch">
                {upcomingAssignments.length > 0 ? (
                    upcomingAssignments.slice(0, 5).map(task => (
                        <Card key={task.id_assignment} variant="outline" _hover={{ borderColor: 'brand.500', bg: 'gray.50' }} transition="all 0.2s">
                            <CardBody py={4}>
                                <HStack justify="space-between">
                                    <VStack align="start" spacing={1}>
                                        <Text fontWeight="bold" fontSize="md">{task.exam_title}</Text>
                                        <HStack spacing={4}>
                                            <Text fontSize="xs" color="gray.600">
                                                <Icon as={FiUsers} mr={1} /> {task.student_name || task.student_email}
                                            </Text>
                                            <Text fontSize="xs" color="brand.600" fontWeight="medium">
                                                <Icon as={FiClock} mr={1} /> 
                                                {task.scheduled_at ? new Date(task.scheduled_at).toLocaleString() : 'Pendiente de fecha'}
                                            </Text>
                                        </HStack>
                                    </VStack>
                                    <Badge colorScheme={task.status === 'ACTIVE' ? 'blue' : 'orange'} variant="subtle" px={3} py={1} borderRadius="full">
                                        {task.status === 'ACTIVE' ? 'EN CURSO' : 'PROGRAMADO'}
                                    </Badge>
                                </HStack>
                            </CardBody>
                        </Card>
                    ))
                ) : (
                    <Card variant="outline" borderStyle="dashed" bg="gray.50">
                        <CardBody textAlign="center" py={12}>
                            <Icon as={FiClock} boxSize={12} color="gray.300" mb={4} />
                            <Heading size="sm" mb={2} color="gray.600">No hay aplicaciones programadas</Heading>
                            <Text color="gray.500" fontSize="sm" mb={4}>Asigna estudiantes a tus exámenes para verlos aquí.</Text>
                            <Button size="sm" colorScheme="brand" onClick={() => navigate('/professor/examenes')}>Ir a mis exámenes</Button>
                        </CardBody>
                    </Card>
                )}
            </VStack>
        </Box>
    );
};

export default ProfessorDashboard;
