
import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    Button,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    Alert,
    AlertIcon,
    VStack,
    HStack,
    Text,
    Divider,
    useToast,
    Spinner,
    Icon,
    Flex,
    Textarea,
    NumberInput,
    NumberInputField,
    NumberInputStepper,
    NumberIncrementStepper,
    NumberDecrementStepper,
    Switch,
    FormControl,
    FormLabel,
    Card,
    CardBody,
    Stat,
    StatLabel,
    StatNumber,
    SimpleGrid,
    Input,
    InputGroup,
    InputLeftElement,
    StatHelpText,
    StatArrow,
    Container
} from '@chakra-ui/react';
import { FiEye, FiCheck, FiX, FiSave, FiSearch, FiFileText } from 'react-icons/fi';
import { getExams, getExamAssignments, getExamResponses, updateResponseGrade, getInstitutionById } from '../../services/api';
import jsPDF from 'jspdf';
import autoTable from 'jspdf-autotable';

const RevisionExamenes = () => {
    const [exams, setExams] = useState([]);
    const [assignments, setAssignments] = useState([]);
    const [selectedExam, setSelectedExam] = useState(null);
    const [selectedAssignment, setSelectedAssignment] = useState(null);
    const [responses, setResponses] = useState([]);
    const [loading, setLoading] = useState(true);
    const [loadingDetails, setLoadingDetails] = useState(false);
    const [studentsCount, setStudentsCount] = useState({}); // { examId: count }
    const [examApplicationDates, setExamApplicationDates] = useState({}); // { examId: date }
    const [searchTerm, setSearchTerm] = useState('');
    const [generatingReport, setGeneratingReport] = useState(null); // ID del examen que se está generando

    const filteredExams = exams.filter(exam =>
        exam.title?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        exam.description?.toLowerCase().includes(searchTerm.toLowerCase())
    );

    // Cálculos para estadísticas
    const totalExams = exams.length;
    const totalStudentsEvaluated = Object.values(studentsCount).reduce((a, b) => a + b, 0);
    const activeExams = exams.filter(e => {
        const date = examApplicationDates[e.id_exam || e.id];
        if (!date) return false;
        const oneWeekAgo = new Date();
        oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
        return date > oneWeekAgo;
    }).length;

    const { isOpen: isExamListOpen, onOpen: onExamListOpen, onClose: onExamListClose } = useDisclosure();
    const { isOpen: isReviewOpen, onOpen: onReviewOpen, onClose: onReviewClose } = useDisclosure();
    const toast = useToast();

    useEffect(() => {
        fetchExams();
    }, []);

    const fetchExams = async () => {
        try {
            setLoading(true);
            const user = JSON.parse(sessionStorage.getItem('user'));
            const userId = Number(user?.id || user?.user_id);
            const data = await getExams();
            // Solo exámenes del profesor
            let professorExams = data.filter(e => Number(e.professor_id) === userId);

            // Obtener cantidad de estudiantes únicos y fecha de aplicación más reciente
            const countMap = {};
            const dateMap = {};
            await Promise.all(
                professorExams.map(async (exam) => {
                    try {
                        const examId = exam.id_exam || exam.id;
                        const assignments = await getExamAssignments(examId);
                        const completedAssignments = assignments.filter(
                            a => a.status === 'COMPLETED' || a.status === 'FINISHED'
                        );
                        // Contar estudiantes únicos, no intentos
                        const uniqueStudentIds = new Set(
                            completedAssignments.map(a => Number(a.student_id))
                        );
                        countMap[examId] = uniqueStudentIds.size;

                        // Obtener la fecha de aplicación más reciente
                        if (completedAssignments.length > 0) {
                            const dates = completedAssignments
                                .map(a => {
                                    // Usar completed_at si existe, sino scheduled_at, sino started_at
                                    return a.completed_at || a.scheduled_at || a.started_at;
                                })
                                .filter(Boolean)
                                .map(d => new Date(d).getTime());

                            if (dates.length > 0) {
                                const mostRecentDate = new Date(Math.max(...dates));
                                dateMap[examId] = mostRecentDate;
                            }
                        }
                    } catch (error) {
                        console.error(`Error al obtener asignaciones para examen ${exam.id_exam || exam.id}:`, error);
                        countMap[exam.id_exam || exam.id] = 0;
                    }
                })
            );
            setStudentsCount(countMap);
            setExamApplicationDates(dateMap);

            // Ordenar exámenes por fecha de aplicación (más cercano a la fecha actual primero)
            professorExams.sort((a, b) => {
                const examIdA = a.id_exam || a.id;
                const examIdB = b.id_exam || b.id;
                const dateA = dateMap[examIdA];
                const dateB = dateMap[examIdB];

                // Si ambos tienen fecha, ordenar por fecha (más reciente primero = más cercano a hoy)
                if (dateA && dateB) {
                    return dateB.getTime() - dateA.getTime();
                }
                // Si solo uno tiene fecha, el que tiene fecha va primero
                if (dateA && !dateB) return -1;
                if (!dateA && dateB) return 1;
                // Si ninguno tiene fecha, mantener orden original
                return 0;
            });

            setExams(professorExams);
        } catch (error) {
            console.error(error);
            toast({ title: 'Error al cargar exámenes', status: 'error' });
        } finally {
            setLoading(false);
        }
    };

    const handleViewAssignments = async (exam) => {
        setSelectedExam(exam);
        try {
            setLoadingDetails(true);
            const data = await getExamAssignments(exam.id_exam || exam.id);
            // Solo completados para revisión
            setAssignments(data.filter(a => a.status === 'COMPLETED' || a.status === 'FINISHED'));
            onExamListOpen();
        } catch (error) {
            toast({ title: 'Error al cargar alumnos', status: 'error' });
        } finally {
            setLoadingDetails(false);
        }
    };

    const handleReviewExam = async (assignment) => {
        setSelectedAssignment(assignment);
        try {
            setLoadingDetails(true);
            const data = await getExamResponses(assignment.id_assignment);
            setResponses(data);
            onReviewOpen();
        } catch (error) {
            toast({ title: 'Error al cargar respuestas', status: 'error' });
        } finally {
            setLoadingDetails(false);
        }
    };

    const handleUpdateGrade = async (responseId, isCorrect, points) => {
        try {
            const result = await updateResponseGrade(selectedAssignment.id_assignment, responseId, {
                is_correct: isCorrect,
                points_earned: points
            });

            // Actualizar localmente
            setResponses(prev => prev.map(r =>
                r.id_response === responseId ? { ...r, is_correct: isCorrect, points_earned: points } : r
            ));

            const newScore = Number(parseFloat(result.new_score).toFixed(2));

            // Actualizar el score en la lista de asignaciones
            setAssignments(prev => prev.map(a =>
                a.id_assignment === selectedAssignment.id_assignment ? { ...a, score: newScore } : a
            ));

            setSelectedAssignment(prev => ({ ...prev, score: newScore }));

            toast({ title: 'Calificación actualizada', status: 'success', duration: 2000 });
        } catch (error) {
            toast({ title: 'Error al actualizar', description: error.message, status: 'error' });
        }
    };

    const handleGenerateExamReport = async (exam) => {
        try {
            setGeneratingReport(exam.id_exam || exam.id);
            const examId = exam.id_exam || exam.id;

            // 1. Obtener todas las asignaciones
            const assignmentsAll = await getExamAssignments(examId);
            const completedAssignments = assignmentsAll.filter(a => a.status === 'COMPLETED' || a.status === 'FINISHED');

            if (completedAssignments.length === 0) {
                toast({ title: 'No hay entregas para generar reporte', status: 'warning' });
                setGeneratingReport(null);
                return;
            }

            // 2. Obtener datos de la Institución (Reutilizando lógica)
            const user = JSON.parse(sessionStorage.getItem('user'));
            let institutionName = "KeQuiz Institution";
            let logoUrl = "/logo.jpeg";

            if (user?.institution_id) {
                try {
                    const inst = await getInstitutionById(user.institution_id);
                    if (inst) {
                        institutionName = inst.name || institutionName;
                        if (inst.logo_url) {
                            logoUrl = inst.logo_url;
                        }
                    }
                } catch (e) {
                    console.warn("No se pudo obtener la institución", e);
                }
            }

            const loadImage = (url) => {
                return new Promise((resolve) => {
                    const img = new Image();
                    img.crossOrigin = "Anonymous";
                    img.src = url;
                    img.onload = () => resolve(img);
                    img.onerror = () => resolve(null);
                });
            };

            let logoImg = await loadImage(logoUrl);
            if (!logoImg && logoUrl !== "/logo.jpeg") {
                logoImg = await loadImage("/logo.jpeg");
            }

            // 3. Iterar y calcular datos
            const tableRows = [];

            for (const assignment of completedAssignments) {
                try {
                    const responsesData = await getExamResponses(assignment.id_assignment);
                    const correctCount = responsesData.filter(r => r.is_correct === true).length;
                    const incorrectCount = responsesData.filter(r => r.is_correct === false).length;
                    const score = Number(parseFloat(assignment.score || 0).toFixed(2));

                    tableRows.push([
                        assignment.student_name || "Estudiante",
                        assignment.attempt_number || 1,
                        correctCount,
                        incorrectCount,
                        `${score}`
                    ]);
                } catch (err) {
                    console.error(`Error fetching responses for assignment ${assignment.id_assignment}`, err);
                }
            }

            // 4. Generar PDF
            const doc = new jsPDF();
            const pageWidth = doc.internal.pageSize.getWidth();
            const pageHeight = doc.internal.pageSize.getHeight();
            const primaryColor = [41, 128, 185];

            // --- HEADER ---
            if (logoImg) {
                doc.addImage(logoImg, 'JPEG', 15, 10, 25, 25);
            }

            doc.setFontSize(16);
            doc.setTextColor(...primaryColor);
            doc.setFont("helvetica", "bold");
            doc.text(institutionName.toUpperCase(), pageWidth / 2, 20, { align: "center" });

            doc.setFontSize(10);
            doc.setTextColor(100);
            doc.setFont("helvetica", "normal");
            const dateStr = new Date().toLocaleDateString();
            doc.text(dateStr, pageWidth - 15, 20, { align: "right" });

            doc.setFontSize(14);
            doc.setTextColor(0, 0, 0);
            doc.text("Reporte General de Resultados", pageWidth / 2, 30, { align: "center" });

            doc.setFontSize(12);
            doc.setFont("helvetica", "bold");
            doc.text(`Examen: ${exam.title}`, pageWidth / 2, 38, { align: "center" });


            doc.setDrawColor(...primaryColor);
            doc.setLineWidth(0.5);
            doc.line(15, 42, pageWidth - 15, 42);

            // --- TABLA ---
            autoTable(doc, {
                startY: 50,
                head: [['Estudiante', 'Intento', 'Correctas', 'Incorrectas', 'Calificación']],
                body: tableRows,
                theme: 'striped',
                headStyles: { fillColor: primaryColor },
                styles: { fontSize: 10 },
                margin: { left: 15, right: 15 }
            });

            // --- FOOTER ---
            const totalPages = doc.internal.getNumberOfPages();
            doc.setFontSize(10);
            doc.setTextColor(150);
            for (let i = 1; i <= totalPages; i++) {
                doc.setPage(i);
                doc.text("KeQuiz - © 2026 KeMarketing - https://kemarketing.mx", pageWidth / 2, pageHeight - 10, { align: "center" });
                doc.text(`Página ${i} de ${totalPages}`, pageWidth - 15, pageHeight - 10, { align: "right" });
            }

            doc.save(`Reporte_General_${(exam.title || 'Examen').replace(/\s+/g, '_')}.pdf`);
            toast({ title: 'Reporte generado con éxito', status: 'success' });

        } catch (error) {
            console.error(error);
            toast({ title: 'Error al generar reporte', description: error.message, status: 'error' });
        } finally {
            setGeneratingReport(null);
        }
    };

    const handleGeneratePDF = async (assignment) => {
        try {
            // 1. Obtener respuestas
            const responsesData = await getExamResponses(assignment.id_assignment);

            // 2. Obtener datos de la Institución
            const user = JSON.parse(sessionStorage.getItem('user'));
            let institutionName = "KeQuiz Institution";
            let logoUrl = "/logo.jpeg";

            if (user?.institution_id) {
                try {
                    const inst = await getInstitutionById(user.institution_id);
                    if (inst) {
                        institutionName = inst.name || institutionName;
                        if (inst.logo_url) {
                            logoUrl = inst.logo_url;
                        }
                    }
                } catch (e) {
                    console.warn("No se pudo obtener la institución", e);
                }
            }

            // Helper para cargar imagen
            const loadImage = (url) => {
                return new Promise((resolve) => {
                    const img = new Image();
                    img.crossOrigin = "Anonymous";
                    img.src = url;
                    img.onload = () => resolve(img);
                    img.onerror = () => resolve(null); // Si falla, resolvemos null para usar fallback o nada
                });
            };

            let logoImg = await loadImage(logoUrl);
            if (!logoImg && logoUrl !== "/logo.jpeg") {
                // Fallback al logo por defecto si falla el de la institución
                logoImg = await loadImage("/logo.jpeg");
            }

            // 3. Calcular estadísticas
            const totalQuestions = responsesData.length;
            const correctCount = responsesData.filter(r => r.is_correct === true).length;
            const incorrectCount = responsesData.filter(r => r.is_correct === false).length;

            // 4. Generar PDF
            const doc = new jsPDF();
            const pageWidth = doc.internal.pageSize.getWidth();
            const pageHeight = doc.internal.pageSize.getHeight();

            const primaryColor = [41, 128, 185];

            // --- HEADER ---
            // Logo (Izquierda)
            if (logoImg) {
                doc.addImage(logoImg, 'JPEG', 15, 10, 25, 25); // x, y, w, h
            }

            // Nombre Institución (Centro)
            doc.setFontSize(16);
            doc.setTextColor(...primaryColor);
            doc.setFont("helvetica", "bold");
            doc.text(institutionName.toUpperCase(), pageWidth / 2, 20, { align: "center" });

            // Fecha (Derecha o debajo del nombre para que se vea bien "en el encabezado")
            // El usuario pidió "Fecha en el encabezado". La pondremos a la derecha alineada.
            doc.setFontSize(10);
            doc.setTextColor(100);
            doc.setFont("helvetica", "normal");
            const dateStr = new Date().toLocaleDateString();
            doc.text(dateStr, pageWidth - 15, 20, { align: "right" });

            // Título del Reporte
            doc.setFontSize(14);
            doc.setTextColor(0, 0, 0);
            doc.text("Reporte de Resultados", pageWidth / 2, 30, { align: "center" });

            // Línea separadora header
            doc.setDrawColor(...primaryColor);
            doc.setLineWidth(0.5);
            doc.line(15, 38, pageWidth - 15, 38);

            // --- CONTENIDO ---

            // Info del Examen y Estudiante
            let yPos = 50;
            const leftColX = 20;
            const valColX = 60;

            doc.setFontSize(12);
            doc.setTextColor(0, 0, 0);

            doc.setFont("helvetica", "bold");
            doc.text("Examen:", leftColX, yPos);
            doc.setFont("helvetica", "normal");
            doc.text(selectedExam?.title || "Examen", valColX, yPos);

            yPos += 10;
            doc.setFont("helvetica", "bold");
            doc.text("Estudiante:", leftColX, yPos);
            doc.setFont("helvetica", "normal");
            doc.text(assignment.student_name || "N/A", valColX, yPos);

            yPos += 10;
            doc.setFont("helvetica", "bold");
            doc.text("Intento:", leftColX, yPos);
            doc.setFont("helvetica", "normal");
            doc.text(String(assignment.attempt_number || 1), valColX, yPos);

            yPos += 10;
            doc.setFont("helvetica", "bold");
            doc.text("Aplicado el:", leftColX, yPos);
            doc.setFont("helvetica", "normal");
            doc.text(new Date(assignment.completed_at || assignment.started_at).toLocaleString(), valColX, yPos);

            yPos += 10;
            doc.setFont("helvetica", "bold");
            doc.text("Calificación:", leftColX, yPos);
            doc.setFont("helvetica", "normal");
            const score = Number(parseFloat(assignment.score || 0).toFixed(2));
            const scoreColor = score >= 7 ? [39, 174, 96] : [192, 57, 43];
            doc.setTextColor(...scoreColor);
            doc.text(`${score}`, valColX, yPos);

            // Resumen Estadístico
            yPos += 20;
            doc.setTextColor(0, 0, 0);
            doc.setFontSize(14);
            doc.text("Resumen de Respuestas", leftColX, yPos);

            const summaryData = [
                ['Total Preguntas', totalQuestions],
                ['Respuestas Correctas', correctCount],
                ['Respuestas Incorrectas (o parciales)', incorrectCount],
            ];

            autoTable(doc, {
                startY: yPos + 5,
                head: [['Métrica', 'Cantidad']],
                body: summaryData,
                theme: 'striped',
                headStyles: { fillColor: primaryColor },
                styles: { fontSize: 12 },
                margin: { left: 15, right: 15 }
            });

            // --- FOOTER ---
            const totalPages = doc.internal.getNumberOfPages();
            doc.setFontSize(10);
            doc.setTextColor(150);
            for (let i = 1; i <= totalPages; i++) {
                doc.setPage(i);
                doc.text("KeQuiz - © 2026 KeMarketing - https://kemarketing.mx", pageWidth / 2, pageHeight - 10, { align: "center" });
                doc.text(`Página ${i} de ${totalPages}`, pageWidth - 15, pageHeight - 10, { align: "right" });
            }

            doc.save(`Reporte_${(selectedExam?.title || 'Examen').replace(/\s+/g, '_')}_${(assignment.student_name || 'Estudiante').replace(/\s+/g, '_')}.pdf`);

            toast({ title: 'Reporte generado con éxito', status: 'success', duration: 3000 });

        } catch (error) {
            console.error(error);
            toast({ title: 'Error al generar reporte', description: error.message, status: 'error' });
        }
    };

    return (
        <Box>
            <Heading mb={6}>Revisión y Calificaciones</Heading>

            {loading ? (
                <Flex justify="center" py={10}><Spinner /></Flex>
            ) : (
                <VStack spacing={8} align="stretch">
                    {/* Estadísticas */}
                    <SimpleGrid columns={{ base: 1, md: 3 }} spacing={5}>
                        <Card>
                            <CardBody>
                                <Stat>
                                    <StatLabel>Total Exámenes</StatLabel>
                                    <StatNumber>{totalExams}</StatNumber>
                                    <StatHelpText>Creados por ti</StatHelpText>
                                </Stat>
                            </CardBody>
                        </Card>
                        <Card>
                            <CardBody>
                                <Stat>
                                    <StatLabel>Estudiantes Evaluados</StatLabel>
                                    <StatNumber>{totalStudentsEvaluated}</StatNumber>
                                    <StatHelpText>Total acumulado</StatHelpText>
                                </Stat>
                            </CardBody>
                        </Card>
                        <Card>
                            <CardBody>
                                <Stat>
                                    <StatLabel>Actividad Reciente</StatLabel>
                                    <StatNumber>{activeExams}</StatNumber>
                                    <StatHelpText>
                                        <StatArrow type={activeExams > 0 ? 'increase' : 'gray'} />
                                        Exámenes esta semana
                                    </StatHelpText>
                                </Stat>
                            </CardBody>
                        </Card>
                    </SimpleGrid>

                    {/* Barra de Búsqueda */}
                    <Box bg="white" p={4} borderRadius="lg" shadow="sm">
                        <InputGroup>
                            <InputLeftElement pointerEvents="none">
                                <Icon as={FiSearch} color="gray.400" />
                            </InputLeftElement>
                            <Input
                                placeholder="Buscar examen por título o descripción..."
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                            />
                        </InputGroup>
                    </Box>

                    {/* Grid de Exámenes */}
                    <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={6}>
                        {filteredExams.map(exam => {
                            const examId = exam.id_exam || exam.id;
                            const studentsCountForExam = studentsCount[examId] || 0;
                            const applicationDate = examApplicationDates[examId];

                            // Verificar si la fecha es anterior a una semana
                            const oneWeekAgo = new Date();
                            oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
                            const isOlderThanWeek = applicationDate && applicationDate < oneWeekAgo;

                            // Formatear fecha
                            const formatDate = (date) => {
                                if (!date) return 'No aplicado';
                                const day = String(date.getDate()).padStart(2, '0');
                                const month = String(date.getMonth() + 1).padStart(2, '0');
                                const year = date.getFullYear();
                                const hours = date.getHours();
                                const minutes = String(date.getMinutes()).padStart(2, '0');
                                const ampm = hours >= 12 ? 'p.m.' : 'a.m.';
                                const displayHours = hours % 12 || 12;
                                return `${day}/${month}/${year}, ${displayHours}:${minutes} ${ampm}`;
                            };

                            return (
                                <Card
                                    key={exam.id}
                                    variant="outline"
                                    _hover={{ borderColor: 'brand.500', shadow: 'sm' }}
                                    cursor="pointer"
                                    onClick={() => handleViewAssignments(exam)}
                                    opacity={isOlderThanWeek ? 0.7 : 1}
                                    bg={isOlderThanWeek ? 'gray.50' : 'white'}
                                >
                                    <CardBody>
                                        <VStack align="start" spacing={3}>
                                            <Badge colorScheme="brand">Examen</Badge>

                                            {/* Información del Examen */}
                                            <Box w="full" p={3} bg="blue.50" borderRadius="md" border="1px solid" borderColor="blue.200">
                                                <VStack align="start" spacing={2}>
                                                    <Box w="full">
                                                        <Text fontSize="xs" fontWeight="bold" color="blue.700" mb={1}>Nombre:</Text>
                                                        <Text fontSize="sm" fontWeight="semibold" color="blue.900">{exam.title || 'N/A'}</Text>
                                                    </Box>
                                                    <Box w="full">
                                                        <Text fontSize="xs" fontWeight="bold" color="blue.700" mb={1}>Descripción:</Text>
                                                        <Text fontSize="xs" color="blue.800" noOfLines={2}>{exam.description || 'Sin descripción'}</Text>
                                                    </Box>
                                                    <Box w="full">
                                                        <Text fontSize="xs" fontWeight="bold" color="blue.700" mb={1}>Estudiantes:</Text>
                                                        <Text fontSize="sm" fontWeight="semibold" color="blue.900">
                                                            {studentsCountForExam} {studentsCountForExam === 1 ? 'estudiante ha aplicado' : 'estudiantes han aplicado'} el examen
                                                        </Text>
                                                    </Box>
                                                    <Box w="full">
                                                        <Text fontSize="xs" fontWeight="bold" color="blue.700" mb={1}>Fecha de Aplicación:</Text>
                                                        <Text
                                                            fontSize="sm"
                                                            fontWeight="semibold"
                                                            color={isOlderThanWeek ? 'gray.500' : 'blue.900'}
                                                        >
                                                            {formatDate(applicationDate)}
                                                        </Text>
                                                    </Box>
                                                </VStack>
                                            </Box>

                                            <Divider />
                                            <HStack spacing={2} width="full">
                                                <Button size="sm" colorScheme="brand" variant="ghost" leftIcon={<FiSearch />} width="50%" onClick={() => handleViewAssignments(exam)}>
                                                    Revisar
                                                </Button>
                                                <Button
                                                    size="sm"
                                                    colorScheme="blue"
                                                    variant="ghost"
                                                    leftIcon={<FiFileText />}
                                                    width="50%"
                                                    onClick={(e) => {
                                                        e.stopPropagation();
                                                        handleGenerateExamReport(exam);
                                                    }}
                                                    isLoading={generatingReport === (exam.id_exam || exam.id)}
                                                    loadingText="Generando"
                                                >
                                                    Reporte
                                                </Button>
                                            </HStack>
                                        </VStack>
                                    </CardBody>
                                </Card>
                            );
                        })}
                        {exams.length === 0 && (
                            <Box py={10} textAlign="center" gridColumn="span 3">
                                <Text color="gray.500">No tienes exámenes creados todavía.</Text>
                            </Box>
                        )}
                    </SimpleGrid>

                    {/* Modal: Listado de Alumnos que entregaron */}
                    <Modal isOpen={isExamListOpen} onClose={onExamListClose} size="4xl">
                        <ModalOverlay />
                        <ModalContent>
                            <ModalHeader>Entregas: {selectedExam?.title}</ModalHeader>
                            <ModalCloseButton />
                            <ModalBody pb={6}>
                                {/* Información del Examen */}
                                <Box mb={6} p={4} bg="blue.50" borderRadius="lg" border="1px solid" borderColor="blue.200">
                                    <SimpleGrid columns={{ base: 1, md: 3 }} spacing={4}>
                                        <Box>
                                            <Text fontSize="xs" fontWeight="bold" color="blue.700" mb={1}>Nombre:</Text>
                                            <Text fontSize="sm" fontWeight="semibold" color="blue.900">{selectedExam?.title || 'N/A'}</Text>
                                        </Box>
                                        <Box>
                                            <Text fontSize="xs" fontWeight="bold" color="blue.700" mb={1}>Descripción:</Text>
                                            <Text fontSize="sm" color="blue.800" noOfLines={2}>{selectedExam?.description || 'Sin descripción'}</Text>
                                        </Box>
                                        <Box>
                                            <Text fontSize="xs" fontWeight="bold" color="blue.700" mb={1}>Estudiantes:</Text>
                                            <Text fontSize="sm" fontWeight="semibold" color="blue.900">
                                                {(() => {
                                                    const uniqueStudents = new Set(assignments.map(a => Number(a.student_id)));
                                                    const count = uniqueStudents.size;
                                                    return `${count} ${count === 1 ? 'estudiante ha aplicado' : 'estudiantes han aplicado'} el examen`;
                                                })()}
                                            </Text>
                                        </Box>
                                    </SimpleGrid>
                                </Box>

                                <Divider mb={4} />

                                <Table variant="simple" size="sm">
                                    <Thead bg="gray.50">
                                        <Tr>
                                            <Th>Estudiante</Th>
                                            <Th>Fecha de Entrega</Th>
                                            <Th isNumeric>Calificación</Th>
                                            <Th textAlign="center">Acciones</Th>
                                        </Tr>
                                    </Thead>
                                    <Tbody>
                                        {assignments.map(a => (
                                            <Tr key={a.id_assignment}>
                                                <Td>
                                                    <VStack align="start" spacing={0}>
                                                        <Text fontWeight="medium">{a.student_name}</Text>
                                                        {a.attempt_number > 1 && (
                                                            <Badge colorScheme="gray" fontSize="9px">Intento #{a.attempt_number}</Badge>
                                                        )}
                                                    </VStack>
                                                </Td>
                                                <Td fontSize="xs">{new Date(a.completed_at).toLocaleString()}</Td>
                                                <Td isNumeric>
                                                    <Badge colorScheme={parseFloat(a.score) >= 7 ? 'green' : 'red'}>
                                                        {Number(parseFloat(a.score).toFixed(2))} / 10
                                                    </Badge>
                                                </Td>
                                                <Td textAlign="center">
                                                    <HStack justify="center" spacing={2}>
                                                        <Button size="xs" colorScheme="blue" leftIcon={<FiEye />} onClick={() => handleReviewExam(a)}>
                                                            Revisar
                                                        </Button>
                                                        <Button size="xs" colorScheme="red" variant="outline" leftIcon={<FiFileText />} onClick={() => handleGeneratePDF(a)} title="Descargar Reporte PDF">
                                                            PDF
                                                        </Button>
                                                    </HStack>
                                                </Td>
                                            </Tr>
                                        ))}
                                        {assignments.length === 0 && (
                                            <Tr>
                                                <Td colSpan={4} textAlign="center" py={10}>
                                                    <Text color="gray.500">No hay entregas finalizadas para este examen.</Text>
                                                </Td>
                                            </Tr>
                                        )}
                                    </Tbody>
                                </Table>
                            </ModalBody>
                        </ModalContent>
                    </Modal>

                    {/* Modal: Revisión Detallada de Respuestas */}
                    <Modal isOpen={isReviewOpen} onClose={onReviewClose} size="full">
                        <ModalOverlay />
                        <ModalContent>
                            <ModalHeader bg="brand.600" color="white">
                                <HStack justify="space-between" pr={10}>
                                    <Text>Revisión de Examen: {selectedAssignment?.student_name}</Text>
                                    <Badge colorScheme="whiteAlpha" fontSize="lg" px={4} py={1}>
                                        Calificación Actual: {Number(parseFloat(selectedAssignment?.score).toFixed(2))} / 10
                                    </Badge>
                                </HStack>
                            </ModalHeader>
                            <ModalCloseButton color="white" />
                            <ModalBody bg="gray.50" py={8}>
                                <VStack spacing={6} maxW="1000px" mx="auto" align="stretch">
                                    {responses.length === 0 ? (
                                        <Box py={20} textAlign="center">
                                            <Icon as={FiSearch} fontSize="4xl" color="gray.300" mb={4} />
                                            <Text color="gray.500" fontSize="lg">No se encontraron respuestas registradas para este intento.</Text>
                                            <Text color="gray.400" fontSize="sm">Es posible que el examen se haya cerrado automáticamente por tiempo o tolerancia.</Text>
                                        </Box>
                                    ) : responses.map((resp, index) => {
                                        const content = typeof resp.question_content === 'string' ? JSON.parse(resp.question_content) : resp.question_content;
                                        const isMultipleChoice = resp.question_type === 'MULTIPLE_CHOICE' || resp.question_type === 'TRUE_FALSE';

                                        return (
                                            <Card key={resp.id_response} variant="outline" borderLeft="5px solid" borderLeftColor={resp.is_correct === true ? "green.400" : resp.is_correct === false ? "red.400" : "orange.400"}>
                                                <CardBody>
                                                    <VStack align="stretch" spacing={4}>
                                                        <Flex justify="space-between">
                                                            <Box>
                                                                <Badge colorScheme="gray" mb={2}>Pregunta {index + 1}</Badge>
                                                                <Text fontWeight="bold" fontSize="lg">{content.question}</Text>
                                                            </Box>
                                                            <VStack align="end">
                                                                <Badge colorScheme={isMultipleChoice ? "purple" : "orange"}>
                                                                    {isMultipleChoice ? "Opción Múltiple" : "Pregunta Abierta"}
                                                                </Badge>
                                                                <Text fontSize="xs" color="gray.500">Puntos: {resp.points_earned}</Text>
                                                            </VStack>
                                                        </Flex>

                                                        <Box p={4} bg="white" borderRadius="md" border="1px solid" borderColor="gray.100">
                                                            <Text fontSize="sm" color="gray.500" mb={1}>Respuesta del estudiante:</Text>
                                                            {isMultipleChoice ? (
                                                                <Text fontWeight="semibold">
                                                                    {(content.options && content.options[parseInt(resp.student_answer)])
                                                                        ? content.options[parseInt(resp.student_answer)]
                                                                        : (resp.student_answer || "Sin respuesta")
                                                                    }
                                                                </Text>
                                                            ) : (
                                                                <Text fontWeight="semibold" whiteSpace="pre-wrap">
                                                                    {resp.student_answer || "El estudiante no respondió."}
                                                                </Text>
                                                            )}
                                                        </Box>

                                                        {isMultipleChoice ? (
                                                            <Alert status={resp.is_correct ? "success" : "error"} variant="subtle" size="sm" borderRadius="md" py={2}>
                                                                <AlertIcon />
                                                                <Box>
                                                                    <Text fontSize="sm" fontWeight="bold">
                                                                        {resp.is_correct ? "Correcta (Auto-calificado)" : "Incorrecta"}
                                                                    </Text>
                                                                    {!resp.is_correct && (
                                                                        <Text fontSize="xs">
                                                                            Respuesta correcta: {resp.correct_answer || "No especificada"}
                                                                        </Text>
                                                                    )}
                                                                </Box>
                                                            </Alert>
                                                        ) : (
                                                            <Box bg="orange.50" p={4} borderRadius="md">
                                                                <FormControl display="flex" alignItems="center">
                                                                    <FormLabel mb="0" flex="1">¿Es correcta esta respuesta?</FormLabel>
                                                                    <HStack>
                                                                        <Button
                                                                            size="sm"
                                                                            colorScheme={resp.is_correct === true ? "green" : "gray"}
                                                                            leftIcon={<FiCheck />}
                                                                            onClick={() => handleUpdateGrade(resp.id_response, true, 1)}
                                                                        >
                                                                            Correcta
                                                                        </Button>
                                                                        <Button
                                                                            size="sm"
                                                                            colorScheme={resp.is_correct === false ? "red" : "gray"}
                                                                            leftIcon={<FiX />}
                                                                            onClick={() => handleUpdateGrade(resp.id_response, false, 0)}
                                                                        >
                                                                            Incorrecta
                                                                        </Button>
                                                                    </HStack>
                                                                </FormControl>
                                                            </Box>
                                                        )}
                                                    </VStack>
                                                </CardBody>
                                            </Card>
                                        );
                                    })}
                                </VStack>
                            </ModalBody>
                            <ModalFooter bg="white">
                                <Button colorScheme="brand" onClick={onReviewClose}>Finalizar Revisión</Button>
                            </ModalFooter>
                        </ModalContent>
                    </Modal>
                </VStack>
            )}

            {/* Modal code remains here */}
        </Box>
    );
};

export default RevisionExamenes;

