import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    FormControl,
    FormLabel,
    Input,
    Button,
    VStack,
    useToast,
    Text,
    Card,
    CardBody,
    Divider
} from '@chakra-ui/react';
import { getSystemConfig, updateSystemConfig } from '../../services/api';

const ConfiguracionFooter = () => {
    const [config, setConfig] = useState({});
    const toast = useToast();

    useEffect(() => {
        const fetchData = async () => {
            try {
                const data = await getSystemConfig();
                // Convert array to object for easier handling in form
                const configObj = {};
                if (Array.isArray(data)) {
                    data.forEach(item => {
                        configObj[item.config_key] = item.config_value;
                    });
                }
                setConfig(configObj);
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar configuración',
                    status: 'error',
                    duration: 3000,
                });
            }
        };
        fetchData();
    }, []);

    const handleSave = async () => {
        try {
            await Promise.all([
                updateSystemConfig('footer_text', config.footer_text),
                updateSystemConfig('footer_link_text', config.footer_link_text),
                updateSystemConfig('footer_link_url', config.footer_link_url)
            ]);

            toast({
                title: 'Configuración guardada',
                description: 'Los cambios se reflejarán en todo el sistema',
                status: 'success',
                duration: 3000,
            });
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    return (
        <Box maxW="container.md">
            <Heading mb={6}>Configuración Global</Heading>

            <Card>
                <CardBody>
                    <VStack spacing={6} align="stretch">
                        <Box>
                            <Heading size="md" mb={2}>Pie de Página (Footer)</Heading>
                            <Text color="gray.500" fontSize="sm" mb={4}>
                                Personaliza el texto y enlaces que aparecen en el pie de página de todas las pantallas del sistema.
                            </Text>
                        </Box>

                        <Divider />

                        <FormControl>
                            <FormLabel>Texto de Copyright</FormLabel>
                            <Input
                                value={config.footer_text}
                                onChange={(e) => setConfig({ ...config, footer_text: e.target.value })}
                                placeholder="© 2025 KeSoftware"
                            />
                            <Text fontSize="xs" color="gray.500" mt={1}>
                                Aparece al final de todas las páginas.
                            </Text>
                        </FormControl>

                        <FormControl>
                            <FormLabel>Texto del Enlace</FormLabel>
                            <Input
                                value={config.footer_link_text}
                                onChange={(e) => setConfig({ ...config, footer_link_text: e.target.value })}
                                placeholder="Soporte Técnico"
                            />
                        </FormControl>

                        <FormControl>
                            <FormLabel>URL del Enlace</FormLabel>
                            <Input
                                value={config.footer_link_url}
                                onChange={(e) => setConfig({ ...config, footer_link_url: e.target.value })}
                                placeholder="https://soporte.kesoftware.com"
                            />
                        </FormControl>

                        <Box pt={4}>
                            <Button colorScheme="brand" onClick={handleSave}>
                                Guardar Cambios
                            </Button>
                        </Box>
                    </VStack>
                </CardBody>
            </Card>
        </Box>
    );
};

export default ConfiguracionFooter;
