import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Checkbox,
    useToast,
    Text,
    Badge,
    HStack,
    Avatar
} from '@chakra-ui/react';
import { getCompanies, getModules, getModulesByCompany, toggleCompanyModule } from '../../services/api';

const ControlModulos = () => {
    const [companies, setCompanies] = useState([]);
    const [modules, setModules] = useState([]);
    const [companyModulesMap, setCompanyModulesMap] = useState({});
    const toast = useToast();

    const fetchData = async () => {
        try {
            const [fetchedCompanies, fetchedModules] = await Promise.all([
                getCompanies(),
                getModules()
            ]);
            setCompanies(fetchedCompanies);
            setModules(fetchedModules);

            const map = {};
            // Fetch modules for each company
            // Note: In a larger system this might be inefficient, but for < 10 companies it's fine.
            // Ideally backend should provide a bulk endpoint or include modules in company list.
            await Promise.all(fetchedCompanies.map(async (comp) => {
                const mods = await getModulesByCompany(comp.id);
                map[comp.id] = mods;
            }));
            setCompanyModulesMap(map);
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar datos',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchData();
    }, []);

    const isModuleEnabled = (companyId, moduleId) => {
        const mods = companyModulesMap[companyId];
        if (!mods) return false;
        // The endpoint returns list of modules with is_enabled flag.
        // We need to find the module in the list.
        // Wait, getModulesByCompany returns ALL modules with is_enabled flag?
        // Let's check the model: getByCompanyId returns m.*, cm.is_enabled.
        // So yes, it returns all modules, some with is_enabled true/false/null.
        const mod = mods.find(m => m.id === moduleId);
        return mod ? !!mod.is_enabled : false;
    };

    const toggleModule = async (companyId, moduleId) => {
        const currentlyEnabled = isModuleEnabled(companyId, moduleId);

        try {
            await toggleCompanyModule(companyId, moduleId, !currentlyEnabled);

            // Update local state optimistically or refetch
            const mods = await getModulesByCompany(companyId);
            setCompanyModulesMap(prev => ({ ...prev, [companyId]: mods }));

            const company = companies.find(c => c.id === companyId);
            const module = modules.find(m => m.id === moduleId);

            toast({
                title: !currentlyEnabled ? 'Módulo habilitado' : 'Módulo deshabilitado',
                description: `${module?.name} para ${company?.name}`,
                status: !currentlyEnabled ? 'success' : 'info',
                duration: 2000,
                isClosable: true,
            });
        } catch (error) {
            toast({
                title: 'Error al actualizar módulo',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    return (
        <Box>
            <Heading mb={6}>Control de Acceso a Módulos</Heading>
            <Text mb={6} color="gray.600">
                Administra qué módulos están disponibles para cada empresa inquilina.
            </Text>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Empresa</Th>
                            {modules.map(module => (
                                <Th key={module.id} textAlign="center">{module.name}</Th>
                            ))}
                        </Tr>
                    </Thead>
                    <Tbody>
                        {companies.map((company) => (
                            <Tr key={company.id}>
                                <Td>
                                    <HStack>
                                        <Avatar size="xs" name={company.name} src={company.logo_url} />
                                        <Box>
                                            <Text fontWeight="bold" fontSize="sm">{company.name}</Text>
                                            {!company.is_active && (
                                                <Badge colorScheme="red" fontSize="xx-small">INACTIVA</Badge>
                                            )}
                                        </Box>
                                    </HStack>
                                </Td>
                                {modules.map(module => (
                                    <Td key={module.id} textAlign="center">
                                        <Checkbox
                                            colorScheme="brand"
                                            isChecked={isModuleEnabled(company.id, module.id)}
                                            onChange={() => toggleModule(company.id, module.id)}
                                            isDisabled={!company.is_active}
                                        />
                                    </Td>
                                ))}
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>
        </Box>
    );
};

export default ControlModulos;
