import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Select,
    useToast,
    HStack,
    Text,
    Avatar,
    FormErrorMessage
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiPower, FiEye, FiEyeOff } from 'react-icons/fi';

const validateRFC = (rfc) => {
    if (!rfc) return true; // Opcional
    const cleanRFC = rfc.replace(/\s/g, '').toUpperCase();
    const rfc12 = /^[A-Z]{3}[0-9]{6}[A-Z]{1}[A-Z0-9]{2}$/;
    const rfc13 = /^[A-Z]{4}[0-9]{6}[A-Z]{1}[A-Z0-9]{2}$/;
    return rfc12.test(cleanRFC) || rfc13.test(cleanRFC);
};

const validatePhone = (phone) => {
    if (!phone) return false;
    const cleanPhone = phone.replace(/\s/g, '');
    return /^[0-9]{10}$/.test(cleanPhone);
};

const validateEmail = (email) => /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);

const formatPhone = (value) => {
    if (!value) return value;
    const phoneNumber = value.replace(/[^\d]/g, '');
    const phoneNumberLength = phoneNumber.length;
    if (phoneNumberLength <= 2) return phoneNumber;
    if (phoneNumberLength <= 6) {
        return `${phoneNumber.slice(0, 2)} ${phoneNumber.slice(2)}`;
    }
    return `${phoneNumber.slice(0, 2)} ${phoneNumber.slice(2, 6)} ${phoneNumber.slice(6, 10)}`;
};

const formatRFC = (value) => {
    if (!value) return value;
    // Solo permitir letras, números y espacios, convertir a mayúsculas
    const cleanValue = value.replace(/[^A-Za-z0-9\s]/g, '').toUpperCase();
    // Remover espacios para validación
    const cleanRFC = cleanValue.replace(/\s/g, '');
    // Limitar a 13 caracteres
    return cleanRFC.slice(0, 13);
};

import { getCompanies, createCompany, updateCompany } from '../../services/api';

const GestionEmpresas = () => {
    const [companies, setCompanies] = useState([]);
    const [currentCompany, setCurrentCompany] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();
    const [errors, setErrors] = useState({});

    const fetchCompanies = async () => {
        try {
            const data = await getCompanies();
            setCompanies(data);
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar empresas',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchCompanies();
    }, []);

    const handleEdit = (company) => {
        setCurrentCompany({
            ...company,
            phone: company.phone ? formatPhone(company.phone) : '',
            rfc: company.rfc ? formatRFC(company.rfc) : ''
        });
        setErrors({});
        onOpen();
    };

    const handleCreate = () => {
        setCurrentCompany({
            name: '',
            rfc: '',
            email: '',
            phone: '',
            is_active: true,
            street: '',
            exterior_number: '',
            neighborhood: '',
            municipality: '',
            state: '',
            postal_code: '',
            max_clients: 0,
            max_employees: 0
        });
        setErrors({});
        onOpen();
    };

    const handleSave = async () => {
        // Validaciones
        const newErrors = {};
        
        if (!currentCompany.email || !validateEmail(currentCompany.email)) {
            newErrors.email = 'Email inválido';
        }
        
        if (!currentCompany.phone || !validatePhone(currentCompany.phone)) {
            newErrors.phone = 'Teléfono inválido (debe tener 10 dígitos)';
        }
        
        if (currentCompany.rfc && !validateRFC(currentCompany.rfc)) {
            newErrors.rfc = 'RFC inválido (formato: 12 o 13 caracteres)';
        }
        
        if (Object.keys(newErrors).length > 0) {
            setErrors(newErrors);
            toast({ 
                title: 'Error de validación', 
                description: 'Por favor corrige los campos marcados', 
                status: 'error', 
                duration: 3000 
            });
            return;
        }
        
        setErrors({});

        try {
            // Limpiar formato del teléfono antes de guardar (quitar espacios)
            const dataToSave = {
                ...currentCompany,
                phone: currentCompany.phone ? currentCompany.phone.replace(/\s/g, '') : '',
                rfc: currentCompany.rfc ? currentCompany.rfc.replace(/\s/g, '').toUpperCase() : ''
            };

            if (currentCompany.id) {
                await updateCompany(currentCompany.id, dataToSave);
                toast({
                    title: 'Empresa actualizada',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                await createCompany({
                    ...dataToSave,
                    logo_url: 'https://via.placeholder.com/150' // Default logo
                });
                toast({
                    title: 'Empresa creada',
                    status: 'success',
                    duration: 2000,
                });
            }
            fetchCompanies();
            onClose();
        } catch (error) {
            // Detectar errores de campos duplicados
            const errorMessage = error.message || error.toString();
            
            let friendlyMessage = errorMessage;
            
            if (errorMessage.includes('correo es utilizado') || errorMessage.includes('users_email_key') || 
                (errorMessage.includes('duplicate key') && errorMessage.includes('email'))) {
                friendlyMessage = 'El correo es utilizado por otro usuario, favor de ingresar otro correo electrónico';
            } else if (errorMessage.includes('teléfono es utilizado') || errorMessage.includes('users_phone_key') || 
                       (errorMessage.includes('duplicate key') && errorMessage.includes('phone'))) {
                friendlyMessage = 'El teléfono es utilizado por otro usuario, favor de ingresar otro teléfono';
            } else if (errorMessage.includes('CURP es utilizado') || errorMessage.includes('users_curp_key') || 
                       (errorMessage.includes('duplicate key') && errorMessage.includes('curp'))) {
                friendlyMessage = 'El CURP es utilizado por otro usuario, favor de ingresar otro CURP';
            }
            
            toast({
                title: 'Error al guardar',
                description: friendlyMessage,
                status: 'error',
                duration: 4000,
            });
        }
    };

    const toggleStatus = async (id) => {
        const company = companies.find(c => c.id === id);
        if (!company) return;

        try {
            const newStatus = !company.is_active;
            await updateCompany(id, { ...company, is_active: newStatus });

            toast({
                title: `Empresa ${newStatus ? 'activada' : 'desactivada'}`,
                status: newStatus ? 'success' : 'warning',
                duration: 2000,
            });
            fetchCompanies();
        } catch (error) {
            toast({
                title: 'Error al actualizar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <Heading>Gestión de Instituciones</Heading>
                <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                    Nueva Institución
                </Button>
            </HStack>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Institución</Th>
                            <Th>Contacto</Th>
                            <Th>Ubicación</Th>
                            <Th>Límites (E/P)</Th>
                            <Th>Estado</Th>

                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {companies.map((company) => (
                            <Tr key={company.id}>
                                <Td>
                                    <HStack>
                                        <Avatar size="sm" name={company.name} src={company.logo_url} />
                                        <Box>
                                            <Text fontWeight="bold">{company.name}</Text>
                                            <Text fontSize="xs" color="gray.500">ID: {company.id}</Text>
                                        </Box>
                                    </HStack>
                                </Td>
                                <Td>
                                    <Text fontSize="sm">{company.email}</Text>
                                    <Text fontSize="xs" color="gray.500">{company.phone}</Text>
                                </Td>
                                <Td>
                                    <Text fontSize="sm">{company.municipality}, {company.state}</Text>
                                </Td>
                                <Td>
                                    <HStack spacing={2}>
                                        <Badge colorScheme="blue" title="Límite de Estudiantes">
                                            {company.max_clients || 0} E
                                        </Badge>
                                        <Badge colorScheme="purple" title="Límite de Profesores">
                                            {company.max_employees || 0} P
                                        </Badge>
                                    </HStack>
                                </Td>
                                <Td>

                                    <Badge colorScheme={company.is_active ? 'green' : 'red'}>
                                        {company.is_active ? 'ACTIVA' : 'INACTIVA'}
                                    </Badge>
                                </Td>
                                <Td>
                                    <HStack spacing={2}>
                                        <IconButton
                                            icon={<FiEdit2 />}
                                            size="sm"
                                            aria-label="Editar"
                                            onClick={() => handleEdit(company)}
                                        />
                                        <IconButton
                                            icon={<FiPower />}
                                            size="sm"
                                            colorScheme={company.is_active ? 'red' : 'green'}
                                            aria-label="Cambiar estado"
                                            onClick={() => toggleStatus(company.id)}
                                        />
                                    </HStack>
                                </Td>
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal de Edición/Creación */}
            <Modal isOpen={isOpen} onClose={onClose} size="xl">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentCompany?.id ? 'Editar Institución' : 'Nueva Institución'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <Box display="grid" gridTemplateColumns="1fr 1fr" gap={4}>
                            <FormControl gridColumn="span 1">
                                <FormLabel>Nombre Institución</FormLabel>
                                <Input
                                    value={currentCompany?.name || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, name: e.target.value })}
                                />
                            </FormControl>

                            <FormControl gridColumn="span 1" isInvalid={!!errors.rfc}>
                                <FormLabel>RFC</FormLabel>
                                <Input
                                    value={currentCompany?.rfc || ''}
                                    onChange={(e) => {
                                        const formatted = formatRFC(e.target.value);
                                        setCurrentCompany({ ...currentCompany, rfc: formatted });
                                        if (errors.rfc) {
                                            setErrors({ ...errors, rfc: null });
                                        }
                                    }}
                                    onBlur={(e) => {
                                        if (e.target.value && !validateRFC(e.target.value)) {
                                            setErrors({ ...errors, rfc: 'RFC inválido (formato: 12 o 13 caracteres)' });
                                        } else {
                                            setErrors({ ...errors, rfc: null });
                                        }
                                    }}
                                    placeholder="ABCD123456EFG"
                                    maxLength={13}
                                />
                                {errors.rfc ? (
                                    <FormErrorMessage>{errors.rfc}</FormErrorMessage>
                                ) : (
                                    <Text fontSize="xs" color="gray.500">Formato: 12 o 13 caracteres</Text>
                                )}
                            </FormControl>


                            <FormControl isRequired isInvalid={!!errors.email}>
                                <FormLabel>Email de Contacto</FormLabel>
                                <Input
                                    type="email"
                                    value={currentCompany?.email || ''}
                                    onChange={(e) => {
                                        setCurrentCompany({ ...currentCompany, email: e.target.value });
                                        if (errors.email) {
                                            setErrors({ ...errors, email: null });
                                        }
                                    }}
                                    onBlur={(e) => {
                                        if (e.target.value && !validateEmail(e.target.value)) {
                                            setErrors({ ...errors, email: 'Email inválido' });
                                        } else {
                                            setErrors({ ...errors, email: null });
                                        }
                                    }}
                                    placeholder="contacto@institucion.com"
                                />
                                {errors.email && <FormErrorMessage>{errors.email}</FormErrorMessage>}
                            </FormControl>

                            <FormControl isRequired isInvalid={!!errors.phone}>
                                <FormLabel>Teléfono</FormLabel>
                                <Input
                                    value={currentCompany?.phone || ''}
                                    onChange={(e) => {
                                        const formatted = formatPhone(e.target.value);
                                        setCurrentCompany({ ...currentCompany, phone: formatted });
                                        if (errors.phone) {
                                            setErrors({ ...errors, phone: null });
                                        }
                                    }}
                                    onBlur={(e) => {
                                        if (e.target.value && !validatePhone(e.target.value)) {
                                            setErrors({ ...errors, phone: 'Teléfono inválido (debe tener 10 dígitos)' });
                                        } else {
                                            setErrors({ ...errors, phone: null });
                                        }
                                    }}
                                    placeholder="55 5555 5555"
                                    maxLength={13}
                                />
                                {errors.phone ? (
                                    <FormErrorMessage>{errors.phone}</FormErrorMessage>
                                ) : (
                                    <Text fontSize="xs" color="gray.500">10 dígitos</Text>
                                )}
                            </FormControl>

                            <FormControl>
                                <FormLabel>Calle</FormLabel>
                                <Input
                                    value={currentCompany?.street || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, street: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Número Exterior</FormLabel>
                                <Input
                                    value={currentCompany?.exterior_number || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, exterior_number: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Colonia</FormLabel>
                                <Input
                                    value={currentCompany?.neighborhood || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, neighborhood: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Municipio</FormLabel>
                                <Input
                                    value={currentCompany?.municipality || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, municipality: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Estado</FormLabel>
                                <Input
                                    value={currentCompany?.state || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, state: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Código Postal</FormLabel>
                                <Input
                                    value={currentCompany?.postal_code || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, postal_code: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel color="blue.600" fontWeight="bold">Límite de Estudiantes</FormLabel>
                                <Input
                                    type="number"
                                    value={currentCompany?.max_clients || 0}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, max_clients: parseInt(e.target.value) || 0 })}
                                    borderColor="blue.200"
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel color="purple.600" fontWeight="bold">Límite de Profesores</FormLabel>
                                <Input
                                    type="number"
                                    value={currentCompany?.max_employees || 0}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, max_employees: parseInt(e.target.value) || 0 })}
                                    borderColor="purple.200"
                                />
                            </FormControl>
                        </Box>

                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionEmpresas;
