import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    useToast,
    HStack,
    Text,
    Avatar
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiPower, FiRefreshCw } from 'react-icons/fi';
import { getUsers, createUser, updateUser } from '../../services/api';

const GestionUsuariosRoot = () => {
    // Filtrar solo usuarios ROOT
    const [rootUsers, setRootUsers] = useState([]);
    const [currentRoot, setCurrentRoot] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    const fetchData = async () => {
        try {
            const usersData = await getUsers();
            setRootUsers(usersData.filter(u => u.role === 'ROOT'));
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar datos',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchData();
    }, []);

    const handleEdit = (rootUser) => {
        setCurrentRoot(rootUser);
        onOpen();
    };

    const handleCreate = () => {
        setCurrentRoot({
            first_name: '',
            last_name: '',
            second_last_name: '',
            email: '',
            role: 'ROOT',
            is_active: true
        });
        onOpen();
    };

    const handleSave = async () => {
        try {
            // Validaciones
            if (!currentRoot.first_name || !currentRoot.first_name.trim()) {
                toast({
                    title: 'Error',
                    description: 'El nombre(s) es obligatorio',
                    status: 'error',
                    duration: 3000,
                });
                return;
            }
            if (!currentRoot.last_name || !currentRoot.last_name.trim()) {
                toast({
                    title: 'Error',
                    description: 'El primer apellido es obligatorio',
                    status: 'error',
                    duration: 3000,
                });
                return;
            }

            if (currentRoot.id) {
                const dataToUpdate = { ...currentRoot };
                if (!dataToUpdate.password) {
                    delete dataToUpdate.password;
                }
                await updateUser(currentRoot.id, dataToUpdate);
                toast({
                    title: 'Superusuario actualizado',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                if (!currentRoot.password) {
                    toast({
                        title: 'Error',
                        description: 'La contraseña es requerida para nuevos superusuarios',
                        status: 'error',
                        duration: 3000,
                    });
                    return;
                }
                await createUser({
                    ...currentRoot,
                    id: `usr-${Date.now()}`,
                    // password will be hashed by backend
                });
                toast({
                    title: 'Superusuario creado',
                    status: 'success',
                    duration: 2000,
                });
            }
            fetchData();
            onClose();
        } catch (error) {
            // Detectar errores de campos duplicados
            const errorMessage = error.message || error.toString();
            
            let friendlyMessage = errorMessage;
            
            if (errorMessage.includes('correo es utilizado') || errorMessage.includes('users_email_key') || 
                (errorMessage.includes('duplicate key') && errorMessage.includes('email'))) {
                friendlyMessage = 'El correo es utilizado por otro usuario, favor de ingresar otro correo electrónico';
            } else if (errorMessage.includes('teléfono es utilizado') || errorMessage.includes('users_phone_key') || 
                       (errorMessage.includes('duplicate key') && errorMessage.includes('phone'))) {
                friendlyMessage = 'El teléfono es utilizado por otro usuario, favor de ingresar otro teléfono';
            } else if (errorMessage.includes('CURP es utilizado') || errorMessage.includes('users_curp_key') || 
                       (errorMessage.includes('duplicate key') && errorMessage.includes('curp'))) {
                friendlyMessage = 'El CURP es utilizado por otro usuario, favor de ingresar otro CURP';
            }
            
            toast({
                title: 'Error al guardar',
                description: friendlyMessage,
                status: 'error',
                duration: 4000,
            });
        }
    };

    const toggleStatus = async (id) => {
        const rootUser = rootUsers.find(r => r.id === id);
        if (!rootUser) return;

        try {
            const newStatus = !rootUser.is_active;
            await updateUser(id, { ...rootUser, is_active: newStatus });

            toast({
                title: `Superusuario ${newStatus ? 'activado' : 'desactivado'}`,
                status: newStatus ? 'success' : 'warning',
                duration: 2000,
            });
            fetchData();
        } catch (error) {
            toast({
                title: 'Error al actualizar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    const resetPassword = (id) => {
        toast({
            title: 'Contraseña restablecida',
            description: 'Se ha enviado un correo con la nueva contraseña',
            status: 'info',
            duration: 3000,
        });
    };

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <Heading>Gestión de Superusuarios</Heading>
                <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                    Nuevo Superusuario
                </Button>
            </HStack>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Usuario</Th>
                            <Th>Estado</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {rootUsers.map((rootUser) => (
                            <Tr key={rootUser.id}>
                                <Td>
                                    <HStack>
                                        <Avatar size="sm" name={rootUser.email} />
                                        <Box>
                                            <Text fontWeight="bold">{rootUser.email}</Text>
                                            <Text fontSize="xs" color="gray.500">ID: {rootUser.id}</Text>
                                        </Box>
                                    </HStack>
                                </Td>
                                <Td>
                                    <Badge colorScheme={rootUser.is_active ? 'green' : 'red'}>
                                        {rootUser.is_active ? 'ACTIVO' : 'INACTIVO'}
                                    </Badge>
                                </Td>
                                <Td>
                                    <HStack spacing={2}>
                                        <IconButton
                                            icon={<FiEdit2 />}
                                            size="sm"
                                            aria-label="Editar"
                                            onClick={() => handleEdit(rootUser)}
                                        />
                                        <IconButton
                                            icon={<FiRefreshCw />}
                                            size="sm"
                                            colorScheme="blue"
                                            variant="ghost"
                                            aria-label="Reset Password"
                                            title="Restablecer Contraseña"
                                            onClick={() => resetPassword(rootUser.id)}
                                        />
                                        <IconButton
                                            icon={<FiPower />}
                                            size="sm"
                                            colorScheme={rootUser.is_active ? 'red' : 'green'}
                                            aria-label="Cambiar estado"
                                            onClick={() => toggleStatus(rootUser.id)}
                                        />
                                    </HStack>
                                </Td>
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal */}
            <Modal isOpen={isOpen} onClose={onClose}>
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentRoot?.id ? 'Editar Superusuario' : 'Nuevo Superusuario'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <Box display="flex" flexDirection="column" gap={4}>
                            <FormControl isRequired>
                                <FormLabel>Nombre(s)</FormLabel>
                                <Input
                                    value={currentRoot?.first_name || ''}
                                    onChange={(e) => setCurrentRoot({ ...currentRoot, first_name: e.target.value })}
                                    placeholder="Ej. Juan Carlos"
                                />
                            </FormControl>

                            <FormControl isRequired>
                                <FormLabel>Primer Apellido</FormLabel>
                                <Input
                                    value={currentRoot?.last_name || ''}
                                    onChange={(e) => setCurrentRoot({ ...currentRoot, last_name: e.target.value })}
                                    placeholder="Ej. Pérez"
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Segundo Apellido (Opcional)</FormLabel>
                                <Input
                                    value={currentRoot?.second_last_name || ''}
                                    onChange={(e) => setCurrentRoot({ ...currentRoot, second_last_name: e.target.value })}
                                    placeholder="Ej. González"
                                />
                            </FormControl>

                            <FormControl isRequired>
                                <FormLabel>Correo Electrónico</FormLabel>
                                <Input
                                    type="email"
                                    value={currentRoot?.email || ''}
                                    onChange={(e) => setCurrentRoot({ ...currentRoot, email: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Contraseña {currentRoot?.id && '(Dejar en blanco para no cambiar)'}</FormLabel>
                                <Input
                                    type="password"
                                    placeholder={currentRoot?.id ? '••••••••' : 'Ingresar contraseña'}
                                    value={currentRoot?.password || ''}
                                    onChange={(e) => setCurrentRoot({ ...currentRoot, password: e.target.value })}
                                />
                            </FormControl>
                        </Box>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionUsuariosRoot;



