import { useState } from 'react';
import {
    Box,
    Heading,
    Card,
    CardBody,
    FormControl,
    FormLabel,
    Input,
    Button,
    VStack,
    Text,
    useToast,
    List,
    ListItem,
    Alert,
    AlertIcon
} from '@chakra-ui/react';
import { updateUser } from '../../services/api';

const MiPerfil = () => {
    const [password, setPassword] = useState('');
    const [confirmPassword, setConfirmPassword] = useState('');
    const [loading, setLoading] = useState(false);
    const toast = useToast();

    const passwordRegex = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&._-])[A-Za-z\d@$!%*?&._-]{8,}$/;
    const isValidPassword = passwordRegex.test(password);
    const doPasswordsMatch = password === confirmPassword && password.length > 0;

    const handleUpdate = async () => {
        if (!isValidPassword || !doPasswordsMatch) return;
        setLoading(true);
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            if (user) {
                await updateUser(user.id, { password });
                toast({
                    title: "Contraseña actualizada",
                    status: "success",
                    duration: 3000,
                });
                setPassword('');
                setConfirmPassword('');
            }
        } catch (error) {
            toast({
                title: "Error",
                description: error.message,
                status: "error",
                duration: 4000,
            });
        } finally {
            setLoading(false);
        }
    };

    return (
        <Box maxW="600px" mx="auto" mt={8}>
            <Heading mb={6}>Mi Perfil</Heading>
            <Card>
                <CardBody>
                    <VStack spacing={6} align="stretch">
                        <Box>
                            <Heading size="md" mb={2}>Cambiar Contraseña</Heading>
                            <Text color="gray.600" fontSize="sm">Actualiza tu contraseña periódicamente por seguridad.</Text>
                        </Box>

                        <Alert status="info" borderRadius="md">
                            <AlertIcon />
                            <Box>
                                <Text fontWeight="bold">Requisitos:</Text>
                                <List spacing={1} mt={2} styleType="disc" ml={4} fontSize="sm">
                                    <ListItem>Mínimo 8 caracteres.</ListItem>
                                    <ListItem>Mayúscula, minúscula, número y carácter especial.</ListItem>
                                </List>
                            </Box>
                        </Alert>

                        <FormControl isRequired>
                            <FormLabel>Nueva Contraseña</FormLabel>
                            <Input
                                type="password"
                                value={password}
                                onChange={(e) => setPassword(e.target.value)}
                                placeholder="Nueva contraseña"
                            />
                        </FormControl>

                        <FormControl isRequired>
                            <FormLabel>Confirmar Contraseña</FormLabel>
                            <Input
                                type="password"
                                value={confirmPassword}
                                onChange={(e) => setConfirmPassword(e.target.value)}
                                placeholder="Repetir contraseña"
                            />
                        </FormControl>

                        <Button
                            colorScheme="brand"
                            onClick={handleUpdate}
                            isLoading={loading}
                            isDisabled={!isValidPassword || !doPasswordsMatch}
                        >
                            Actualizar Contraseña
                        </Button>
                    </VStack>
                </CardBody>
            </Card>
        </Box>
    );
};

export default MiPerfil;




