import { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import {
    Box,
    SimpleGrid,
    Stat,
    StatLabel,
    StatNumber,
    StatHelpText,
    Card,
    CardHeader,
    CardBody,
    Heading,
    Text,
    Icon,
    Flex,
    Progress,
    Button,
    Breadcrumb,
    BreadcrumbItem,
    BreadcrumbLink,
    Badge,
    VStack,
    HStack,
    Divider
} from '@chakra-ui/react';
import {
    FiUsers,
    FiUserCheck,
    FiUserX,
    FiHardDrive,
    FiActivity,
    FiArrowLeft,
    FiFileText,
    FiPieChart
} from 'react-icons/fi';
import { getCompanyStats, getCompanyById } from '../../services/api';

const ReportesEmpresa = () => {
    const { companyId } = useParams();
    const navigate = useNavigate();
    const [stats, setStats] = useState(null);
    const [company, setCompany] = useState(null);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        const fetchData = async () => {
            try {
                const [statsData, companyData] = await Promise.all([
                    getCompanyStats(companyId),
                    getCompanyById(companyId)
                ]);
                setStats(statsData);
                setCompany(companyData);
            } catch (error) {
                console.error("Error fetching company stats:", error);
            } finally {
                setLoading(false);
            }
        };
        fetchData();
    }, [companyId]);

    if (loading) return <Box p={8}><Text>Cargando estadísticas...</Text></Box>;
    if (!stats || !company) return <Box p={8}><Text>No se encontraron datos para esta empresa.</Text></Box>;

    return (
        <Box p={4}>
            <Flex align="center" mb={6}>
                <Button
                    leftIcon={<FiArrowLeft />}
                    variant="ghost"
                    onClick={() => navigate('/root')}
                    mr={4}
                >
                    Volver
                </Button>
                <VStack align="start" spacing={0}>
                    <Heading size="lg">Reporte: {company.name}</Heading>
                    <Text color="gray.500">Estadísticas detalladas de uso y recursos</Text>
                </VStack>
            </Flex>

            <Breadcrumb mb={8}>
                <BreadcrumbItem>
                    <BreadcrumbLink onClick={() => navigate('/root')}>Dashboard</BreadcrumbLink>
                </BreadcrumbItem>
                <BreadcrumbItem isCurrentPage>
                    <BreadcrumbLink>{company.name}</BreadcrumbLink>
                </BreadcrumbItem>
            </Breadcrumb>

            <SimpleGrid columns={{ base: 1, md: 2, lg: 4 }} spacing={6} mb={8}>
                <Card bg="blue.50">
                    <CardBody>
                        <Stat>
                            <Flex align="center" mb={2}>
                                <Icon as={FiUsers} color="blue.500" mr={2} />
                                <StatLabel>Estudiantes Totales</StatLabel>
                            </Flex>
                            <StatNumber>{stats.clients.active + stats.clients.inactive}</StatNumber>
                            <StatHelpText>
                                <Badge colorScheme="green" mr={2}>{stats.clients.active} Activos</Badge>
                                <Badge colorScheme="red">{stats.clients.inactive} Inactivos</Badge>
                            </StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card bg="purple.50">
                    <CardBody>
                        <Stat>
                            <Flex align="center" mb={2}>
                                <Icon as={FiUserCheck} color="purple.500" mr={2} />
                                <StatLabel>Profesores Totales</StatLabel>
                            </Flex>
                            <StatNumber>{stats.employees.active + stats.employees.inactive}</StatNumber>
                            <StatHelpText>
                                <Badge colorScheme="green" mr={2}>{stats.employees.active} Activos</Badge>
                                <Badge colorScheme="red">{stats.employees.inactive} Inactivos</Badge>
                            </StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card bg="orange.50">
                    <CardBody>
                        <Stat>
                            <Flex align="center" mb={2}>
                                <Icon as={FiFileText} color="orange.500" mr={2} />
                                <StatLabel>Exámenes Totales</StatLabel>
                            </Flex>
                            <StatNumber>{stats.usage.totalServices}</StatNumber>
                            <StatHelpText>Gestionados en el sistema</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card bg="teal.50">
                    <CardBody>
                        <Stat>
                            <Flex align="center" mb={2}>
                                <Icon as={FiHardDrive} color="teal.500" mr={2} />
                                <StatLabel>Espacio en Disco</StatLabel>
                            </Flex>
                            <StatNumber>{stats.storage.mb} MB</StatNumber>
                            <StatHelpText>{stats.storage.gb} GB utilizados</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>
            </SimpleGrid>

            <SimpleGrid columns={{ base: 1, lg: 2 }} spacing={8}>
                <Card>
                    <CardHeader>
                        <Heading size="md">Estado de Exámenes</Heading>
                    </CardHeader>
                    <CardBody>
                        <VStack spacing={4} align="stretch">
                            {Object.entries(stats.ordersByStatus).map(([status, count]) => (
                                <Box key={status}>
                                    <Flex justify="space-between" mb={1}>
                                        <Text fontSize="sm" fontWeight="bold">{status}</Text>
                                        <Text fontSize="sm">{count}</Text>
                                    </Flex>
                                    <Progress
                                        value={(count / stats.usage.totalServices) * 100}
                                        size="sm"
                                        colorScheme={
                                            status === 'FINISHED' ? 'green' :
                                                status === 'IN_PROGRESS' ? 'blue' :
                                                    status === 'PENDING' ? 'yellow' : 'gray'
                                        }
                                        borderRadius="full"
                                    />
                                </Box>
                            ))}
                            {Object.keys(stats.ordersByStatus).length === 0 && (
                                <Text color="gray.500">No hay exámenes registrados.</Text>
                            )}
                        </VStack>
                    </CardBody>
                </Card>

                <Card>
                    <CardHeader>
                        <Heading size="md">Uso de Recursos de Almacenamiento</Heading>
                    </CardHeader>
                    <CardBody>
                        <VStack spacing={6} align="stretch">
                            <Box>
                                <Flex justify="space-between" mb={2}>
                                    <Text fontWeight="medium">Cuota de Almacenamiento</Text>
                                    <Text>{stats.storage.mb} MB / 1024 MB</Text>
                                </Flex>
                                <Progress value={(stats.storage.mb / 1024) * 100} colorScheme="teal" borderRadius="full" height="20px" />
                                <Text fontSize="xs" color="gray.500" mt={2}>
                                    * Incluye logos, fotos y evidencias de exámenes asociados a esta institución.
                                </Text>
                            </Box>

                            <Divider />

                            <Box>
                                <Heading size="sm" mb={4}>Estadística de Actividad</Heading>
                                <SimpleGrid columns={2} spacing={4}>
                                    <Box p={3} border="1px solid" borderColor="gray.100" borderRadius="md">
                                        <Text fontSize="xs" color="gray.500">Promedio de exámenes por estudiante</Text>
                                        <Text fontSize="xl" fontWeight="bold">
                                            {(stats.usage.totalServices / (stats.clients.active || 1)).toFixed(1)}
                                        </Text>
                                    </Box>
                                    <Box p={3} border="1px solid" borderColor="gray.100" borderRadius="md">
                                        <Text fontSize="xs" color="gray.500">Carga por profesor</Text>
                                        <Text fontSize="xl" fontWeight="bold">
                                            {(stats.usage.totalServices / (stats.employees.active || 1)).toFixed(1)}
                                        </Text>
                                    </Box>
                                </SimpleGrid>
                            </Box>
                        </VStack>
                    </CardBody>
                </Card>
            </SimpleGrid>
        </Box>
    );
};

export default ReportesEmpresa;
