import { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import {
    Box,
    SimpleGrid,
    Stat,
    StatLabel,
    StatNumber,
    StatHelpText,
    StatArrow,
    Card,
    CardHeader,
    CardBody,
    Heading,
    Text,
    Icon,
    Flex,
    Progress,
    Button,
    Tooltip,
    HStack
} from '@chakra-ui/react';
import { FiBriefcase, FiUsers, FiCheckCircle, FiXCircle, FiBarChart2 } from 'react-icons/fi';
import { getCompanies, getUsers } from '../../services/api';

const RootDashboard = () => {
    const navigate = useNavigate();
    const [companies, setCompanies] = useState([]);
    const [users, setUsers] = useState([]);


    useEffect(() => {
        const fetchData = async () => {
            try {
                console.log("RootDashboard: Fetching data...");
                const fetchedCompanies = await getCompanies();
                console.log("RootDashboard: Companies received:", fetchedCompanies.length);
                setCompanies(fetchedCompanies);
                
                const fetchedUsers = await getUsers();
                console.log("RootDashboard: Users received:", fetchedUsers.length);
                setUsers(fetchedUsers);
            } catch (error) {
                console.error("Error fetching dashboard data:", error);
            }
        };
        fetchData();
    }, []);

    // Calcular estadísticas reales basadas en datos
    const totalCompanies = companies.length;
    const activeCompanies = companies.filter(c => c.is_active).length;
    const inactiveCompanies = totalCompanies - activeCompanies;

    const totalUsers = users.length;
    const activeUsers = users.filter(u => u.is_active).length;

    // Calcular porcentaje de ocupación (simulado)
    const storageUsed = 45; // %

    return (
        <Box>
            <Heading mb={6}>Dashboard General</Heading>

            {/* Stats Grid */}
            <SimpleGrid columns={{ base: 1, md: 2, lg: 4 }} spacing={6} mb={8}>
                <Card>
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="blue.100" borderRadius="md" mr={3}>
                                    <Icon as={FiBriefcase} color="blue.500" boxSize={6} />
                                </Box>
                                <StatLabel fontSize="lg">Instituciones</StatLabel>
                            </Flex>
                            <StatNumber>{totalCompanies}</StatNumber>
                            <StatHelpText>
                                <StatArrow type="increase" />
                                {activeCompanies} Activas
                            </StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card>
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="green.100" borderRadius="md" mr={3}>
                                    <Icon as={FiCheckCircle} color="green.500" boxSize={6} />
                                </Box>
                                <StatLabel fontSize="lg">Estado del Sistema</StatLabel>
                            </Flex>
                            <StatNumber>98.5%</StatNumber>
                            <StatHelpText>Uptime (Últimos 30 días)</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card>
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="purple.100" borderRadius="md" mr={3}>
                                    <Icon as={FiUsers} color="purple.500" boxSize={6} />
                                </Box>
                                <StatLabel fontSize="lg">Usuarios Totales</StatLabel>
                            </Flex>
                            <StatNumber>{totalUsers}</StatNumber>
                            <StatHelpText>
                                {activeUsers} Activos actualmente
                            </StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card>
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="orange.100" borderRadius="md" mr={3}>
                                    <Icon as={FiXCircle} color="orange.500" boxSize={6} />
                                </Box>
                                <StatLabel fontSize="lg">Instituciones Inactivas</StatLabel>
                            </Flex>
                            <StatNumber>{inactiveCompanies}</StatNumber>
                            <StatHelpText>Suspendidas por falta de pago</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>
            </SimpleGrid>

            {/* Main Content Grid */}
            <SimpleGrid columns={{ base: 1, lg: 2 }} spacing={8}>
                {/* Recent Activity / Companies List */}
                <Card>
                    <CardHeader>
                        <Heading size="md">Instituciones Registradas</Heading>
                    </CardHeader>
                    <CardBody>
                        <Box>
                            {companies.map((company) => (
                                <Flex
                                    key={company.id}
                                    justify="space-between"
                                    align="center"
                                    p={3}
                                    borderBottom="1px solid"
                                    borderColor="gray.100"
                                    _last={{ borderBottom: 'none' }}
                                >
                                    <Box>
                                        <Text fontWeight="bold">{company.name}</Text>
                                        <Text fontSize="sm" color="gray.500">{company.email}</Text>
                                    </Box>
                                    <HStack spacing={4}>
                                        <Box
                                            as="span"
                                            px={2}
                                            py={1}
                                            borderRadius="full"
                                            fontSize="xs"
                                            fontWeight="bold"
                                            bg={company.is_active ? "green.100" : "red.100"}
                                            color={company.is_active ? "green.800" : "red.800"}
                                        >
                                            {company.is_active ? "ACTIVA" : "INACTIVA"}
                                        </Box>
                                        <Tooltip label="Ver reporte detallado">
                                            <Button
                                                size="sm"
                                                variant="solid"
                                                colorScheme="blue"
                                                onClick={() => navigate(`/root/reporte/${company.id}`)}
                                                leftIcon={<FiBarChart2 />}
                                            >
                                                Ver Reporte
                                            </Button>

                                        </Tooltip>
                                    </HStack>

                                </Flex>
                            ))}
                        </Box>
                    </CardBody>
                </Card>

                {/* System Health / Resources */}
                <Card>
                    <CardHeader>
                        <Heading size="md">Recursos del Sistema</Heading>
                    </CardHeader>
                    <CardBody>
                        <Box mb={6}>
                            <Flex justify="space-between" mb={2}>
                                <Text fontWeight="medium">Almacenamiento</Text>
                                <Text color="gray.500">{storageUsed}%</Text>
                            </Flex>
                            <Progress value={storageUsed} colorScheme="blue" borderRadius="full" />
                            <Text fontSize="sm" color="gray.500" mt={1}>450 GB de 1 TB utilizados</Text>
                        </Box>

                        <Box mb={6}>
                            <Flex justify="space-between" mb={2}>
                                <Text fontWeight="medium">Ancho de Banda</Text>
                                <Text color="gray.500">23%</Text>
                            </Flex>
                            <Progress value={23} colorScheme="green" borderRadius="full" />
                            <Text fontSize="sm" color="gray.500" mt={1}>Tráfico normal</Text>
                        </Box>

                        <Box>
                            <Flex justify="space-between" mb={2}>
                                <Text fontWeight="medium">Base de Datos</Text>
                                <Text color="gray.500">12%</Text>
                            </Flex>
                            <Progress value={12} colorScheme="purple" borderRadius="full" />
                            <Text fontSize="sm" color="gray.500" mt={1}>Rendimiento óptimo</Text>
                        </Box>
                    </CardBody>
                </Card>
            </SimpleGrid>
        </Box>
    );
};

export default RootDashboard;
