
import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    SimpleGrid,
    Card,
    CardBody,
    CardHeader,
    Text,
    Button,
    useToast,
    Spinner,
    Center,
    VStack,
    HStack,
    Badge
} from '@chakra-ui/react';
import { useNavigate } from 'react-router-dom';
import { getServicesByCompany, getStudentExams } from '../../services/api';
import useServerTime from '../../hooks/useServerTime';

const AplicarExamen = () => {
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const [studentAssignments, setStudentAssignments] = useState([]);
    const [loading, setLoading] = useState(true);
    const toast = useToast();
    const navigate = useNavigate();
    const { currentTime } = useServerTime();

    // Función para formatear fechas en formato dd/mm/aaaa, hh:mm a.m./p.m.
    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        try {
            const date = new Date(dateString);
            if (isNaN(date.getTime())) return 'N/A';

            const day = String(date.getDate()).padStart(2, '0');
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const year = date.getFullYear();

            let hours = date.getHours();
            const minutes = String(date.getMinutes()).padStart(2, '0');
            const ampm = hours >= 12 ? 'p.m.' : 'a.m.';
            hours = hours % 12;
            hours = hours ? hours : 12; // 0 debería ser 12
            const formattedHours = String(hours).padStart(2, '0');

            return `${day}/${month}/${year}, ${formattedHours}:${minutes} ${ampm}`;
        } catch (e) {
            return 'N/A';
        }
    };

    useEffect(() => {
        const fetchData = async () => {
            try {
                setLoading(true);
                const user = JSON.parse(sessionStorage.getItem('user'));
                const instId = user?.institution_id || user?.company_id;

                if (user && instId) {
                    const [catalog, assignments] = await Promise.all([
                        getServicesByCompany(instId),
                        getStudentExams(user.id)
                    ]);

                    // Filtrar solo servicios (exámenes) activos
                    setServicesCatalog(catalog.filter(s => s.is_active));

                    // Filtrar asignaciones válidas (eliminar registros "fantasma" sin fecha programada y pendientes)
                    const validAssignments = assignments.filter(a => {
                        // Si está COMPLETED o IN_PROGRESS, siempre es válido
                        if (a.assignment_status === 'COMPLETED' || a.assignment_status === 'IN_PROGRESS') {
                            return true;
                        }
                        // Si está PENDING, solo es válido si tiene fecha programada
                        if (a.assignment_status === 'PENDING') {
                            return !!(a.scheduled_at || a.assignment_scheduled_at);
                        }
                        return true;
                    });

                    setStudentAssignments(validAssignments);
                } else {
                    console.error("No company ID found for user", user);
                    toast({
                        title: 'Error de sesión',
                        description: 'No se pudo identificar la institución del estudiante.',
                        status: 'error',
                        duration: 3000,
                    });
                }
            } catch (error) {
                console.error("Error fetching data:", error);
                toast({
                    title: 'Error al cargar los datos',
                    description: error.message,
                    status: 'error',
                    duration: 3000,
                });
            } finally {
                setLoading(false);
            }
        };
        fetchData();
    }, [toast]);

    const getAssignment = (examId) => {
        const assignments = studentAssignments.filter(a => a.exam_id === examId || a.id_exam === examId);
        if (assignments.length === 0) return null;

        // Ordenar por número de intento descendente
        const sorted = [...assignments].sort((a, b) => (b.attempt_number || 0) - (a.attempt_number || 0));

        const latest = sorted[0];

        // Si el último es PENDING pero NO tiene fecha programada, 
        // y existe uno previo COMPLETED, preferimos mostrar el COMPLETED.
        // Esto evita que "ghost records" sin fecha oculten el resultado anterior.
        if (latest.assignment_status === 'PENDING' && !latest.scheduled_at && !latest.assignment_scheduled_at) {
            const completed = sorted.find(a => a.assignment_status === 'COMPLETED');
            if (completed) return completed;
        }

        return latest;
    };

    // Filtrar solo exámenes que tienen asignaciones válidas
    const examsWithAssignments = servicesCatalog.filter(service => {
        const examId = service.id || service.id_exam;
        const assignment = studentAssignments.find(a =>
            (a.exam_id === examId || a.id_exam === examId)
        );
        // Solo mostrar si tiene una asignación válida
        return !!assignment;
    });

    // Agrupar exámenes por categoría (solo los que tienen asignaciones)
    const categories = [...new Set(examsWithAssignments.map(s => s.category))];

    if (loading) {
        return (
            <Center h="60vh">
                <Spinner size="xl" color="brand.500" thickness="4px" />
            </Center>
        );
    }

    return (
        <Box>
            <Heading mb={6}>Exámenes Disponibles para Aplicar</Heading>
            <Text mb={8} color="gray.600">
                Selecciona la evaluación que deseas realizar de tu catálogo institucional.
            </Text>

            {categories.length > 0 ? (
                categories.map((category, catIndex) => (
                    <Box key={`category-${category || 'uncategorized'}-${catIndex}`} mb={8}>
                        <Heading size="md" mb={4} color="brand.600">{category}</Heading>
                        <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={6}>
                            {examsWithAssignments
                                .filter(s => s.category === category)
                                .map((service) => {
                                    const examId = service.id || service.id_exam;
                                    const assignment = getAssignment(examId);
                                    const isCompleted = assignment?.assignment_status === 'COMPLETED';

                                    // Si hay una asignación PENDING sin fecha programada, no mostrar el botón de aplicar
                                    const hasInvalidAssignment = assignment &&
                                        assignment.assignment_status === 'PENDING' &&
                                        !assignment.scheduled_at &&
                                        !assignment.assignment_scheduled_at;

                                    return (
                                        <Card key={`exam-${examId}`} _hover={!isCompleted && !hasInvalidAssignment ? { shadow: 'md', transform: 'translateY(-2px)', transition: 'all 0.2s' } : {}}>
                                            <CardHeader pb={2} display="flex" justifyContent="space-between" alignItems="start">
                                                <Heading size="sm">{service.title || service.name}</Heading>
                                                {isCompleted && (
                                                    <Badge colorScheme="green" variant="solid">Aplicado</Badge>
                                                )}
                                            </CardHeader>
                                            <CardBody>
                                                <Text fontSize="sm" color="gray.600" mb={4} noOfLines={3}>
                                                    {service.description || 'Sin descripción adicional.'}
                                                </Text>

                                                {isCompleted && assignment ? (
                                                    <Box p={3} bg="gray.50" borderRadius="md" mb={4} border="1px solid" borderColor="gray.200">
                                                        <VStack align="stretch" spacing={1}>
                                                            <Text fontSize="xs" fontWeight="bold" color="gray.500">RESUMEN DE APLICACIÓN:</Text>
                                                            <HStack justify="space-between" fontSize="xs">
                                                                <Text>Inicio:</Text>
                                                                <Text fontWeight="bold">{formatDate(assignment.started_at)}</Text>
                                                            </HStack>
                                                            <HStack justify="space-between" fontSize="xs">
                                                                <Text>Fin:</Text>
                                                                <Text fontWeight="bold">{formatDate(assignment.completed_at)}</Text>
                                                            </HStack>
                                                            <HStack justify="space-between" fontSize="xs" pt={1} borderTop="1px dashed" borderColor="gray.300">
                                                                <Text>Calificación:</Text>
                                                                <Text fontWeight="bold" color="brand.600">{assignment.score !== null && assignment.score !== undefined ? `${Number(parseFloat(assignment.score).toFixed(2))} / 10` : 'Pendiente'}</Text>
                                                            </HStack>
                                                        </VStack>
                                                    </Box>
                                                ) : (
                                                    <VStack align="stretch" spacing={2} mb={4}>
                                                        {assignment && (assignment.scheduled_at || assignment.assignment_scheduled_at) && (
                                                            <Box p={2} bg="blue.50" borderRadius="md" border="1px solid" borderColor="blue.200" mb={2}>
                                                                <HStack spacing={1} fontSize="xs">
                                                                    <Text color="blue.700" fontWeight="bold">📅 Programado:</Text>
                                                                    <Text color="blue.800" fontWeight="semibold">
                                                                        {assignment.assignment_scheduled_end_at || assignment.scheduled_end_at
                                                                            ? `${formatDate(assignment.assignment_scheduled_at || assignment.scheduled_at)} - ${formatDate(assignment.assignment_scheduled_end_at || assignment.scheduled_end_at)}`
                                                                            : formatDate(assignment.assignment_scheduled_at || assignment.scheduled_at)
                                                                        }
                                                                    </Text>
                                                                </HStack>
                                                            </Box>
                                                        )}
                                                        <HStack justify="space-between" fontSize="xs">
                                                            <Text color="gray.500">Duración:</Text>
                                                            <Text fontWeight="bold">{service.settings?.timer_minutes || 60} min.</Text>
                                                        </HStack>
                                                        <HStack justify="space-between" fontSize="xs">
                                                            <Text color="gray.500">Tolerancia:</Text>
                                                            <Text fontWeight="bold">{service.tolerance_minutes || 15} min.</Text>
                                                        </HStack>

                                                        {/* Hora Servidor Display requested by user */}
                                                        <HStack justify="space-between" fontSize="xs" pt={2} borderTop="1px dashed" borderColor="gray.200">
                                                            <Text color="gray.500">Hora Servidor:</Text>
                                                            <Badge colorScheme="purple" variant="outline" fontSize="xs">
                                                                {currentTime.toLocaleTimeString('en-US', { hour: '2-digit', minute: '2-digit', second: '2-digit', hour12: true })}
                                                            </Badge>
                                                        </HStack>
                                                    </VStack>
                                                )}

                                                {!isCompleted && !hasInvalidAssignment && (
                                                    <Button
                                                        colorScheme="brand"
                                                        width="full"
                                                        variant="solid"
                                                        onClick={() => navigate(`/student/examen/${examId}`)}
                                                    >
                                                        Comenzar Evaluación
                                                    </Button>
                                                )}

                                                {isCompleted && (
                                                    <Text fontSize="xs" color="gray.400" textAlign="center" fontStyle="italic">
                                                        Este examen ya ha sido aplicado y se encuentra en revisión.
                                                    </Text>
                                                )}

                                                {hasInvalidAssignment && (
                                                    <Text fontSize="xs" color="orange.500" textAlign="center" fontStyle="italic" fontWeight="medium">
                                                        Este examen está pendiente de programación por tu profesor.
                                                    </Text>
                                                )}
                                            </CardBody>
                                        </Card>
                                    );
                                })}
                        </SimpleGrid>
                    </Box>
                ))
            ) : (
                <Center py={20} bg="gray.50" borderRadius="xl" border="2px dashed" borderColor="gray.200">
                    <VStack spacing={3}>
                        <Text color="gray.500" fontSize="lg" fontWeight="medium">
                            No tienes exámenes asignados
                        </Text>
                        <Text color="gray.400" fontSize="sm">
                            Tu profesor aún no te ha asignado ningún examen. Contacta con él para más información.
                        </Text>
                    </VStack>
                </Center>
            )}
        </Box>
    );
};

export default AplicarExamen;
