
import { useState, useEffect, useCallback } from 'react';
import {
    Box,
    VStack,
    Heading,
    Text,
    Button,
    Radio,
    RadioGroup,
    Stack,
    Card,
    CardBody,
    useToast,
    Progress,
    HStack,
    Flex,
    Icon,
    Spinner,
    Alert,
    AlertIcon,
    Badge,
    Center,
    Textarea,
    Image,
    Divider,
    Container
} from '@chakra-ui/react';
import { useParams, useNavigate } from 'react-router-dom';
import { io } from 'socket.io-client';
import { BASE_URL, getExamById, startExam, submitExam, getCompanyById, saveResponse, getExamResponses } from '../../services/api';
import { FiClock, FiArrowRight, FiArrowLeft, FiCheckCircle } from 'react-icons/fi';

const ExamenActivo = () => {
    const { examId } = useParams();
    const navigate = useNavigate();
    const [exam, setExam] = useState(null);
    const [company, setCompany] = useState(null);
    const [questions, setQuestions] = useState([]);
    const [currentIndex, setCurrentIndex] = useState(0);
    const [answers, setAnswers] = useState({});
    const [timeLeft, setTimeLeft] = useState(null); // en segundos
    const [isLoading, setIsLoading] = useState(true);
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [isSavingProgress, setIsSavingProgress] = useState(false);
    const [hasFinished, setHasFinished] = useState(false); // Prevenir bucles infinitos
    const [socket, setSocket] = useState(null);
    const [examStarted, setExamStarted] = useState(false);
    const toast = useToast();

    const user = JSON.parse(sessionStorage.getItem('user'));
    const studentId = user?.id || user?.user_id;

    const handleProgressSave = async (questionId, value) => {
        // No guardar si ya terminó
        if (!exam?.assignment?.id_assignment || isSubmitting || hasFinished) return;

        // Optimistic update: actualizamos el estado local inmediatamente
        setAnswers(prev => ({ ...prev, [questionId]: value }));

        try {
            setIsSavingProgress(true);
            await saveResponse(exam.assignment.id_assignment, questionId, value);
        } catch (error) {
            console.error('Error al guardar progreso:', error);
            // Opcional: Mostrar indicador de error discreto
        } finally {
            setIsSavingProgress(false);
        }
    };

    const handleFinish = useCallback(async (isAuto = false) => {
        if (isSubmitting || hasFinished) return;
        setIsSubmitting(true);

        try {
            if (isAuto) {
                toast({
                    title: 'Tiempo agotado',
                    description: 'Tu examen se ha enviado automáticamente.',
                    status: 'info',
                    duration: 5000,
                });
            }

            const result = await submitExam(examId, answers);
            setHasFinished(true);

            // Emitir evento de finalización de examen
            if (socket) {
                socket.emit('exam-event', {
                    examId,
                    studentId,
                    type: 'EXAMEN_FINALIZADO',
                    timestamp: new Date(),
                    score: result.score
                });
            }

            toast({
                title: 'Examen completado',
                description: `Tu calificación final es: ${Number(parseFloat(result.score).toFixed(2))}`,
                status: 'success',
                duration: 5000,
            });

            navigate('/student');
        } catch (error) {
            console.error('Error al enviar examen:', error);

            // Si ya estaba completado, redirigir igualmente
            if (error.message.includes('completed') || error.message.includes('completado')) {
                setHasFinished(true);
                navigate('/student');
                return;
            }

            toast({
                title: 'Error al enviar',
                description: error.message,
                status: 'error',
            });

            // Solo resetear isSubmitting si NO fue por tiempo agotado
            if (!isAuto) {
                setIsSubmitting(false);
            } else {
                // Si fue por tiempo y falló, marcamos como terminado igualmente para no reintentar infinitamente
                setHasFinished(true);
            }
        }
    }, [examId, answers, navigate, toast, isSubmitting, hasFinished]);

    useEffect(() => {
        const fetchExamData = async () => {
            try {
                const data = await getExamById(examId);
                setExam(data);
                setQuestions(data.questions || []);

                // Fetch company info for branding
                const instId = user?.institution_id || user?.company_id;
                if (instId) {
                    const companyData = await getCompanyById(instId);
                    setCompany(companyData);
                }

                // Usar tiempo persistente del servidor o el tiempo total si es primera vez
                if (data.timeLeftSeconds !== undefined) {
                    setTimeLeft(data.timeLeftSeconds);
                } else {
                    const timerMinutes = parseInt(data.settings?.timer_minutes) || 60;
                    setTimeLeft(timerMinutes * 60);
                }

                // Marcar inicio en el servidor si es la primera vez
                if (data.assignment?.status === 'PENDING') {
                    await startExam(examId);
                    setExamStarted(true);
                } else if (data.assignment?.status === 'IN_PROGRESS') {
                    // Si ya estaba en progreso, también marcamos como iniciado
                    setExamStarted(true);
                }

                // Cargar respuestas ya guardadas (si existen)
                if (data.assignment?.id_assignment) {
                    const savedResponses = await getExamResponses(data.assignment.id_assignment);
                    const answersMap = {};
                    savedResponses.forEach(r => {
                        answersMap[r.question_id] = r.student_answer;
                    });
                    setAnswers(answersMap);
                }

                setIsLoading(false);
            } catch (error) {
                console.error('Error loading exam:', error);
                toast({
                    title: 'Error',
                    description: error.message || 'No se pudo cargar el examen o ya ha sido completado.',
                    status: 'error',
                    duration: 5000,
                });
                navigate('/student');
            }
        };

        if (examId) {
            fetchExamData();
        }
    }, [examId, navigate, toast]);

    // Socket y Lockdown
    useEffect(() => {
        if (!examId || !studentId) return;

        const newSocket = io(BASE_URL);
        setSocket(newSocket);

        newSocket.emit('join-exam', examId);

        // Notificar que el estudiante ha entrado al examen
        newSocket.emit('exam-event', {
            examId,
            studentId,
            type: 'CONECTADO',
            timestamp: new Date()
        });

        const handleVisibilityChange = () => {
            if (document.hidden) {
                newSocket.emit('exam-event', {
                    examId,
                    studentId,
                    type: 'TAB_CHANGE',
                    timestamp: new Date()
                });
                toast({
                    title: '¡Alerta!',
                    description: 'Se ha detectado que saliste de la pestaña del examen. Esto queda registrado.',
                    status: 'warning',
                    duration: 5000,
                });
            }
        };

        const handleBlur = () => {
            newSocket.emit('exam-event', {
                examId,
                studentId,
                type: 'FOCUS_LOST',
                timestamp: new Date()
            });
        };

        document.addEventListener('visibilitychange', handleVisibilityChange);
        window.addEventListener('blur', handleBlur);

        // Heartbeat periódico
        const heartbeat = setInterval(() => {
            newSocket.emit('exam-event', {
                examId,
                studentId,
                type: 'HEARTBEAT',
                timestamp: new Date()
            });
        }, 30000);

        return () => {
            document.removeEventListener('visibilitychange', handleVisibilityChange);
            window.removeEventListener('blur', handleBlur);
            clearInterval(heartbeat);
            newSocket.disconnect();
        };
    }, [examId, studentId, toast]);

    // Emitir evento de inicio cuando el socket esté listo y el examen haya sido iniciado
    useEffect(() => {
        if (socket && examStarted && examId && studentId) {
            socket.emit('exam-event', {
                examId,
                studentId,
                type: 'EXAMEN_INICIADO',
                timestamp: new Date()
            });
        }
    }, [socket, examStarted, examId, studentId]);

    // Lógica del Timer
    useEffect(() => {
        if (timeLeft === null || isSubmitting || hasFinished) return;

        if (timeLeft <= 0) {
            handleFinish(true);
            return;
        }

        const timer = setInterval(() => {
            setTimeLeft(prev => prev - 1);
        }, 1000);

        return () => clearInterval(timer);
    }, [timeLeft, handleFinish, isSubmitting, hasFinished]);

    const formatTime = (seconds) => {
        const h = Math.floor(seconds / 3600);
        const m = Math.floor((seconds % 3600) / 60);
        const s = seconds % 60;
        return `${h > 0 ? h + ':' : ''}${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`;
    };

    if (isLoading) {
        return (
            <Center h="100vh">
                <VStack spacing={4}>
                    <Spinner size="xl" color="brand.500" thickness="4px" />
                    <Text color="gray.500">Preparando tu evaluación...</Text>
                </VStack>
            </Center>
        );
    }

    const currentQuestion = questions[currentIndex];
    const progress = ((currentIndex + 1) / questions.length) * 100;
    const content = typeof currentQuestion?.content === 'string' ? JSON.parse(currentQuestion.content) : currentQuestion?.content;

    return (
        <Box minH="100vh" bg="gray.50">
            {/* Logo y Encabezado de Institución */}
            <Box bg="white" borderBottom="1px solid" borderColor="gray.200" py={4} boxShadow="sm">
                <Container maxW="800px">
                    <Flex align="center" justify="space-between">
                        <HStack spacing={4}>
                            <Image
                                src={company?.logo_url || '/logo.jpeg'}
                                fallbackSrc="/logo.jpeg"
                                alt="Logo Institución"
                                h="60px"
                                objectFit="contain"
                            />
                            <VStack align="start" spacing={0}>
                                <Heading size="sm" color="gray.700">
                                    {company?.name || 'KeQuiz'}
                                </Heading>
                                {(company?.tagline || company?.slogan) && (
                                    <Text fontSize="xs" color="gray.500 italic">
                                        {company.tagline || company.slogan}
                                    </Text>
                                )}
                            </VStack>
                        </HStack>
                        <Badge colorScheme="brand" variant="solid" px={3} py={1} borderRadius="full">
                            EXAMEN EN CURSO
                        </Badge>
                    </Flex>
                </Container>
            </Box>

            <VStack spacing={6} maxW="800px" mx="auto" align="stretch" py={8} px={4}>
                {/* Header info */}
                <Card variant="solid" bg="white" boxShadow="sm">
                    <CardBody py={4}>
                        <Flex justify="space-between" align="center">
                            <VStack align="start" spacing={0}>
                                <Heading size="md" color="gray.800">{exam.title}</Heading>
                                <HStack spacing={2}>
                                    <Text fontSize="xs" color="gray.500">Evaluación en curso</Text>
                                    {isSavingProgress && (
                                        <HStack spacing={1}>
                                            <Spinner size="xs" color="brand.500" />
                                            <Text fontSize="10px" color="brand.500" fontWeight="bold">Guardando...</Text>
                                        </HStack>
                                    )}
                                </HStack>
                            </VStack>
                            <HStack bg={timeLeft < 60 ? 'red.50' : 'blue.50'} px={4} py={2} borderRadius="lg" border="1px solid" borderColor={timeLeft < 60 ? 'red.200' : 'blue.200'}>
                                <Icon as={FiClock} color={timeLeft < 60 ? 'red.500' : 'blue.500'} />
                                <Text fontWeight="bold" color={timeLeft < 60 ? 'red.600' : 'blue.600'} fontFamily="mono" fontSize="lg">
                                    {formatTime(timeLeft)}
                                </Text>
                            </HStack>
                        </Flex>
                    </CardBody>
                </Card>

                <Alert status="warning" borderRadius="lg" fontSize="sm" variant="subtle">
                    <AlertIcon />
                    No cierres esta ventana ni cambies de pestaña. Tu actividad está siendo monitoreada.
                </Alert>

                <Box px={2}>
                    <Flex justify="space-between" mb={2}>
                        <Text fontSize="xs" fontWeight="bold" color="gray.500">PROGRESO</Text>
                        <Text fontSize="xs" fontWeight="bold" color="gray.500">{currentIndex + 1} de {questions.length}</Text>
                    </Flex>
                    <Progress value={progress} colorScheme="brand" size="sm" borderRadius="full" />
                </Box>

                {/* Pregunta Actual */}
                <Card variant="outline" boxShadow="md" bg="white" borderRadius="xl">
                    <CardBody p={8}>
                        <VStack spacing={8} align="stretch">
                            <Box>
                                <Badge colorScheme="brand" mb={2} variant="subtle">Pregunta {currentIndex + 1}</Badge>
                                <Heading size="md" lineHeight="tall">
                                    {content?.question}
                                </Heading>
                            </Box>

                            {currentQuestion.type === 'MULTIPLE_CHOICE' || currentQuestion.type === 'TRUE_FALSE' ? (
                                <RadioGroup
                                    key={`question-${currentQuestion.id_question}-${currentIndex}`}
                                    onChange={(val) => handleProgressSave(currentQuestion.id_question, val)}
                                    value={answers[currentQuestion.id_question] || undefined}
                                >
                                    <Stack spacing={4}>
                                        {content?.options?.map((option, idx) => {
                                            const currentAnswer = answers[currentQuestion.id_question];
                                            const isSelected = currentAnswer !== undefined && currentAnswer !== null && currentAnswer === idx.toString();

                                            return (
                                                <Box
                                                    key={idx}
                                                    p={4}
                                                    border="1px solid"
                                                    borderColor={isSelected ? 'brand.500' : 'gray.200'}
                                                    bg={isSelected ? 'brand.50' : 'white'}
                                                    borderRadius="lg"
                                                    cursor="pointer"
                                                    onClick={() => handleProgressSave(currentQuestion.id_question, idx.toString())}
                                                    _hover={{ borderColor: 'brand.200', bg: 'gray.50' }}
                                                    transition="all 0.2s"
                                                >
                                                    <Radio value={idx.toString()} colorScheme="brand" width="full">
                                                        <Text ml={2}>{option}</Text>
                                                    </Radio>
                                                </Box>
                                            );
                                        })}
                                    </Stack>
                                </RadioGroup>
                            ) : (
                                <Box>
                                    <Textarea
                                        key={`textarea-${currentQuestion.id_question}-${currentIndex}`}
                                        placeholder="Escribe tu respuesta aquí..."
                                        size="lg"
                                        minH="200px"
                                        value={answers[currentQuestion.id_question] || ''}
                                        onChange={(e) => setAnswers({ ...answers, [currentQuestion.id_question]: e.target.value })}
                                        onBlur={(e) => handleProgressSave(currentQuestion.id_question, e.target.value)}
                                        bg="white"
                                        _focus={{ borderColor: 'brand.500', boxShadow: 'none' }}
                                    />
                                    <Text fontSize="xs" color="gray.500" mt={2}>
                                        Esta pregunta será calificada manualmente por tu profesor.
                                    </Text>
                                </Box>
                            )}
                        </VStack>
                    </CardBody>
                </Card>

                {/* Navegación */}
                <HStack justify="space-between" pt={4}>
                    <Button
                        variant="ghost"
                        leftIcon={<FiArrowLeft />}
                        onClick={() => setCurrentIndex(prev => prev - 1)}
                        isDisabled={currentIndex === 0 || isSubmitting}
                    >
                        Anterior
                    </Button>

                    {currentIndex < questions.length - 1 ? (
                        <Button
                            colorScheme="brand"
                            rightIcon={<FiArrowRight />}
                            onClick={() => setCurrentIndex(prev => prev + 1)}
                            isDisabled={isSubmitting}
                            px={8}
                        >
                            Siguiente
                        </Button>
                    ) : (
                        <Button
                            colorScheme="green"
                            leftIcon={<FiCheckCircle />}
                            onClick={() => handleFinish(false)}
                            isLoading={isSubmitting}
                            loadingText="Enviando..."
                            px={10}
                        >
                            Finalizar Examen
                        </Button>
                    )}
                </HStack>
            </VStack>

            {/* Footer de la Institución */}
            <Box py={10} mt={10} bg="white" borderTop="1px solid" borderColor="gray.200">
                <Container maxW="800px" textAlign="center">
                    {company?.exam_footer || company?.service_completion_message ? (
                        <Text fontSize="sm" color="gray.600" whiteSpace="pre-wrap">
                            {company.exam_footer || company.service_completion_message}
                        </Text>
                    ) : (
                        <Text fontSize="xs" color="gray.400">
                            © {new Date().getFullYear()} {company?.name || 'KeQuiz'} • Todos los derechos reservados
                        </Text>
                    )}
                </Container>
            </Box>
        </Box>
    );
};

export default ExamenActivo;
