
import {
    Box,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    Button,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    VStack,
    HStack,
    Text,
    Divider,
    SimpleGrid,
    useToast,
    Spinner,
    Icon
} from '@chakra-ui/react';
import { useState, useEffect } from 'react';
import { FiEye, FiCheckCircle, FiXCircle, FiClock, FiFileText } from 'react-icons/fi';
import { getStudentExams, getServicesByCompany } from '../../services/api';

const HistorialExamenes = () => {
    const [myExams, setMyExams] = useState([]);
    const [loading, setLoading] = useState(true);
    const [selectedExam, setSelectedExam] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    useEffect(() => {
        const fetchData = async () => {
            try {
                setLoading(true);
                const user = JSON.parse(sessionStorage.getItem('user'));
                const instId = user?.institution_id || user?.company_id;
                const userId = user?.id || user?.user_id;

                if (userId) {
                    const exams = await getStudentExams(userId);
                    setMyExams(Array.isArray(exams) ? exams : []);
                }
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar historial',
                    status: 'error',
                    duration: 3000,
                });
            } finally {
                setLoading(false);
            }
        };
        fetchData();
    }, []);

    const handleViewDetails = (exam) => {
        setSelectedExam(exam);
        onOpen();
    };

    const getStatusBadge = (status) => {
        switch (status) {
            case 'COMPLETED': return <Badge colorScheme="green"><Icon as={FiCheckCircle} mr={1} />Finalizado</Badge>;
            case 'IN_PROGRESS': return <Badge colorScheme="blue"><Icon as={FiClock} mr={1} />En Curso</Badge>;
            case 'PENDING': return <Badge colorScheme="orange"><Icon as={FiClock} mr={1} />Pendiente</Badge>;
            case 'CANCELLED': return <Badge colorScheme="red"><Icon as={FiXCircle} mr={1} />Cancelado</Badge>;
            default: return <Badge>{status}</Badge>;
        }
    };

    return (
        <Box>
            <Heading mb={6}>Mi Historial de Exámenes</Heading>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto" border="1px solid" borderColor="gray.100">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>ID</Th>
                            <Th>Examen</Th>
                            <Th>Fecha de Aplicación</Th>
                            <Th>Calificación</Th>
                            <Th>Estado</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {loading ? (
                            <Tr>
                                <Td colSpan={6} textAlign="center" py={10}>
                                    <Spinner color="brand.500" />
                                    <Text mt={2} color="gray.500">Cargando historial...</Text>
                                </Td>
                            </Tr>
                        ) : myExams.length > 0 ? (
                            myExams.map((exam) => (
                                <Tr key={exam.id_assignment}>
                                    <Td fontWeight="bold">#{exam.id_exam || exam.id}</Td>
                                    <Td>
                                        <VStack align="start" spacing={0}>
                                            <HStack>
                                                <Text fontWeight="medium">{exam.title}</Text>
                                                {exam.attempt_number > 1 && (
                                                    <Badge colorScheme="gray" fontSize="9px">Intento #{exam.attempt_number}</Badge>
                                                )}
                                            </HStack>
                                            <Text fontSize="xs" color="gray.500" noOfLines={1}>{exam.description || 'Sin descripción'}</Text>
                                        </VStack>
                                    </Td>
                                    <Td fontSize="sm">
                                        {exam.completed_at 
                                            ? new Date(exam.completed_at).toLocaleString() 
                                            : exam.started_at 
                                                ? `Iniciado: ${new Date(exam.started_at).toLocaleString()}`
                                                : (exam.assignment_scheduled_at || exam.scheduled_at)
                                                    ? `Prog: ${new Date(exam.assignment_scheduled_at || exam.scheduled_at).toLocaleString()}`
                                                    : 'No programada'}
                                    </Td>
                                    <Td>
                                        {exam.score !== null ? (
                                            <Badge fontSize="md" colorScheme={parseFloat(exam.score) >= 7 ? 'green' : 'red'} variant="subtle" px={3}>
                                                {Number(parseFloat(exam.score).toFixed(2))} / 10
                                            </Badge>
                                        ) : '-'}
                                    </Td>
                                    <Td>
                                        {getStatusBadge(exam.assignment_status)}
                                    </Td>
                                    <Td>
                                        <Button
                                            leftIcon={<FiEye />}
                                            size="sm"
                                            variant="ghost"
                                            colorScheme="brand"
                                            onClick={() => handleViewDetails(exam)}
                                        >
                                            Detalles
                                        </Button>
                                    </Td>
                                </Tr>
                            ))
                        ) : (
                            <Tr>
                                <Td colSpan={6} textAlign="center" py={10}>
                                    <Text color="gray.500">Aún no tienes exámenes realizados.</Text>
                                </Td>
                            </Tr>
                        )}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal Detalles */}
            <Modal isOpen={isOpen} onClose={onClose} size="lg">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Detalles de la Evaluación</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        {selectedExam && (
                            <VStack align="stretch" spacing={6} py={2}>
                                <Box>
                                    <HStack justify="space-between" align="start">
                                        <VStack align="start" spacing={1}>
                                            <Text fontSize="xs" fontWeight="bold" color="gray.500" textTransform="uppercase" mb={1}>Examen</Text>
                                            <Heading size="md">{selectedExam.title}</Heading>
                                        </VStack>
                                        <Badge colorScheme="brand" px={3} py={1} borderRadius="md">
                                            Intento #{selectedExam.attempt_number}
                                        </Badge>
                                    </HStack>
                                    <Text color="gray.600" mt={2}>{selectedExam.description || 'Sin descripción'}</Text>
                                </Box>

                                <Divider />

                                <SimpleGrid columns={2} spacing={6}>
                                    <Box>
                                        <Text fontSize="xs" fontWeight="bold" color="gray.500" textTransform="uppercase" mb={1}>Estado</Text>
                                        {getStatusBadge(selectedExam.assignment_status)}
                                    </Box>
                                    <Box>
                                        <Text fontSize="xs" fontWeight="bold" color="gray.500" textTransform="uppercase" mb={1}>Calificación</Text>
                                        <Text fontSize="2xl" fontWeight="bold" color={parseFloat(selectedExam.score) >= 7 ? 'green.500' : 'red.500'}>
                                            {selectedExam.score !== null ? `${Number(parseFloat(selectedExam.score).toFixed(2))} / 10` : 'N/A'}
                                        </Text>
                                    </Box>
                                </SimpleGrid>

                                <SimpleGrid columns={2} spacing={6}>
                                    <Box>
                                        <Text fontSize="xs" fontWeight="bold" color="gray.500" textTransform="uppercase" mb={1}>Fecha de Inicio</Text>
                                        <Text fontSize="sm">{selectedExam.started_at ? new Date(selectedExam.started_at).toLocaleString() : 'No iniciada'}</Text>
                                    </Box>
                                    <Box>
                                        <Text fontSize="xs" fontWeight="bold" color="gray.500" textTransform="uppercase" mb={1}>Fecha de Finalización</Text>
                                        <Text fontSize="sm">{selectedExam.completed_at ? new Date(selectedExam.completed_at).toLocaleString() : 'No finalizada'}</Text>
                                    </Box>
                                </SimpleGrid>

                                {selectedExam.assignment_status === 'COMPLETED' && (
                                    <Box bg="blue.50" p={4} borderRadius="lg" border="1px solid" borderColor="blue.100">
                                        <HStack spacing={3}>
                                            <Icon as={FiFileText} color="blue.500" boxSize={5} />
                                            <Box>
                                                <Text fontWeight="bold" fontSize="sm" color="blue.800">Reporte de Resultados</Text>
                                                <Text fontSize="xs" color="blue.600">Puedes solicitar tu constancia de resultados con tu profesor.</Text>
                                            </Box>
                                        </HStack>
                                    </Box>
                                )}
                            </VStack>
                        )}
                    </ModalBody>
                    <ModalFooter>
                        <Button colorScheme="brand" onClick={onClose}>
                            Cerrar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default HistorialExamenes;
