
import { useState, useEffect } from 'react';
import {
    Box,
    SimpleGrid,
    Stat,
    StatLabel,
    StatNumber,
    StatHelpText,
    Card,
    CardBody,
    Heading,
    Text,
    Icon,
    Flex,
    Badge,
    VStack,
    Button,
    HStack,
    useToast,
    Spinner,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td
} from '@chakra-ui/react';
import { FiClock, FiCheckCircle, FiEdit3, FiBook, FiArrowRight } from 'react-icons/fi';
import { Link as RouterLink } from 'react-router-dom';
import { getStudentExams } from '../../services/api';
import useServerTime from '../../hooks/useServerTime';

const TimeStatusBadge = ({ scheduledAt, currentTime }) => {
    if (!scheduledAt || !currentTime) return null;

    const start = new Date(scheduledAt).getTime();
    const now = currentTime.getTime();
    const diff = now - start;
    const isPast = diff > 0;

    // Función para formatear duración
    const formatDuration = (ms) => {
        const totalSeconds = Math.floor(ms / 1000);
        const seconds = totalSeconds % 60;
        const minutes = Math.floor(totalSeconds / 60) % 60;
        const hours = Math.floor(totalSeconds / 3600);

        const pad = (n) => n.toString().padStart(2, '0');

        // Formato HH:MM:SS, si hay días los sumamos a las horas o mostramos días
        // El usuario pidió "hh:mm:ss", así que intentaremos mantenerlo simple si es < 24h
        // Si es mayor, p.ej 25:00:00 o 1d 01:00:00. Usaremos formato extendido si incluye días para claridad.

        if (hours >= 24) {
            const days = Math.floor(totalSeconds / 86400);
            const hoursInDay = Math.floor((totalSeconds % 86400) / 3600);
            return `${days}d ${pad(hoursInDay)}:${pad(minutes)}:${pad(seconds)}`;
        }

        return `${pad(hours)}:${pad(minutes)}:${pad(seconds)}`;
    };

    if (isPast) {
        return (
            <Badge colorScheme="purple" variant="solid" fontSize="10px" mt={1}>
                Tiempo transcurrido: {formatDuration(diff)}
            </Badge>
        );
    } else {
        return (
            <Badge colorScheme="orange" variant="solid" fontSize="10px" mt={1}>
                Comienza en: {formatDuration(Math.abs(diff))}
            </Badge>
        );
    }
};

const StudentDashboard = () => {
    const [myExams, setMyExams] = useState([]);
    const [loading, setLoading] = useState(true);
    const { currentTime } = useServerTime();
    const toast = useToast();

    useEffect(() => {
        const fetchData = async () => {
            try {
                setLoading(true);
                const user = JSON.parse(sessionStorage.getItem('user'));
                const userId = user?.id || user?.user_id;
                if (userId) {
                    const data = await getStudentExams(userId);
                    setMyExams(Array.isArray(data) ? data : []);
                }
            } catch (error) {
                // No mostrar error si es de autenticación (401/403) - se redirige automáticamente al login
                if (error.message && (error.message.includes('Invalid token') || error.message.includes('403') || error.message.includes('401'))) {
                    return; // La redirección al login se maneja en api.js
                }
                console.error(error);
                toast({
                    title: 'Error al cargar dashboard',
                    description: error.message || 'No se pudieron cargar los datos',
                    status: 'error',
                    duration: 3000,
                });
            } finally {
                setLoading(false);
            }
        };
        fetchData();
    }, []);

    // Agrupar por examen para mostrar el último intento arriba y el historial abajo
    const examsGrouped = myExams.reduce((acc, curr) => {
        if (!acc[curr.id_exam]) acc[curr.id_exam] = [];
        acc[curr.id_exam].push(curr);
        return acc;
    }, {});

    const latestAttempts = Object.values(examsGrouped).map(group =>
        group.sort((a, b) => b.attempt_number - a.attempt_number)[0]
    );

    const pendingExams = latestAttempts.filter(e => e.assignment_status === 'PENDING' || e.assignment_status === 'IN_PROGRESS');
    const finishedExams = myExams.filter(e => e.assignment_status === 'COMPLETED');
    const averageScore = finishedExams.length > 0
        ? Number((finishedExams.reduce((acc, curr) => acc + (parseFloat(curr.score) || 0), 0) / finishedExams.length).toFixed(2))
        : '0.0';

    const formatDate = (dateStr) => {
        try {
            if (!dateStr) return 'N/A';
            const date = new Date(dateStr);
            if (isNaN(date.getTime())) return 'Fecha inválida';
            return date.toLocaleString();
        } catch (e) {
            return 'N/A';
        }
    };

    if (loading) {
        return (
            <Flex justify="center" align="center" h="60vh">
                <Spinner size="xl" color="brand.500" thickness="4px" />
            </Flex>
        );
    }

    return (
        <Box>
            <Flex justify="space-between" align="center" mb={6}>
                <Heading size="lg">Mi Resumen Académico</Heading>
                <Badge colorScheme="purple" p={2} borderRadius="md" variant="subtle">
                    <HStack>
                        <Icon as={FiClock} />
                        <Text fontSize="sm">Hora Servidor: {currentTime.toLocaleTimeString()}</Text>
                    </HStack>
                </Badge>
            </Flex>

            {/* Stats Grid */}
            <SimpleGrid columns={{ base: 1, md: 3 }} spacing={6} mb={8}>
                <Card variant="outline" boxShadow="sm">
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="blue.100" borderRadius="md" mr={3}>
                                    <Icon as={FiBook} color="blue.500" boxSize={5} />
                                </Box>
                                <StatLabel fontSize="md" fontWeight="medium">Exámenes Disponibles</StatLabel>
                            </Flex>
                            <StatNumber fontSize="3xl">{pendingExams.length}</StatNumber>
                            <StatHelpText>Pendientes por realizar</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card variant="outline" boxShadow="sm">
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="green.100" borderRadius="md" mr={3}>
                                    <Icon as={FiCheckCircle} color="green.500" boxSize={5} />
                                </Box>
                                <StatLabel fontSize="md" fontWeight="medium">Finalizados</StatLabel>
                            </Flex>
                            <StatNumber fontSize="3xl">{finishedExams.length}</StatNumber>
                            <StatHelpText>Historial completo</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card variant="outline" boxShadow="sm">
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="orange.100" borderRadius="md" mr={3}>
                                    <Icon as={FiClock} color="orange.500" boxSize={5} />
                                </Box>
                                <StatLabel fontSize="md" fontWeight="medium">Promedio General</StatLabel>
                            </Flex>
                            <StatNumber fontSize="3xl">{averageScore}</StatNumber>
                            <StatHelpText>Calificación actual</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>
            </SimpleGrid>

            <Heading size="md" mb={4}>Próximas Evaluaciones</Heading>
            <VStack spacing={4} align="stretch">
                {pendingExams.length > 0 ? (
                    pendingExams.map(exam => (
                        <Card key={exam.id_assignment} variant="outline" _hover={{ shadow: 'md', borderColor: 'brand.500' }} transition="all 0.2s">
                            <CardBody py={4}>
                                <Flex justify="space-between" align="center" direction={{ base: 'column', md: 'row' }} gap={4}>
                                    <HStack spacing={4} flex="1">
                                        <Box p={3} bg="brand.50" color="brand.600" borderRadius="lg">
                                            <Icon as={FiEdit3} boxSize={5} />
                                        </Box>
                                        <VStack align="start" spacing={0}>
                                            <HStack>
                                                <Text fontWeight="bold" fontSize="lg">{exam.title}</Text>
                                                {exam.attempt_number > 1 && (
                                                    <Badge variant="solid" colorScheme="orange" fontSize="10px">Re-aplicación</Badge>
                                                )}
                                            </HStack>
                                            <Text color="gray.500" fontSize="sm" noOfLines={1}>{exam.description || 'Sin descripción adicional'}</Text>
                                            {(exam.assignment_scheduled_at || exam.scheduled_at) && (
                                                <Badge mt={1} colorScheme="blue" variant="subtle" px={2} borderRadius="full">
                                                    {(exam.assignment_scheduled_end_at || exam.scheduled_end_at)
                                                        ? `🕒 Disponible: ${formatDate(exam.assignment_scheduled_at || exam.scheduled_at)} al ${formatDate(exam.assignment_scheduled_end_at || exam.scheduled_end_at)}`
                                                        : `📅 Programado: ${formatDate(exam.assignment_scheduled_at || exam.scheduled_at)}`
                                                    }
                                                </Badge>
                                            )}
                                            {exam.assignment_status === 'PENDING' && exam.scheduled_at && (
                                                <VStack align="start" spacing={0} mt={1}>
                                                    <TimeStatusBadge
                                                        scheduledAt={exam.assignment_scheduled_at || exam.scheduled_at}
                                                        currentTime={currentTime}
                                                    />
                                                    <Text fontSize="xs" color="orange.600" mt={1} fontWeight="medium">
                                                        Tolerancia: {exam.tolerance_minutes || 15} min.
                                                    </Text>
                                                </VStack>
                                            )}
                                        </VStack>
                                    </HStack>
                                    <Button
                                        as={RouterLink}
                                        to={`/student/examen/${exam.id_exam || exam.id}`}
                                        rightIcon={<FiArrowRight />}
                                        colorScheme="brand"
                                        size="md"
                                        px={8}
                                        w={{ base: 'full', md: 'auto' }}
                                    >
                                        Iniciar Examen
                                    </Button>
                                </Flex>
                            </CardBody>
                        </Card>
                    ))
                ) : (
                    <Card variant="outline" borderStyle="dashed" bg="gray.50">
                        <CardBody textAlign="center" py={12}>
                            <Icon as={FiBook} boxSize={12} color="gray.300" mb={4} />
                            <Heading size="sm" mb={2} color="gray.600">No hay exámenes programados</Heading>
                            <Text color="gray.500" fontSize="sm" mb={4}>No tienes evaluaciones pendientes en este momento.</Text>
                            <Button
                                as={RouterLink}
                                to="/student/historial"
                                variant="outline"
                                colorScheme="brand"
                                size="sm"
                            >
                                Ver mi historial de exámenes
                            </Button>
                        </CardBody>
                    </Card>
                )}
            </VStack>

            {myExams.some(e => e.attempt_number > 1 || e.assignment_status === 'COMPLETED') && (
                <>
                    <Heading size="md" mt={10} mb={4}>Historial de Intentos</Heading>
                    <Box bg="white" borderRadius="lg" border="1px solid" borderColor="gray.100" overflowX="auto" mb={8}>
                        <Table variant="simple" size="sm">
                            <Thead bg="gray.50">
                                <Tr>
                                    <Th>Examen</Th>
                                    <Th>Intento</Th>
                                    <Th>Fecha</Th>
                                    <Th>Resultado</Th>
                                </Tr>
                            </Thead>
                            <Tbody>
                                {myExams
                                    .filter(e => e.assignment_status === 'COMPLETED')
                                    .sort((a, b) => new Date(b.completed_at || 0) - new Date(a.completed_at || 0))
                                    .map(exam => (
                                        <Tr key={exam.id_assignment}>
                                            <Td fontWeight="medium">{exam.title}</Td>
                                            <Td>#{exam.attempt_number}</Td>
                                            <Td fontSize="xs">{formatDate(exam.completed_at)}</Td>
                                            <Td>
                                                <Badge colorScheme={parseFloat(exam.score) >= 7 ? 'green' : 'red'}>
                                                    {Number(parseFloat(exam.score).toFixed(2))} / 10
                                                </Badge>
                                            </Td>
                                        </Tr>
                                    ))}
                            </Tbody>
                        </Table>
                    </Box>
                </>
            )}
        </Box>
    );
};

export default StudentDashboard;
