const isProduction = true;/*window.location.hostname !== 'localhost' && window.location.hostname !== '127.0.0.1';*/

export const BASE_URL = isProduction
    ? 'https://kequiz.kemarketing.mx'
    : ''; // En desarrollo, usamos el proxy de Vite

const API_URL = isProduction ? `${BASE_URL}/api` : '/api';

const fetchApi = async (endpoint, options = {}) => {
    const token = sessionStorage.getItem('token');
    const headers = {
        'Content-Type': 'application/json',
        ...options.headers
    };

    if (token) {
        headers['Authorization'] = `Bearer ${token}`;
    }

    const config = {
        ...options,
        headers
    };

    const response = await fetch(`${API_URL}${endpoint}`, config);

    // Si la respuesta es 204 No Content, no intentamos parsear JSON
    if (response.status === 204) return null;

    if (!response.ok) {
        let errorMsg = 'Error en la petición';
        let isAuthError = false;

        try {
            const errorData = await response.json();
            errorMsg = errorData.error || errorMsg;

            // Solo considerar errores de autenticación si el mensaje indica claramente un problema de token
            // Los errores 403 de negocio (como "Ya has completado este examen") NO deben redirigir al login
            if (response.status === 401) {
                isAuthError = true;
            } else if (response.status === 403) {
                // Solo es error de autenticación si el mensaje es específico de token inválido
                const authErrorMessages = ['invalid token', 'token missing', 'token expired', 'unauthorized'];
                isAuthError = authErrorMessages.some(msg => errorMsg.toLowerCase().includes(msg));
            }
        } catch (e) {
            // Si no es JSON, verificamos el status
            if (response.status === 401) {
                isAuthError = true;
            }
            errorMsg = `Error ${response.status}: ${response.statusText || 'Respuesta inesperada del servidor'}`;
        }

        // Solo redirigir al login si es un error de autenticación real
        if (isAuthError) {
            sessionStorage.removeItem('token');
            sessionStorage.removeItem('user');
            if (!window.location.pathname.includes('/login') && window.location.pathname !== '/') {
                window.location.href = '/login';
            }
        }

        throw new Error(errorMsg);
    }

    try {
        return await response.json();
    } catch (e) {
        // Si esperamos JSON pero no lo es, y fue exitoso, retornamos null o el texto
        const text = await response.text();
        return text || null;
    }
};

export const login = async (email, password) => {
    const response = await fetch(`${API_URL}/auth/login`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, password })
    });
    if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Error en el inicio de sesión');
    }
    return response.json();
};

// Institutions
export const getInstitutions = () => fetchApi('/institutions');
export const getInstitutionById = (id) => fetchApi(`/institutions/${id}`);
export const createInstitution = (data) => fetchApi('/institutions', {
    method: 'POST',
    body: JSON.stringify(data)
});
export const updateInstitution = (id, data) => fetchApi(`/institutions/${id}`, {
    method: 'PUT',
    body: JSON.stringify(data)
});
export const deleteInstitution = (id) => fetchApi(`/institutions/${id}`, {
    method: 'DELETE'
});

// Users
export const getUsers = () => fetchApi('/users');
export const createUser = (data) => fetchApi('/users', {
    method: 'POST',
    body: JSON.stringify(data)
});
export const updateUser = (id, data) => fetchApi(`/users/${id}`, {
    method: 'PUT',
    body: JSON.stringify(data)
});
export const deleteUser = (id) => fetchApi(`/users/${id}`, {
    method: 'DELETE'
});

// Exams
export const getExams = () => fetchApi('/exams');
export const getExamById = (id) => fetchApi(`/exams/${id}`);
export const createExam = (data) => fetchApi('/exams', {
    method: 'POST',
    body: JSON.stringify(data)
});
export const updateExam = (id, data) => fetchApi(`/exams/${id}`, {
    method: 'PUT',
    body: JSON.stringify(data)
});
export const deleteExam = (id) => fetchApi(`/exams/${id}`, {
    method: 'DELETE'
});
export const getExamAssignments = (examId) => fetchApi(`/exams/${examId}/assignments`);
export const assignExam = (examId, studentId, scheduledAt = null, scheduledEndAt = null) => fetchApi(`/exams/${examId}/assign`, {
    method: 'POST',
    body: JSON.stringify({
        student_id: studentId,
        scheduled_at: scheduledAt,
        scheduled_end_at: scheduledEndAt
    })
});
export const unassignExam = (examId, studentId) => fetchApi(`/exams/${examId}/assign/${studentId}`, {
    method: 'DELETE'
});

export const startExam = (examId) => fetchApi(`/exams/${examId}/start`, { method: 'POST' });
export const submitExam = (examId, answers) => fetchApi(`/exams/${examId}/submit`, {
    method: 'POST',
    body: JSON.stringify({ answers })
});

export const saveResponse = (assignmentId, questionId, studentAnswer) => fetchApi(`/exams/assignments/${assignmentId}/responses`, {
    method: 'POST',
    body: JSON.stringify({
        question_id: questionId,
        student_answer: studentAnswer
    })
});

export const getExamResponses = (assignmentId) => fetchApi(`/exams/assignments/${assignmentId}/responses`);
export const updateResponseGrade = (assignmentId, responseId, data) => fetchApi(`/exams/assignments/${assignmentId}/responses/${responseId}`, {
    method: 'PUT',
    body: JSON.stringify(data)
});

// Questions
export const getQuestions = () => fetchApi('/questions');
export const createQuestion = (data) => fetchApi('/questions', {
    method: 'POST',
    body: JSON.stringify(data)
});
export const updateQuestion = (id, data) => fetchApi(`/questions/${id}`, {
    method: 'PUT',
    body: JSON.stringify(data)
});
export const deleteQuestion = (id) => fetchApi(`/questions/${id}`, {
    method: 'DELETE'
});
export const deleteQuestionsBulk = (ids) => fetchApi('/questions/bulk-delete', {
    method: 'POST',
    body: JSON.stringify({ ids })
});

// Autenticación
export const clientLogin = async (identifier, folio, password) => {
    // Si loginMode es 'password', el tercer argumento es el password real.
    // Si loginMode es 'folio', el segundo argumento (folio) actúa como password.
    const finalPassword = password || folio;
    return login(identifier, finalPassword);
};

// Instituciones / Empresas
export const getCompanies = getInstitutions;
export const getCompanyById = getInstitutionById;
export const createCompany = createInstitution;
export const updateCompany = updateInstitution;
export const deleteCompany = deleteInstitution;
export const uploadCompanyLogo = (id, file) => {
    const formData = new FormData();
    formData.append('logo', file);
    return fetch(`${API_URL}/institutions/${id}/logo`, {
        method: 'POST',
        headers: { 'Authorization': `Bearer ${sessionStorage.getItem('token')}` },
        body: formData
    }).then(res => res.json());
};

// Usuarios (Mapeos para Clientes y Empleados)
export const getClients = async () => {
    try {
        const users = await getUsers();
        return users.filter(u => {
            const role = (u.role || '').toUpperCase();
            return role === 'ESTUDIANTE' || role === 'CLIENT';
        });
    } catch (e) { return []; }
};
export const getStudents = getClients;
export const createClient = createUser;
export const updateClient = updateUser;

export const getEmployees = async () => {
    try {
        const users = await getUsers();
        return users.filter(u => {
            const role = (u.role || '').toUpperCase();
            return role === 'PROFESOR' || role === 'EMPLOYEE';
        });
    } catch (e) { return []; }
};
export const createEmployee = createUser;
export const updateEmployee = updateUser;
export const uploadEmployeePhoto = (id, file) => {
    const formData = new FormData();
    formData.append('photo', file);
    const API_URL_INTERNAL = window.location.hostname !== 'localhost' && window.location.hostname !== '127.0.0.1'
        ? 'https://kequiz.kemarketing.mx/api'
        : '/api';
    return fetch(`${API_URL_INTERNAL}/users/${id}/photo`, {
        method: 'POST',
        headers: { 'Authorization': `Bearer ${sessionStorage.getItem('token')}` },
        body: formData
    }).then(res => {
        if (!res.ok) throw new Error('Error al subir la foto');
        return res.json();
    });
};
export const getEmployeeByUserId = async (userId) => {
    const employees = await getEmployees();
    return employees.find(e => e.id === userId);
};

export const getStudentExams = (studentId) => fetchApi(`/users/${studentId}/exams`);

// Órdenes / Asignaciones
export const getOrders = getExams;
export const createOrder = createExam;
export const updateOrder = updateExam;
export const getContractedServicesByClient = async (clientId) => {
    const exams = await getExams();
    return exams;
};
export const getContractedServicesByEmployee = async (profId) => {
    const exams = await getExams();
    return exams;
};

// Servicios / Catálogo
export const getServicesByCompany = (companyId) => getExams(); // Catálogo ahora son exámenes

// Notificaciones
export const getNotifications = () => fetchApi('/notifications');
export const getUnreadNotificationsCount = () => fetchApi('/notifications/unread-count');
export const markNotificationAsRead = (id) => fetchApi(`/notifications/${id}/read`, { method: 'PUT' });
export const markAllNotificationsAsRead = () => fetchApi('/notifications/read-all', { method: 'PUT' });

// Configuración y Módulos
export const getSystemConfig = () => fetchApi('/config');
export const getSystemConfigValue = async (key) => {
    try {
        const config = await fetchApi(`/config/${key}`);
        return config ? config.config_value : null;
    } catch (e) { return null; }
};
export const updateSystemConfig = (key, value) => fetchApi(`/config/${key}`, {
    method: 'PUT',
    body: JSON.stringify({ config_value: value })
});
export const getModules = () => fetchApi('/modules');
export const getModulesByCompany = (id) => fetchApi(`/institutions/${id}/modules`);
export const toggleCompanyModule = (instId, modId, enabled) => fetchApi(`/institutions/${instId}/modules/${modId}`, {
    method: 'PUT',
    body: JSON.stringify({ is_enabled: enabled })
});

// Reportes
export const getCompanyStats = (id) => fetchApi(`/root/institution-stats/${id}`);

// Utilidades
export const getServerTime = () => fetchApi('/root/time');

// Evidencias / Logs
export const getEvidencesByService = (examId) => fetchApi(`/exams/${examId}/logs`);

// Especialidades
export const getSpecialties = () => fetchApi('/specialties');
export const getSpecialtyById = (id) => fetchApi(`/specialties/${id}`);
export const createSpecialty = (data) => fetchApi('/specialties', {
    method: 'POST',
    body: JSON.stringify(data)
});
export const updateSpecialty = (id, data) => fetchApi(`/specialties/${id}`, {
    method: 'PUT',
    body: JSON.stringify(data)
});
export const deleteSpecialty = (id) => fetchApi(`/specialties/${id}`, {
    method: 'DELETE'
});

