
/**
 * Procesa una imagen para ajustarla a las dimensiones requeridas.
 * @param {File} file - El archivo de imagen original.
 * @param {Object} options - Opciones de procesamiento.
 * @param {number} options.minWidth - Ancho mínimo.
 * @param {number} options.minHeight - Alto mínimo.
 * @param {number} options.maxWidth - Ancho máximo.
 * @param {number} options.maxHeight - Alto máximo.
 * @returns {Promise<File>} - Una promesa que resuelve con el archivo procesado.
 */
export const processImage = (file, { minWidth = 160, minHeight = 160, maxWidth = 900, maxHeight = 900 } = {}) => {
    return new Promise((resolve, reject) => {
        const img = new Image();
        const reader = new FileReader();

        reader.onload = (e) => {
            img.onload = () => {
                let width = img.width;
                let height = img.height;

                // Calcular nuevas dimensiones manteniendo la relación de aspecto
                if (width > maxWidth || height > maxHeight) {
                    const ratio = Math.min(maxWidth / width, maxHeight / height);
                    width = Math.round(width * ratio);
                    height = Math.round(height * ratio);
                }

                if (width < minWidth || height < minHeight) {
                    const ratio = Math.max(minWidth / width, minHeight / height);
                    width = Math.round(width * ratio);
                    height = Math.round(height * ratio);
                }

                const canvas = document.createElement('canvas');
                canvas.width = width;
                canvas.height = height;

                const ctx = canvas.getContext('2d');
                ctx.drawImage(img, 0, 0, width, height);

                canvas.toBlob((blob) => {
                    if (!blob) {
                        reject(new Error('Error al procesar la imagen'));
                        return;
                    }
                    const processedFile = new File([blob], file.name, {
                        type: 'image/jpeg',
                        lastModified: Date.now(),
                    });
                    resolve(processedFile);
                }, 'image/jpeg', 0.85);
            };
            img.onerror = () => reject(new Error('Error al cargar la imagen'));
            img.src = e.target.result;
        };
        reader.onerror = () => reject(new Error('Error al leer el archivo'));
        reader.readAsDataURL(file);
    });
};






